////////////////////////////////////////////////////////////////////// 
// libsrc/MathVector.cpp 
// (c) 2000-2001 Goncalo Abecasis
// 
// This file is distributed as part of the GOLD source code package   
// and may not be redistributed in any form, without prior written    
// permission from the author. Permission is granted for you to       
// modify this file for your own personal use, but modified versions  
// must retain this copyright notice and must not be distributed.     
// 
// Permission is granted for you to use this file to compile GOLD.    
// 
// All computer programs have bugs. Use this file at your own risk.   
// 
// Thursday November 08, 2001
// 
 
#include "MathVector.h"
#include "MathMatrix.h"
#include "Sort.h"
#include "Error.h"

#include <string.h>

int Vector::alloc = 2;

void Vector::Init()
   {
   dim = size = 0;
   label = "Unknown";
   data = NULL;
   }

Vector::~Vector()
   {
   // printf(" Deleting vector %s ...\n", (const char *) label);
   if (data != NULL) delete [] data;
   }

void Vector::Dimension(int d)
   {
   if (d > size)
      {
      size = (d + alloc) / alloc * alloc;
      double * newData = new double [size];
      if (data != NULL)
         {
         for (int i = 0; i < dim; i++)
            newData[i] = data[i];
         delete [] data;
         }
      data = newData;
      }
   dim = d;
   }

void Vector::Negate()
   {
   for (int i = 0; i < dim; i++)
      data[i] = -data[i];
   }

void Vector::Add(double n)
   {
   for (int i = 0; i< dim; i++)
      data[i] += n;
   }

void Vector::Multiply(double k)
   {
   for (int i = 0; i < dim; i++)
      data[i] *= k;
   }

void Vector::Copy(const Vector & v)
   {
   Dimension(v.dim);

   if (v.data != NULL)
      for (int i=0; i < dim; i++)
         data[i] = v.data[i];
   }

Vector & Vector::operator = (const Vector & rhs)
   {
   Copy(rhs);
   return *this;
   }

void Vector::Add(Vector & v)
   {
   if (dim != v.dim)
      error("Vector.Add - vectors have different dimensions\n"
            "Vectors    - %s [%d] + %s [%d] ",
            (const char *) label, dim, (const char  *) v.label, v.dim);

   for (int i = 0; i < dim; i++)
      data[i] += v.data[i];
   }

void Vector::AddMultiple(double k, Vector & v)
   {
   if (dim != v.dim)
      error("Vector.AddMultiple - vectors are incompatible\n"
            "Vectors            - %s [%d] + %s [%d] ",
            (const char  *) label, dim, (const char  *) v.label, v.dim);

   for (int i = 0; i < dim; i++)
      data[i] += k * v.data[i];
   }


void Vector::Subtract(Vector & v)
   {
   if (dim != v.dim)
      error("Vector.Subtract - vectors have different dimensions\n"
            "Vectors         - %s [%d] + %s [%d] ",
            (const char  *) label, dim, (const char  *) v.label, v.dim);

   for (int i = 0; i < dim; i++)
      data[i] -= v.data[i];
   }


void Vector::Zero()
   {
   for (int i = 0; i < dim; i++)
      data[i] = 0.0;
   }

void Vector::Set(double k)
   {
   for (int i = 0; i < dim; i++)
      data[i] = k;
   }

void Vector::SetMultiple(double k, Vector & v)
   {
   Dimension(v.dim);

   for (int i = 0; i < dim; i++)
      data[i] = k * v[i];
   }

double Vector::InnerProduct(Vector & v)
   {
   if (dim != v.dim)
      error("Vector.InnerProduct - vectors have different dimensions\n"
            "Vectors             - %s[%d] * %s[%d] ",
            (const char  *) label, dim, (const char  *) v.label, v.dim);

   double sum = 0.0;
   for (int i = 0; i < dim; i++)
      sum += data[i] * v.data[i];

   return sum;
   }

void Vector::DeleteDimension(int n)
   {
   for (int i = n; i < dim - 1; i++)
      data[i] = data[i + 1];
   dim --;
   }

void Vector::Product(Matrix & m, Vector & v)
   {
   if (m.cols != v.dim)
      error ("Vector.Product - Cannot Multiply Matrix by Vector\n"
             "Vectors        - %s [%d, %d] * %s [%d]\n",
             (const char  *) m.label, m.rows, m.cols,
             (const char  *) v.label, v.dim);

   Dimension(m.rows);
   Zero();

   for(int i = 0; i < m.rows; i++)
      for (int j = 0; j < m.cols; j++)
         data[i] += m[i][j] * v[j];
   }

double Vector::Average() const
   {
   if (dim == 0)
      error("Average undefined for null vector %s",
            (const char  *) label);

   return Sum() / dim;
   }

double Vector::Product() const
   {
   double product = 1.0;

   for (int j = 0; j < dim; j++)
      product *= data[j];

   return product;
   }

double Vector::Sum() const
   {
   double sum = 0.0;

   for (int j=0; j<dim; j++)
      sum += data[j];

   return sum;
   }

double Vector::SumSquares() const
   {
   double sum = 0.0;

   for (int j=0; j<dim; j++)
      sum += data[j] * data[j];

   return sum;
   }

void Vector::AveVar(double & ave, double & var) const
   {
   // uses a two pass method to correct for
   // round-off errors

   if (dim == 0)
      error("Average and Variance undefined for null vector %s",
            (const char  *) label);

   double s, ep;

   ave = var = ep = 0.0;

   for (int j=0; j<dim; j++)
      ave += data[j];

   ave /= dim;

   for (int j=0; j<dim; j++)
      {
      s = data[j] - ave;
      ep += s;
      var += s*s;
      }

   if (dim > 1)
      var = (var - ep*ep/dim)/(dim-1);
   }

double Vector::Var() const
   {
   double mean, var;
   AveVar(mean, var);
   return var;
   }

void Vector::Print(FILE * f, int d)
   {
   if (d == -1 || d > dim) d = dim;

   fprintf(f, "%.15s : ", (const char  *) label);
   for (int i = 0; i < d; i++)
      fprintf(f, "%7.3f ", data[i]);
   fprintf(f, "\n");
   }

int Vector::CompareDouble(const double * a, const double * b)
   {
   if (*a < *b) return -1;
   if (*b > *a) return 1;
   return 0;
   }

void Vector::Sort()
   {
   QuickSort(data, dim, sizeof(double), COMPAREFUNC CompareDouble);
   }

void Vector::Sort(Vector & freeRider)
   {
   QuickSort2(data, freeRider.data, dim, sizeof(double),
              COMPAREFUNC CompareDouble);
   }

bool Vector::operator == (const Vector & rhs) const
   {
   if (rhs.dim != dim) return false;

   for (int i = 0; i < dim; i++)
      if (data[i] != rhs[i])
         return false;
   return true;
   }

// These functions are useful for simulation
//

int Vector::CountIfGreater(double treshold) const
   {
   int count = 0;

   for (int i = 0; i < dim; i++)
      if (data[i] > treshold)
         count++;

   return count;
   }

int Vector::CountIfGreaterOrEqual(double treshold) const
   {
   int count = 0;

   for (int i = 0; i < dim; i++)
      if (data[i] >= treshold)
         count++;

   return count;
   }

// Min and max functions
//

double Vector::Min() const
   {
   if (dim == 0)
      return 0.0;

   double min = data[0];

   for (int i = 1; i < dim; i++)
      if (data[i] < min)
         min = data[i];

   return min;
   }

double Vector::Max() const
   {
   if (dim == 0)
      return 0.0;

   double max = data[0];

   for (int i = 1; i < dim; i++)
      if (data[i] > max)
         max = data[i];

   return max;
   }

// Push and Pop functions for using vector as a stack
//

void Vector::Push(double value)
   {
   Dimension(dim + 1);
   data[dim - 1] = value;
   }

void Vector::Stack(const Vector & v)
   {
   int end = dim;

   Dimension(dim + v.dim);

   for (int i = 0; i < v.dim; i++)
      data[i + end] = v[i];
   }

// VectorFunc class
//

VectorFunc::VectorFunc()
   { f = NULL; }

VectorFunc::VectorFunc(double (*func)(Vector &))
   { f = func; }

double VectorFunc::Evaluate(Vector & v)
   { return f(v); }


 
