////////////////////////////////////////////////////////////////////// 
// libsrc/Parameters.cpp 
// (c) 2000-2001 Goncalo Abecasis
// 
// This file is distributed as part of the GOLD source code package   
// and may not be redistributed in any form, without prior written    
// permission from the author. Permission is granted for you to       
// modify this file for your own personal use, but modified versions  
// must retain this copyright notice and must not be distributed.     
// 
// Permission is granted for you to use this file to compile GOLD.    
// 
// All computer programs have bugs. Use this file at your own risk.   
// 
// Thursday November 08, 2001
// 
 
#include "Parameters.h"
#include "Constant.h"
#include "MathConstant.h"
#include "Error.h"

#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>

int Parameter::nameCol = 30;
int Parameter::statusCol = 15;

Parameter::Parameter(char c, const char * desc, void * v)
   {
   ch = (char) tolower(c);
   description = new char [strlen(desc) + 1];
   strcpy(description, desc);
   var = v;
   }

bool Parameter::Read(int , char ** argv, int argn)
   {
   int   p = 0;
   char  c = (char) tolower(argv[argn][p]);

   if ((c == '-') || (c == '/'))
      {
      p++;
      c = (char) tolower(argv[argn][p]);
      }

   if (c == ch)
      {
      Translate(&(argv[argn][++p]));
      return true;
      }
   return false;
   }

void IntParameter::Translate(const char * value)
   {
   * (int *) var = atoi(value);
   }

void IntParameter::Status()
   {
   printf("%*s : %*d (-%c9999)\n", nameCol, description,
         statusCol, * (int *) var, ch);
   }

void SwitchParameter::Translate(const char * value)
   {
   switch (*value)
      {
      case '+' :
         * (bool *) var = true;
         break;
      case '-' :
         * (bool *) var = false;
         break;
      case 0 :
         * (bool *) var = ! * (bool *) var;
         break;
      default :
         warning("In parameter %c%s the option '%c' has no meaning",
                 ch, value, value[0]);
      }
   }

void SwitchParameter::Status()
   {
   printf("%*s : %*s (-%c[+|-])\n", nameCol, description,
          statusCol, * (bool *) var == false ? "OFF" : "ON", ch);
   }

void DoubleParameter::Translate(const char * value)
   {
   if (value[0])
      * (double *) var = atof(value);
   else
      * (double *) var = _NAN_;
   }

void DoubleParameter::Status()
   {
   if (* (double *) var == _NAN_)
      printf("%*s : %*s (-%c99.999)\n", nameCol, description,
          statusCol, "NAN", ch);
   else if (* (double *) var >= 0.00095)
      printf("%*s : %*.3f (-%c99.999)\n", nameCol, description,
          statusCol, * (double *) var, ch);
   else
      printf("%*s : %*.0e (-%c99.999)\n", nameCol, description,
          statusCol, * (double *) var, ch);
   }

void StringParameter::Translate(const char * value)
   {
   String * s = (String *) var;

   *s = value;
   }

void StringParameter::Status()
   {
   printf("%*s : %*s (-%cname)\n", nameCol, description,
          statusCol, (const char *) (* (String *) var), ch);
   }

void ListParameter::Status()
   {
   OptionList * l;

   for(l = options; l->ch != 0; l++)
      if (l->code == *((int *)var))
         break;

   printf("%*s : %*s (-%c[%s])\n", nameCol, description,
          statusCol, l->description, ch, (const char *) key);
   }

void ListParameter::Translate(const char * value)
   {
   OptionList * l;

   for(l = options; l->ch != 0; l++)
      if (tolower(l->ch) == tolower(value[0]))
         break;

   if (l->ch == 0 && tolower(value[0]) != 0)
      warning("In parameter %c%s the option '%c' has no meaning",
               ch, value, value[0], (const char *) key);

   * ((int*) var) = l->code;
   }

ListParameter::ListParameter(char c, const char * desc, int & v, OptionList * opt)
      : Parameter(c, desc, &v)
   {
   options = opt;

   for (OptionList * l = options; l->ch != 0; l++)
      {
      key += l->ch;
      key += '|';
      }

   key.SetLength(key.Length() - 1);
   }

SetParameter::SetParameter(char c, const char * desc, int & v, OptionList * opt)
      : Parameter(c, desc, &v)
   {
   options = opt;

   for (OptionList * l = options; l->ch != 0; l++)
      {
      key += l->ch;
      key += '|';
      }
   }

void SetParameter::Status()
   {
   bool first = 0;
   int  temp = * (int *) var;

   for(OptionList * l = options; l->ch != 0; l++)
      if ((l->code & temp) || (l->code == *(int *) var) )
         {
         if (!first)
            printf("%*s : %*s (-%c{%s})\n", nameCol, description,
            statusCol, l->description, ch, (const char *) key);
         else
            printf("%*s & %*s\n", nameCol, "",
            statusCol, l->description);
         first = true;
         temp &= ~l->code;
         }
   }

void SetParameter::Translate(const char * value)
   {
   *(int*)var = 0;

   for(const char * chr = value; *chr != 0; chr++)
      {
      int valid = false;

      for(OptionList * l = options; l->ch != 0; l++)
         if (tolower(l->ch) == tolower(*chr))
            {
            * ((int*) var) |= l->code;
            valid = true;
            }

      if (!valid)
         warning("In parameter %c%s, the option '%c' has no meaning",
                 ch, value, *chr);
      }
   }

LongParameters::LongParameters(const char * desc, LongParameterList * lst)
   : Parameter('-', desc, NULL)
   {
   list = lst;

   index.Clear();
   group_len = 0;

   for (LongParameterList * ptr = list + 1; ptr->description != NULL; ptr++)
      if (ptr->value != NULL)
         index.Add(ptr->description, ptr);
      else
         group_len = max(strlen(ptr->description), group_len);

   }

void LongParameters::Translate(const char * cstr)
   {
   String value(cstr);

   int p = value.FastFindChar(':');
   int option = p == -1 ? index.FindStem(value) : index.FindStem(value.Left(p));

   if (option >= 0)
      {
      LongParameterList * ptr = (LongParameterList *) index.Object(option);

      if (ptr->type == LP_BOOL_PARAMETER)
         {
         if (p == -1)
            * (bool *) ptr->value ^= true;
         else
            * (bool *) ptr->value = value.SubStr(p + 1).SlowCompare("ON") == 0;

         // In exclusive groups, only one option may be selected
         if (ptr->exclusive)
            {
            for (int i = -1; ptr[i].exclusive; i--) * (bool *)ptr[i].value = false;
            for (int i =  1; ptr[i].exclusive; i++) * (bool *)ptr[i].value = false;
            }
         }
      else if (ptr->type == LP_INT_PARAMETER)
         if (p == -1)
            * (int *) ptr->value = * (int *) ptr->value ? 0 : 1;
         else
            * (int *) ptr->value = value.SubStr(p + 1).SlowCompare("ON") == 0 ?
                                   1 : value.SubStr(p + 1).AsInteger();
      }
   else if (option == -1)
      warning("The parameter --%s is undefined\n", (const char *) value);
   else if (option == -2)
      warning("The parameter --%s is ambiguous\n", (const char *) value);
   }

void LongParameters::Status()
   {
   if (description != NULL)
      printf("\n%s\n", description);

   bool need_a_comma = false;
   int line_len = 0;

   String state;

   for (LongParameterList * ptr = list + 1; ptr->description != NULL; ptr++)
      if (ptr->value == NULL)
         {
         printf("%s %*s :", need_a_comma ? "\n" : "",
                group_len + 2, ptr->description);
         need_a_comma = false;
         line_len = group_len + 5;
         }
      else
         {
         if (ptr->type == LP_BOOL_PARAMETER)
            state = * (bool *) ptr->value ? " [ON]" : "";
         else if (ptr->type == LP_INT_PARAMETER)
            if (* (int *) ptr->value < 2)
               state = * (int *) ptr->value ? " [ON]" : "";
            else
               state = " [", state += * (int *) ptr->value, state += ']';

         int item_len = 3 + strlen(ptr->description) +
                      need_a_comma + state.Length();

         if (item_len + line_len > 78)
            {
            line_len = group_len ? group_len + 5 : 0;
            printf("%s%*s\n", need_a_comma ? "," : "", line_len,  "");
            need_a_comma = 0;
            item_len -= 1;
            }

         printf("%s --%s%s", need_a_comma++ ? "," : "",
                ptr->description, (const char *) state);

         need_a_comma = true;
         line_len += item_len;
         }

   printf("\n");
   }

void ParameterList::Add(Parameter * p)
   {
   if (count + 1 >= size)
      error("Parameter list size should be increased");

   pl[count++] = p;
   };

void ParameterList::Read(int argc, char ** argv, int start)
   {
   MakeString(argc, argv, start);
   for (int i=start; i < argc; i++)
      {
      bool success = false;

      if (argv[i][0] == '-' && argv[i][1])
         for (int j=0; j<count; j++)
            {
            success = tolower(argv[i][1]) == pl[j]->ch;

            if (success)
               {
               if (argv[i][2] == 0 && (i+1 < argc) && (argv[i + 1][0] != '-'))
                  pl[j]->Translate(argv[++i]);
               else
                  pl[j]->Translate(argv[i] + 2);
               break;
               }
            }

      if (!success)
         warning("Command line parameter %s (#%d) ignored", argv[i], i);
      }
   };

int ParameterList::ReadWithTrailer(int argc, char ** argv, int start)
   {
   MakeString(argc, argv, start);

   int last_success = start - 1;
   bool split = false;

   for (int i=start; i < argc; i++)
      {
      bool success = false;

      if (argv[i][0] == '-' && argv[i][1])
         for (int j=0; j<count; j++)
            {
            success = tolower(argv[i][1]) == pl[j]->ch;

            if (success)
               {
               if (argv[i][2] == 0 && (i+1 < argc) && (argv[i + 1][0] != '-'))
                  pl[j]->Translate(argv[i + 1]), split = true;
               else
                  pl[j]->Translate(argv[i] + 2);
               break;
               }
            }

      if (success)
         for (last_success++; last_success < i; last_success++)
            warning("Command line parameter %s (#%d) ignored",
                    argv[last_success], last_success);

      if (split) { split = false; last_success++; i++; }
      }

   return last_success;
   };


void ParameterList::Status()
   {
   printf("\nThe following parameters are in effect:\n");

   for (int i=0; i<count; i++)
      pl[i]->Status();

   printf("\n");
   }

void ParameterList::MakeString(int argc, char ** argv, int start)
   {
   int len = 0;

   for (int i=start; i<argc; i++)
      len += strlen(argv[i]) + 1;

   string = new char [len+1];
   string[0] = 0;

   for (int i=start; i<argc; i++)
      {
      strcat(string, argv[i]);
      strcat(string, " ");
      }
   }

ParameterList::~ParameterList()
   {
   for (int i = 0; i < count; i++)
      delete pl[i];
   delete [] pl;
   delete [] string;
   };


 
