unit GLayout;

interface

uses Windows, ExtCtrls, Graphics, Math, Printers;

type GoldLayout = class
   Bounds : TRect;
   Graph, Map, Legend, Title : TRect;
   scaleX, scaleY : real;
   digitHeight, digitWidth, square : integer;
   Len : integer;
   function MapX(x : integer) : integer;
   function MapY(y : integer) : integer;
   procedure PrepareForScreen(GraphBox, MapBox, LegendBox: TPaintBox; MapLength : integer );
   procedure PrepareForBitmap(Bitmap : TBitmap; MapLength : integer; r1, r2 : real);
   procedure PrepareForPrinting(MapLength : integer; r1, r2 : real);
   procedure Recalculate(MapLength : integer);
end;

implementation

{ GoldLayout }

function GoldLayout.MapX(x: integer): integer;
begin
   MapX := Round(Bounds.Left + scaleX * x);
end;

function GoldLayout.MapY(y: integer): integer;
begin
   MapY := Round(Bounds.Bottom - scaleY * y);
end;

procedure GoldLayout.PrepareForBitmap(Bitmap: TBitmap; MapLength: integer; r1, r2 : real);
begin
   Bitmap.Canvas.Font.Name := 'Arial';
   Bitmap.Canvas.Font.Size := 14;

   Graph.Left := 0;
   Graph.Top := 0;
   Graph.Right := Round(Bitmap.Width / (1 + r2) + 1);
   Graph.Bottom := Round(Bitmap.Height / (1 + r1) + 1);

   Map.Left := 0;
   Map.Top := Graph.Bottom;
   Map.Right := Bitmap.Width;
   Map.Bottom := Bitmap.Height;

   Legend.Left := Graph.Right;
   Legend.Top := 0;
   Legend.Right := Bitmap.Width;
   Legend.Bottom := Graph.Bottom;

   digitWidth := Bitmap.Canvas.TextWidth('0123456789') div 10;
   digitHeight := Bitmap.Canvas.TextHeight('0123456789');

   Recalculate(MapLength);

end;

procedure GoldLayout.PrepareForPrinting(MapLength: integer; r1, r2 : real);
begin
  with Printer do
  begin
   Canvas.Font.Name := 'Arial';
   Canvas.Font.Size := 14;

   Graph.Left := 0;
   Graph.Top := 0;
   Graph.Right := Round(PageWidth / (1 + r2) + 1);
   Graph.Bottom := Round(PageHeight / (1 + r1) + 1);

   Map.Left := 0;
   Map.Top := Graph.Bottom;
   Map.Right := Graph.Right;
   Map.Bottom := PageHeight;

   Legend.Left := Graph.Right;
   Legend.Top := 0;
   Legend.Right := PageWidth;
   Legend.Bottom := Graph.Bottom;

   digitWidth := Canvas.TextWidth('0123456789') div 10;
   digitHeight := Canvas.TextHeight('0123456789');

   Recalculate(MapLength);
  end
end;


procedure GoldLayout.PrepareForScreen
   (GraphBox, MapBox, LegendBox: TPaintBox; MapLength : integer );
begin
   Graph.Left := 0;
   Graph.Top := 0;
   Graph.Right := GraphBox.ClientWidth;
   Graph.Bottom := GraphBox.ClientHeight;

   Map.Left := 0;
   Map.Top := 0;
   Map.Right := MapBox.ClientWidth;
   Map.Bottom := MapBox.ClientHeight;

   Legend.Left := 0;
   Legend.Top := 0;
   Legend.Right := LegendBox.ClientWidth;
   Legend.Bottom := LegendBox.ClientHeight;

   digitWidth := GraphBox.Canvas.TextWidth('0123456789') div 10;
   digitHeight := GraphBox.Canvas.TextHeight('0123456789');

   Recalculate(MapLength);
end;

procedure GoldLayout.Recalculate(MapLength : integer);
var midx, midy : integer;
begin
  Len := max(MapLength, 1);

  Bounds.Left := Graph.Left + digitWidth * 2;
  Bounds.Right := Graph.Right - digitHeight * 2;
  Bounds.Top := Graph.Top + digitHeight;
  Bounds.Bottom := Graph.Bottom - digitHeight;

  square := min(Bounds.Right - Bounds.Left, Bounds.Bottom - Bounds.Top) div 2;
  midx := (Bounds.Right + Bounds.Left) div 2;
  midy := (Bounds.Top + Bounds.Bottom) div 2;

  Bounds.Left := midx - square;
  Bounds.Right := midx + square;
  Bounds.Top := midy - square;
  Bounds.Bottom := midy + square;

  scaleX := (Bounds.Right - Bounds.Left) / Len;
  scaleY := (Bounds.Bottom - Bounds.Top) / Len;
end;

end.
