unit Main;

interface

uses
  SysUtils, Windows, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, ExtCtrls, Menus, StrTable,
  PhysMap, PrettyNumbers, CUtils, Math, ToolWin, ComCtrls, ImgList,
  ActnList, GLayout, Printers;

type
  SurfaceTransformation = (tfNone, tfExp, tfLog, tfLn, tfSqr, tfSqrt);
  XYZ = record
      x, y, z  : integer;
      height : real;
  end;
  Landmarks = array of XYZ;
  TMainForm = class(TForm)
    OpenDialog: TOpenDialog;
    PrintDialog: TPrintDialog;
    PrintSetupDialog: TPrinterSetupDialog;
    Panel2: TPanel;
    GraphBox: TPaintBox;
    MapDialog: TOpenDialog;
    Splitter1: TSplitter;
    Panel1: TPanel;
    Splitter2: TSplitter;
    LegendBox: TPaintBox;
    MapBox: TPaintBox;
    Panel3: TPanel;
    MoreButton: TButton;
    LessButton: TButton;
    ImageList1: TImageList;
    ToolBar1: TToolBar;
    ToolButton1: TToolButton;
    ToolButton2: TToolButton;
    PrintButton: TToolButton;
    PrintSetupButton: TToolButton;
    ToolButton5: TToolButton;
    GridButton: TToolButton;
    FillButton: TToolButton;
    ContourButton: TToolButton;
    LeftHandedButton: TToolButton;
    InvertButton: TToolButton;
    ZeroOneButton: TToolButton;
    ToolButton12: TToolButton;
    DataBox: TComboBox;
    ToolButton13: TToolButton;
    TransformBox: TComboBox;
    ToolButton14: TToolButton;
    ExitButton: TToolButton;
    PopupMenu1: TPopupMenu;
    DisequilibriumData1: TMenuItem;
    GeneticMap1: TMenuItem;
    PopupMenu2: TPopupMenu;
    NoGridMenu: TMenuItem;
    OverlayGridMenu: TMenuItem;
    RawDataMenu: TMenuItem;
    SaveButton: TToolButton;
    SaveDialog: TSaveDialog;
    procedure FileOpen(Sender: TObject);
    procedure FilePrint(Sender: TObject);
    procedure FilePrintSetup(Sender: TObject);
    procedure FileExit(Sender: TObject);
    procedure HelpAbout(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure GraphBoxPaint(Sender: TObject);
    procedure OpenMap(Sender: TObject);
    procedure MapBoxPaint(Sender: TObject);
    procedure LegendBoxPaint(Sender: TObject);
    procedure Splitter1CanResize(Sender: TObject; var NewSize: Integer;
      var Accept: Boolean);
    procedure Splitter2CanResize(Sender: TObject; var NewSize: Integer;
      var Accept: Boolean);
    procedure DataBoxChange(Sender: TObject);
    procedure TransformBoxChange(Sender: TObject);
    procedure MoreButtonClick(Sender: TObject);
    procedure LessButtonClick(Sender: TObject);
    procedure FillButtonClick(Sender: TObject);
    procedure ContourButtonClick(Sender: TObject);
    procedure InvertButtonClick(Sender: TObject);
    procedure LeftHandedButtonClick(Sender: TObject);
    procedure ZeroOneButtonClick(Sender: TObject);
    procedure LegendBoxDblClick(Sender: TObject);
    procedure NoGridMenuClick(Sender: TObject);
    procedure OverlayGridMenuClick(Sender: TObject);
    procedure RawDataMenuClick(Sender: TObject);
    procedure SaveButtonClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  public
    curTransform : surfaceTransformation;
    Surface : array of array of real;
    Distances : array of integer;
    Cutoffs : array of real;
    Zhi, Zlo, Zdelta : real;
    Zscale : PrettyInteger;
    PhysicalMap : MarkerMap;
    Disequilibrium : StringTable;
    drawContours, fillContours, invertColors, leftHandedView, zeroOneScale,
    overlayGrid, interpolation, fillBW : boolean;
    procedure SurveySurface;
    procedure CalculateLegend;
    procedure DrawSurface(canvas : TCanvas; L : GoldLayout);
    procedure DrawGrid(canvas : TCanvas; L : GoldLayout );
    procedure DrawRawData(canvas : TCanvas; L : GoldLayout);
    procedure DrawVerticalMap(canvas : TCanvas; L: GoldLayout);
    procedure DrawHorzMap(canvas : TCanvas; L: GoldLayout);
    procedure DrawLegend(canvas : TCanvas; L: GoldLayout);
    procedure QuickLandmarks(vertex : Landmarks; var p : Landmarks);
    procedure QuickSurface(canvas : TCanvas; p : Landmarks);
    procedure QuickContour(canvas : TCanvas; p : Landmarks; L : GoldLayout);
    procedure QuickLine(canvas : TCanvas; x1, y1, x2, y2: integer);
    procedure QuickText(canvas : TCanvas; x, y : integer; text : string;
                        align : integer);
    function CalculateColor(p : real) : TColor;
    function Transform(x : real) : real;
    function isUpper() : boolean;
    function isLower() : boolean;
    function hasDiagonal() : boolean;
  end;

var
  MainForm: TMainForm;

implementation

{$R *.DFM}

const qthLeft = 0;
      qthRight = 1;
      qthCenter = 2;
      qtvTop = 0;
      qtvBottom = 4;
      qtvCenter = 8;

procedure TMainForm.FileOpen(Sender: TObject);
var i : integer;
begin
  if OpenDialog.Execute then
  begin
      Disequilibrium.Read(OpenDialog.FileName, 0);
      PhysicalMap.DefaultMap(0);
      DataBox.Clear;
      if Length(Disequilibrium.header) > 2 then
      begin
         for i := 2 to Length(Disequilibrium.header) - 1 do
            DataBox.Items.Add(Disequilibrium.header[i]);
         DataBox.ItemIndex := DataBox.Items.Count - 1;
         DataBox.Enabled := true;
         TransformBox.ItemIndex := 0;
         TransformBox.Enabled := true;
         SurveySurface;
      end
      else
         Disequilibrium.valid := false;
  end;
end;

procedure TMainForm.FilePrint(Sender: TObject);
var layout : GoldLayout;
begin
  if PrintDialog.Execute then
  with Printer do
  begin
     Title := 'GOLD Printout';
     BeginDoc;

     Layout := GoldLayout.Create;
     Layout.PrepareForPrinting(PhysicalMap.MapLength,
                               MapBox.Height / GraphBox.Height,
                               LegendBox.Width / GraphBox.Width);

     if (Length(PhysicalMap.Markers) > 0) then
     begin
        if (interpolation) then DrawSurface(Canvas, layout);
        if (overlayGrid) then DrawGrid(Canvas, layout);
        if (not interpolation) then DrawRawData(Canvas, layout);
        DrawVerticalMap(Canvas, layout);
        DrawHorzMap(Canvas, layout);
        DrawLegend(Canvas, layout);
     end;

     EndDoc;
  end
end;

procedure TMainForm.FilePrintSetup(Sender: TObject);
begin
  PrintSetupDialog.Execute;
end;

procedure TMainForm.FileExit(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.HelpAbout(Sender: TObject);
begin
  { Add code to show program's About Box }
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
   Disequilibrium := StringTable.Create;
   PhysicalMap := MarkerMap.Create;
   DataBox.ItemIndex := 0;
   TransformBox.ItemIndex := 0;
   SetLength(Cutoffs, 8);
   Zhi := 1;
   Zlo := 0;
   drawContours := ContourButton.Down;
   fillContours := fillButton.Down;
   overlayGrid := false;
   interpolation := true;
   invertColors := invertButton.Down;
   leftHandedView := leftHandedButton.Down;
   zeroOneScale := zeroOneButton.Down;
   fillBW := false;
end;

procedure TMainForm.GraphBoxPaint(Sender: TObject);
var layout : GoldLayout;
begin
   GraphBox.Canvas.Brush.Color := clBtnFace;
   GraphBox.Canvas.FillRect(Rect(0,0,Width,Height));

   if Length(PhysicalMap.markers) <= 0 then Exit;

   layout := GoldLayout.Create;
   layout.PrepareForScreen(GraphBox, MapBox, LegendBox, PhysicalMap.MapLength);
   if (interpolation) then DrawSurface(GraphBox.Canvas, layout);
   if (overlayGrid) then DrawGrid(GraphBox.Canvas, layout);
   if (not interpolation) then DrawRawData(GraphBox.Canvas, layout);
   DrawVerticalMap(GraphBox.canvas, layout);
end;

procedure TMainForm.OpenMap(Sender: TObject);
begin
  if MapDialog.Execute then
  begin
     PhysicalMap.Read(MapDialog.FileName);
     SurveySurface;
     MapBox.Invalidate;
     GraphBox.Invalidate;
  end;
end;

procedure TMainForm.MapBoxPaint(Sender: TObject);
var layout    : GoldLayout;
begin
   layout := GoldLayout.Create;
   layout.PrepareForScreen(GraphBox, MapBox, LegendBox, PhysicalMap.MapLength);

   MapBox.Canvas.Brush.Color := clBtnFace;
   MapBox.Canvas.FillRect(Rect(0,0,MapBox.Width, MapBox.Height));

   DrawHorzMap(MapBox.Canvas, layout);
end;

procedure TMainForm.LegendBoxPaint(Sender: TObject);
var layout : GoldLayout;
begin
   layout := GoldLayout.Create;
   layout.PrepareForScreen(GraphBox, MapBox, LegendBox, PhysicalMap.MapLength);

   LegendBox.Canvas.Brush.Color := clBtnFace;
   LegendBox.Canvas.FillRect(Rect(0,0,LegendBox.Width, LegendBox.Height));

   DrawLegend(LegendBox.Canvas, layout);
end;

procedure TMainForm.Splitter1CanResize(Sender: TObject;
  var NewSize: Integer; var Accept: Boolean);
begin
   if NewSize > 150 then Accept := False;
end;

procedure TMainForm.Splitter2CanResize(Sender: TObject;
  var NewSize: Integer; var Accept: Boolean);
begin
   if NewSize > 150 then Accept := False;
end;

procedure TMainForm.QuickLine(canvas: TCanvas; x1, y1, x2, y2: integer);
begin
   canvas.MoveTo(x1, y1);
   canvas.LineTo(x2, y2);
end;

procedure TMainForm.QuickText(canvas: TCanvas; x, y: integer;
  text: string; align: integer);
var h, v : integer;
begin
   canvas.Brush.Style := bsClear;
   h := canvas.TextWidth(text);
   v := canvas.TextHeight(text);
   if (align and qthCenter) <> 0 then x := x - h div 2;
   if (align and qthRight) <> 0 then x := x - h;
   if (align and qtvCenter) <> 0 then y := y - v div 2;
   if (align and qtvBottom) <> 0 then y := y - v;
   canvas.TextOut(x, y, text);
end;

function TMainForm.CalculateColor(p: real): TColor;
const third = 1.0 / 3.0;
begin
   if invertColors then p := 1.0 - p;
   if fillBW then
      result := Round(p*255) + Round(p*255) * $100 + Round(p*255) * $10000
   else if p < 0.2 then
      result := $7F0000 + Round(p / 0.2 * 127) * $10000
   else if p < 0.4 then
      result := $FF0000 + Round((p - 0.2) / 0.2 * 255) * $100
   else if p < 0.6 then
      result := $7F00 + Round((p - 0.4) / 0.2 * 127) * $100
   else if p < 0.8 then
      result := $00FF00 + Round((p - 0.6) / 0.2 * 255) * $01
   else
      result := $FF + Round((1.0 - p) / 0.2 * 255) * $100;
end;

procedure TMainForm.SurveySurface;
var len, len2, i, j, x, y, col : integer;
    z : real;
    init : boolean;
    lower, upper, diagonal : boolean;
begin
   col := DataBox.ItemIndex + 2;
   if (not PhysicalMap.isValid) and Disequilibrium.valid then
   begin
      len := Max(Disequilibrium.GetHighInt(1), Disequilibrium.GetHighInt(0));
      if (len < 2) then
      begin
         Disequilibrium.valid := false;
         MessageDlg('Problem Parsing Data Matrix' + #13#13 +
            'Each line should have three or more columns: ' + #13 +
            'MARKER1 -- index for marker1 (starting at 1)' + #13 +
            'MARKER2 -- index for marker2 (starting at 1)' + #13 +
            'STATISTIC1 -- statistic for this marker pair' + #13 +
            '... -- additional statistics may follow' + #13#13 +
            'Labels for each statistic should be defined in a header line',
            mtError, [mbOk], 0);
      end
      else
         PhysicalMap.DefaultMap(len);
   end;
   if Disequilibrium.valid and (col >= 2) then
   begin
      len := PhysicalMap.CountMarkers - 1;
      SetLength(surface, len + 1, len + 1);
      SetLength(distances, len + 1);
      for i := 0 to len do
         for j := 0 to len do
            surface[i][j] := 0;

      init := false;
      lower := isUpper();
      upper := isLower();
      diagonal := hasDiagonal();

      len2 := Length(Disequilibrium.data) - 1;
      for i := 0 to len2 do
      begin
         x := PhysicalMap.LookupIndex(atoi(disequilibrium.data[i][0]));
         y := PhysicalMap.LookupIndex(atoi(disequilibrium.data[i][1]));
         if (x < 0) or (y < 0) then continue;

         z := Transform(atof(disequilibrium.data[i][col]));
         if not init then
         begin
            Zhi := z;
            Zlo := z;
            init := true;
         end;
         surface[x][y] := z;

         if Upper or Lower then surface[y][x] := z;

         if z > Zhi then Zhi := z;
         if z < Zlo then Zlo := z;
      end;

      if (zhi <= 1) and (zlo >= 0) and (zeroOneScale) then
      begin
         zhi := 1;
         zlo := 0;
      end;

      Zhi := Zhi + 0.0001;
      Zdelta := Zhi - Zlo;
      Zscale := RoundUp(1);

      if not diagonal then
      begin
         for i := 1 to len - 1 do
            surface[i][i] := (surface[i-1][i] + surface[i+1][i] +
                              surface[i][i-1] + surface[i][i+1]) / 4;
         surface[0][0] := (surface[1][0] + surface[0][1]) / 2;
         surface[len][len]:=(surface[len][len-1]+surface[len-1][len])/2;
      end;

      for i := 0 to len do
         distances[i] := PhysicalMap.markers[i].position;
   end
   else
   begin
      Zhi := 1;
      Zlo := 0;
      SetLength(surface, 0, 0);
      SetLength(distances, 0);
   end;
   CalculateLegend;
   GraphBox.Invalidate;
   MapBox.Invalidate;
   LegendBox.Invalidate;
end;

procedure TMainForm.DataBoxChange(Sender: TObject);
begin
   SurveySurface;
end;

procedure TMainForm.TransformBoxChange(Sender: TObject);
var s : string;
begin
   curTransform := tfNone;
   if TransformBox.ItemIndex > 0 then
   begin
      s := TransformBox.Items.Strings[TransformBox.ItemIndex];
      if s = 'Exp' then curTransform := tfExp else
      if s = 'Log' then curTransform := tfLog else
      if s = 'Ln' then curTransform := tfLn else
      if s = 'Sqr' then curTransform := tfSqr else
      if s = 'Sqrt' then curTransform := tfSqrt else
   end;
   SurveySurface;
end;

function TMainForm.Transform(x: real): real;
begin
   if (x < 0) and (curTransform in [tfLn, tfLog, tfSqrt]) then
   begin
      Transform := 0;
      Exit;
   end;
   if (x = 0) and (curTransform in [tfLn, tfLog, tfSqrt]) then
      x := 1e-5;
   if (x >= 30) and (curTransform = tfExp) then
   begin
      Transform := 10e10;
      Exit;
   end;
   case curTransform of
      tfExp : transform := exp(x);
      tfLn : transform := ln(x);
      tfLog : transform := log10(x);
      tfSqr : transform := x * x;
      tfSqrt : transform := sqrt(x);
   else
      transform := x;
   end;
end;

procedure TMainForm.MoreButtonClick(Sender: TObject);
begin
   SetLength(cutoffs, Length(cutoffs) + 1);
   LessButton.Enabled := Length(cutoffs) > 3;
   CalculateLegend;
end;

procedure TMainForm.LessButtonClick(Sender: TObject);
begin
   SetLength(cutoffs, Length(cutoffs) - 1);
   LessButton.Enabled := Length(cutoffs) > 3;
   CalculateLegend;
end;

procedure TMainForm.CalculateLegend;
var i, len : integer;
begin
   len := Length (cutoffs) - 1;
   for i := 0 to len do
      cutoffs[i] := Zdelta / len * i + Zlo;
   LegendBox.Invalidate;
   GraphBox.Invalidate;
end;

procedure TMainForm.QuickSurface(canvas: TCanvas; p: Landmarks);
var i, k, hi, len : integer;
    polygon : array of TPoint;

   procedure DeletePoint(del : integer);
   var temp : Landmarks;
       itemp, jtemp : integer;
   begin
      SetLength(temp, Length(p) - 1);
      jtemp := 0;
      for itemp := 0 to Length(p) - 1 do
      if itemp <> del then
      begin
         temp[jtemp] := p[itemp];
         Inc(jtemp);
      end;
      p := temp;
   end;

begin

   while (Length(p) > 0) do
   begin
      SetLength(polygon, 0);
      hi := 0;

      len := Length(p) - 1;
      for i := 0 to len do
         if p[i].Z > p[hi].Z then hi := i;

      while (hi < len) do
         if p[hi].Z = p[hi + 1].Z then
            hi := hi + 1
         else
            break;

      k := p[hi].Z;

      while k = p[hi].Z do
      begin
         SetLength(polygon, Length(polygon) + 1);
         polygon[Length(polygon) - 1] := Point(p[hi].X, p[hi].Y);
         DeletePoint(hi);
         if Length(p) = 0 then break;
         hi := (hi + Length(p) - 1) mod Length(p);
      end;
      canvas.Brush.color := CalculateColor(1/(Length(cutoffs) - 2) * abs(k));
      canvas.Pen.Color := canvas.Brush.Color;
      canvas.Polygon(polygon);
   end;
end;

procedure TMainForm.QuickLandmarks(vertex : Landmarks; var p: Landmarks);

   var i : integer;

   procedure CalculateEdge(x0, y0, z0 : integer; h0 : real;
                           x1, y1, z1 : integer; h1 : real);
   var i, prev : integer;
       d : real;
   begin
      SetLength(p, Length(p) + 1);
      p[Length(p) - 1].x := x0;
      p[Length(p) - 1].y := y0;
      p[Length(p) - 1].z := z0;

      i := z0;
      while i <> z1 do
      begin
         prev := i;
         SetLength(p, Length(p) + 2);
         if i > z1 then
         begin
            d := (cutoffs[i] - h0) / (h1 - h0);
            Dec(i);
         end
         else
         begin
            Inc(i);
            d := (cutoffs[i] - h0) / (h1 - h0);
         end;
         if (d < 0) or (d > 1) then
            d := Round(d);
         p[Length(p) - 2].x := Round(d * (x1 - x0) + x0);
         p[Length(p) - 2].y := Round(d * (y1 - y0) + y0);
         p[Length(p) - 2].z := prev;
         p[Length(p) - 1].x := p[Length(p) - 2].x;
         p[Length(p) - 1].y := p[Length(p) - 2].y;
         p[Length(p) - 1].z := i;
      end;
   end;

begin
   SetLength(p, 0);
   CalculateEdge(vertex[0].x, vertex[0].y, vertex[0].z, vertex[0].height,
                 vertex[1].x, vertex[1].y, vertex[1].z, vertex[1].height);
   CalculateEdge(vertex[1].x, vertex[1].y, vertex[1].z, vertex[1].height,
                 vertex[2].x, vertex[2].y, vertex[2].z, vertex[2].height);
   CalculateEdge(vertex[2].x, vertex[2].y, vertex[2].z, vertex[2].height,
                 vertex[3].x, vertex[3].y, vertex[3].z, vertex[3].height);
   CalculateEdge(vertex[3].x, vertex[3].y, vertex[3].z, vertex[3].height,
                 vertex[0].x, vertex[0].y, vertex[0].z, vertex[0].height);
   if leftHandedView then for i := 0 to Length(p) - 1 do p[i].Z := - p[i].Z;
end;

procedure TMainForm.FillButtonClick(Sender: TObject);
begin
   FillContours := FillButton.Down;
   LeftHandedButton.Enabled := drawContours or fillContours;
   GraphBox.Invalidate;
end;

procedure TMainForm.ContourButtonClick(Sender: TObject);
begin
   DrawContours := ContourButton.Down;
   LeftHandedButton.Enabled := drawContours or fillContours;
   GraphBox.Invalidate;
end;

procedure TMainForm.InvertButtonClick(Sender: TObject);
begin
   invertColors := InvertButton.Down;
   LegendBox.Invalidate;
   GraphBox.Invalidate;
end;

procedure TMainForm.QuickContour(canvas: TCanvas; p: Landmarks; l : GoldLayout);
var i, k, hi : integer;

   procedure DeletePoint(del : integer);
   var temp : Landmarks;
       itemp, jtemp, len : integer;
   begin
      len := Length(p) - 1;
      SetLength(temp, len);
      jtemp := 0;
      for itemp := 0 to len do
      if itemp <> del then
      begin
         temp[jtemp] := p[itemp];
         Inc(jtemp);
      end;
      p := temp;
   end;

begin

   Canvas.Pen.Color := clBlack;
   if (Length(cutoffs) > 6) then
      Canvas.Pen.Width := 1
   else
      Canvas.Pen.Width := 2;

   while (Length(p) > 0) do
   begin
      hi := 0;

      for i := 0 to Length(p) - 1 do
         if p[i].Z > p[hi].Z then hi := i;

      while (hi < Length(p) - 1) do
         if p[hi].Z = p[hi + 1].Z then
            hi := hi + 1
         else
            break;

      if hi = Length(p) - 1 then Exit;

      k := p[hi].Z;
      if p[hi + 1].Z < k then
         canvas.MoveTo(p[hi].x, p[hi].y);

      while k = p[hi].Z do
      begin
         DeletePoint(hi);
         if Length(p) = 0 then break;
         hi := (hi + Length(p) - 1) mod Length(p);
      end;

      if p[hi].Z < k then
         canvas.LineTo(p[hi].x, p[hi].y);

      canvas.MoveTo(L.Bounds.left, L.Bounds.top);
      canvas.LineTo(L.Bounds.left, L.Bounds.bottom);
      canvas.LineTo(L.Bounds.right, L.Bounds.bottom);
      canvas.LineTo(L.Bounds.right, L.Bounds.top);
      canvas.LineTo(L.Bounds.left, L.Bounds.top);
   end;

   Canvas.Pen.Width := 1;
end;

procedure TMainForm.LeftHandedButtonClick(Sender: TObject);
begin
   leftHandedView := LeftHandedButton.Down;
   GraphBox.Invalidate;
end;

procedure TMainForm.ZeroOneButtonClick(Sender: TObject);
begin
   zeroOneScale := ZeroOneButton.Down;
   SurveySurface;
   GraphBox.Invalidate;
   LegendBox.Invalidate;
end;

function TMainForm.hasDiagonal: boolean;
var i, len : integer;
begin
  hasDiagonal := false;
  len := Length(Disequilibrium.data) - 1;
  for i := 0 to len do
      if atoi(disequilibrium.data[i][0]) =
         atoi(disequilibrium.data[i][1]) then
         begin
            hasDiagonal := true;
            Exit;
         end
end;

function TMainForm.isLower: boolean;
var i, len : integer;
begin
  isLower := true;
  len := Length(Disequilibrium.data) - 1;
  for i := 0 to len do
      if atoi(disequilibrium.data[i][0]) >
         atoi(disequilibrium.data[i][1]) then
         begin
            isLower := false;
            Exit;
         end
end;

function TMainForm.isUpper: boolean;
var i, len : integer;
begin
  isUpper := true;
  len := Length(Disequilibrium.data) - 1;
  for i := 0 to len do
      if atoi(disequilibrium.data[i][0]) <
         atoi(disequilibrium.data[i][1]) then
         begin
            isUpper := false;
            Exit;
         end
end;

procedure TMainForm.LegendBoxDblClick(Sender: TObject);
begin
   fillBW := not fillBW;
   LegendBox.Invalidate;
   GraphBox.Invalidate;
end;

procedure TMainForm.DrawGrid(canvas: TCanvas; L : GoldLayout);
var len, m, pos : integer;
begin
   len := Length(PhysicalMap.markers);
   Canvas.Pen.Color := clDkGray;

   for m := 0 to len - 1 do
   begin
     pos := PhysicalMap.markers[m].position;
     QuickLine(canvas, L.MapX(pos), L.Bounds.top, L.MapX(pos), L.Bounds.bottom);
     QuickLine(canvas, L.Bounds.left, L.MapY(pos), L.Bounds.right, L.MapY(pos));
   end;
end;

procedure TMainForm.DrawRawData(canvas: TCanvas; L : GoldLayout);
var radius, col, len, x, y, i, k : integer;
    z : real;
begin
    radius := 1 + L.square * 2 div (Length(PhysicalMap.markers) * 3);

    col := DataBox.ItemIndex + 2;
    len := Length(Disequilibrium.data) - 1;

    for i := 0 to len do
    begin
       x := PhysicalMap.LookupIndex(atoi(disequilibrium.data[i][0]));
       y := PhysicalMap.LookupIndex(atoi(disequilibrium.data[i][1]));
       z := Transform(atof(disequilibrium.data[i][col]));
       if (x < 0) or (y < 0) then continue;

       x := PhysicalMap.markers[x].position;
       y := PhysicalMap.markers[y].position;

       x := L.MapX(x);
       y := L.MapY(y);

       k := 0;
       while z > cutoffs[k+1] do Inc(k);
       canvas.Brush.color := CalculateColor(1/(Length(cutoffs) - 2) * abs(k));

       Canvas.Ellipse(x - radius, y - radius, x + radius, y + radius);
    end;
end;

procedure TMainForm.DrawSurface(canvas: TCanvas; L : GoldLayout);
var p, vertex : Landmarks;
    i, j, len : integer;
begin
  with canvas do
  begin
    SetLength(vertex, 4);
    len := Length(Surface) - 1;
    for i := 1 to len do
       for j := 1 to len do
          begin
             vertex[0].height := Surface[i - 1][j - 1];
             vertex[1].height := Surface[i][j - 1];
             vertex[2].height := Surface[i][j];
             vertex[3].height := Surface[i - 1][j];
             vertex[0].z := 0;
             while vertex[0].height>cutoffs[vertex[0].z+1] do Inc(vertex[0].z);
             vertex[1].z := 0;
             while vertex[1].height>cutoffs[vertex[1].z+1] do Inc(vertex[1].z);
             vertex[2].z := 0;
             while vertex[2].height>cutoffs[vertex[2].z+1] do Inc(vertex[2].z);
             vertex[3].z := 0;
             while vertex[3].height>cutoffs[vertex[3].z+1] do Inc(vertex[3].z);
             vertex[0].x := L.MapX(PhysicalMap.markers[i-1].position);
             vertex[1].x := max(L.MapX(PhysicalMap.markers[i].position), vertex[0].x + 1);
             vertex[2].x := vertex[1].x;
             vertex[3].x := vertex[0].x;
             vertex[0].y := L.MapY(PhysicalMap.markers[j-1].position);
             vertex[1].y := vertex[0].y;
             vertex[2].y := min(L.MapY(PhysicalMap.markers[j].position), vertex[0].y + 1);
             vertex[3].y := vertex[2].y;
             QuickLandmarks(vertex, p);
             if fillContours then QuickSurface(Canvas, p);
             if drawContours then QuickContour(Canvas, p, l);
          end;
  end;
end;

procedure TMainForm.NoGridMenuClick(Sender: TObject);
begin
   NoGridMenu.Checked := true;

   interpolation := true;
   overlayGrid := false;

   FillButton.Enabled := true;
   ContourButton.Enabled := true;
   LeftHandedButton.Enabled := true;

   GraphBox.Invalidate;
end;

procedure TMainForm.OverlayGridMenuClick(Sender: TObject);
begin
   OverlayGridMenu.Checked := true;

   interpolation := true;
   overlayGrid := true;

   FillButton.Enabled := true;
   ContourButton.Enabled := true;
   LeftHandedButton.Enabled := true;

   GraphBox.Invalidate;
end;

procedure TMainForm.RawDataMenuClick(Sender: TObject);
begin
   RawDataMenu.Checked := True;

   interpolation := false;
   overlayGrid := true;

   FillButton.Enabled := false;
   ContourButton.Enabled := false;
   LeftHandedButton.Enabled := false;

   GraphBox.Invalidate;
end;

procedure TMainForm.DrawVerticalMap(canvas: TCanvas; L : GoldLayout);
var x, len, m, y, mergey : integer;
    merge : string;
begin
   if (L.Bounds.Left) < 8 * L.digitWidth then Exit;

   x := L.Bounds.Left - L.digitWidth * 4;

   Canvas.Pen.Width := 2;
   Canvas.Pen.Color := clBlack;

   QuickLine(canvas, x, L.Bounds.top, x, L.Bounds.bottom);
   QuickLine(canvas, x - L.digitHeight div 2, L.Bounds.top,
                     x + L.digitHeight div 2, L.Bounds.top);
   QuickLine(canvas, x - L.digitHeight div 2, L.Bounds.bottom,
                     x + L.digitHeight div 2, L.Bounds.bottom);

   canvas.Pen.Width := 1;
   len := Length(PhysicalMap.Markers);

   for m := 0 to len - 1 do
   begin
      y := L.MapY(PhysicalMap.markers[m].position);
      QuickLine(canvas, x, y, x - L.digitHeight div 2, y);
   end;

   canvas.Brush.Style := bsClear;

   if (L.Bounds.Left) < 12 * L.digitWidth then Exit;

   merge := PhysicalMap.markers[0].name + ' ';
   mergey := L.MapY(PhysicalMap.markers[0].position);
   y := mergey;

   for m := 1 to len - 1 do
   begin
      y := L.MapY(PhysicalMap.markers[m].position);
      if (mergey - y < L.digitHeight) then
         merge := merge + PhysicalMap.markers[m].name + ' '
      else
      begin
         QuickText(canvas,
                   x - L.digitWidth * 1,
                  (mergey + L.MapY(PhysicalMap.markers[m-1].position)) div 2,
                   merge, qtvCenter or qthRight);
         merge := PhysicalMap.markers[m].name + ' ';
         mergey := y;
      end;
   end;
   QuickText(canvas, x - L.digitWidth * 1, (mergey + y) div 2, merge, qtvCenter or qthRight);
end;

procedure TMainForm.DrawHorzMap(Canvas: TCanvas; L: GoldLayout);
var legend    : PrettyInteger;
    scale     : integer;
    scaleText : string;
    rows, m   : integer;
    len       : integer;
begin
   with Canvas do
   begin
      Pen.Color := clBlack;
      Pen.Width := 2;
      QuickLine(Canvas, L.Bounds.left,
                        L.Map.Top + L.digitHeight * 5 div 2,
                        L.Bounds.left,
                        L.Map.Top + L.digitHeight * 7 div 2);
      QuickLine(Canvas, L.Bounds.left,
                        L.Map.Top + L.digitHeight * 3,
                        L.Bounds.right,
                        L.Map.Top +  L.digitHeight * 3);
      QuickLine(Canvas, L.Bounds.right,
                        L.Map.Top + L.digitHeight * 5 div 2,
                        L.Bounds.right,
                        L.Map.Top + L.digitHeight * 7 div 2);

      Pen.Width := 1;
      legend := RoundUp((PhysicalMap.MapLength + 9) div 10);
      scale := 1;
      scaleText := ' bp';
      if (legend.adjusted) > 1000 then
      begin
         scale := 1000;
         scaleText := ' Kb';
      end;
      if (legend.adjusted) > 1000000 then
      begin
         scale := 1000000;
         scaleText := ' Mb';
      end;

      QuickLine(Canvas,
                L.MapX(legend.adjusted),
                L.Map.Top + L.digitHeight * 5 div 2,
                L.MapX(legend.adjusted),
                L.Map.Top + L.digitHeight * 3);
      QuickText(Canvas,
                L.MapX(legend.adjusted),
                L.Map.Top + L.digitHeight * 3 div 2,
                itoa(legend.adjusted div scale) + scaleText,
                qthCenter or qtvCenter);

      if legend.adjusted * 5 < PhysicalMap.MapLength then
      begin
         legend.adjusted := legend.adjusted * 5;
         QuickLine(Canvas,
                   L.MapX(legend.adjusted),
                   L.Map.Top + L.digitHeight * 5 div 2,
                   L.MapX(legend.adjusted),
                   L.Map.Top + L.digitHeight * 3);
         QuickText(Canvas,
                   L.MapX(legend.adjusted),
                   L.Map.Top + L.digitHeight * 3 div 2,
                   itoa(legend.adjusted div scale) + scaleText,
                   qthCenter or qtvCenter);
      end;

      rows := (L.Map.Bottom - L.Map.Top - L.digitHeight * 4) div (L.digitHeight + 2);

      len := Length(PhysicalMap.markers);
      for m := 0 to len - 1 do
      begin
         QuickLine(Canvas,
                   L.MapX(PhysicalMap.markers[m].position),
                   L.Map.Top + L.digitHeight * 3,
                   L.MapX(PhysicalMap.markers[m].position),
                   L.Map.Top + L.digitHeight * 7 div 2);
         if rows = 0 then exit;
         QuickText(Canvas,
                   L.MapX(PhysicalMap.markers[m].position),
                   L.Map.Top + (L.digitHeight + 2) * (4 + m mod rows),
                   PhysicalMap.markers[m].name, qtvCenter or qthCenter);
      end;
   end;
end;

procedure TMainForm.DrawLegend(canvas: TCanvas; L: GoldLayout);
var legendWidth, margin, bands, i : integer;
begin
   with Canvas do
   begin
      legendWidth := (L.Legend.Right - L.Legend.Left) div 2;
      QuickText(Canvas,
                L.Legend.Left + legendWidth,
                L.Legend.Top + L.digitHeight * 2,
                ftoa(zhi), qthCenter or qtvCenter);

      QuickText(Canvas,
                L.Legend.Left + legendWidth,
                L.Legend.Bottom - L.digitHeight * 2,
                ftoa(zlo), qthCenter or qtvCenter);

      margin := L.digitHeight * 7;
      bands := Length(cutoffs) - 1;

      for i := 0 to bands - 1 do
      begin
         Brush.Color := CalculateColor(1 - 1 / (bands - 1) * i);
         Rectangle(L.Legend.Left + L.digitWidth,
                   Round(margin/2 + (L.Legend.Bottom - margin)/bands * i) - 1,
                   L.Legend.Right - L.digitWidth,
                   Round(margin/2 + (L.Legend.Bottom - margin)/bands * (i+1)));
      end;
   end;

end;

procedure TMainForm.SaveButtonClick(Sender: TObject);
var Bitmap : TBitmap;
    Layout : GoldLayout;
begin
   if SaveDialog.Execute then
   begin
      Bitmap := TBitmap.Create;
      Bitmap.Width := 1024;
      Bitmap.Height := 768;

      Layout := GoldLayout.Create;
      Layout.PrepareForBitmap(Bitmap, PhysicalMap.MapLength,
                              MapBox.Height / GraphBox.Height,
                              LegendBox.Width / GraphBox.Width);

      Bitmap.Canvas.Brush.Color := clWhite;
      Bitmap.Canvas.FillRect(Rect(0, 0, Bitmap.Width, Bitmap.Height));

      if (Length(PhysicalMap.Markers) > 0) then
      begin
         if (interpolation) then DrawSurface(Bitmap.Canvas, layout);
         if (overlayGrid) then DrawGrid(Bitmap.Canvas, layout);
         if (not interpolation) then DrawRawData(Bitmap.Canvas, layout);
         DrawVerticalMap(Bitmap.Canvas, layout);
         DrawHorzMap(Bitmap.Canvas, layout);
         DrawLegend(Bitmap.Canvas, layout);
      end;

      Bitmap.SaveToFile(SaveDialog.FileName);
   end;
end;

procedure TMainForm.FormShow(Sender: TObject);
begin
   if (Panel1.Height - 60 < LegendBox.Height) then
      begin
         LegendBox.Height := Panel1.Height - 80;
         MoreButton.Top := LegendBox.Height + 4;
         LessButton.Top := MoreButton.Top + MoreButton.Height + 2;
      end;
end;

end.
