unit PhysMap;

interface

uses StrTable;

type MarkerInfo = record
   id : integer;
   name : string;
   position : integer;
end;

type MarkerMap = class
   markers : array of MarkerInfo;
   rawData : StringTable;
   constructor Create;
   function isValid : boolean;
   function MapLength : integer;
   function CountMarkers : integer;
   procedure Read(fileName : string);
   procedure Sort;
   procedure DefaultMap(markerCount : integer);
   function  LookupIndex(markerId : integer) : integer;
private
   procedure Convert;
   procedure Adjust;
end;

implementation

uses Math, Dialogs;

{ MarkerMap }

procedure MarkerMap.Adjust;
var i : integer;
    lo : integer;
begin
   if not isValid then exit;

   lo := markers[0].position;
   for i := 1 to Length(markers) - 1 do
      lo := min(lo, markers[i].position);

   for i := 0 to Length(markers) - 1 do
      markers[i].position := markers[i].position - lo;
end;

procedure MarkerMap.Convert;
var  i : integer;
     error : integer;
     msg : string;
begin
   SetLength(markers,0);

   if Length(rawData.data) = 0 then exit;
   SetLength(markers, Length(rawData.data));

   for i := 0 to Length(rawData.data) - 1 do
   begin
      Val(rawData.data[i][0], markers[i].id, error);
      if error <> 0 then break;
      Val(rawData.data[i][2], markers[i].position, error);
      if error <> 0 then break;
      markers[i].name := rawData.data[i][1];
   end;

   if error <> 0 then
   begin
      SetLength(markers,0);
      Str(i + 1, msg);
      MessageDlg('Error Processing Map File (line ' + msg + ')' + #13#13 +
         'Each mapfile line should have three columns: ' + #13 +
         'MARKER_ID -- integer index for marker (starting at 1)' + #13 +
         'MARKER_NAME -- one word marker description (e.g., 1235A/C)' + #13 +
         'MARKER_POS -- marker position in base-pairs (e.g., 1235)',
         mtError, [mbOk], 0);
   end
end;

function MarkerMap.CountMarkers: integer;
begin
   CountMarkers := Length(markers);
end;

constructor MarkerMap.Create;
begin
   rawData := StringTable.Create;
end;

procedure MarkerMap.DefaultMap(markerCount: integer);
var i : integer;
begin
   SetLength(markers, markerCount);
   for i := 0 to markerCount - 1 do
   begin
      markers[i].id := i + 1;
      Str(i + 1, markers[i].name);
      markers[i].position := i * 1000000;
   end;
end;

function MarkerMap.isValid: boolean;
begin
   isValid := Length(markers) <> 0;
end;

function MarkerMap.LookupIndex(markerId: integer): integer;
var i : integer;
begin
   for i := 0 to Length(markers) - 1 do
   if markers[i].id = markerId then
   begin
      LookupIndex := i;
      Exit;
   end;
   LookupIndex := -1;
end;

function MarkerMap.MapLength: integer;
begin
   if isValid then
      MapLength := markers[Length(markers)-1].position
   else
      MapLength := 1
end;

procedure MarkerMap.Read(fileName: string);
begin
   rawData.Read(fileName, 3);
   Convert;
   Adjust;
   Sort;
end;

procedure MarkerMap.Sort;
var i, j, lo, loPosition : integer;
    temp : MarkerInfo;
begin
   for i := 0 to Length(markers) - 1 do
   begin
      lo := i;
      loPosition := markers[i].position;

      for j := i + 1 to Length(markers) - 1 do
         if markers[j].position < loPosition then
         begin
            loPosition := markers[j].position;
            lo := j;
         end;

      if lo <> i then
      begin
         temp := markers[i];
         markers[i] := markers[lo];
         markers[lo] := temp;
      end;
   end;
end;

end.
