unit StrTable;

interface

type
   TableLine = array of string;
   StringTable = class
      header : TableLine;
      data : array of TableLine;
      valid : boolean;
      constructor Create;
      procedure Read(fileName : string; cols : integer);
      function GetHighInt(col : integer) : integer;
      function GetLowInt(col : integer) : integer;
      function GetHighValue(col : integer) : real;
      function GetLowValue(col : integer) : real;
   end;

procedure ParseLine(var tokens : TableLine; line : string);

implementation

uses SysUtils, Dialogs, Math, CUtils;

const separators : string = #9' ';

procedure SafeReadLn(var F : TextFile; var s : string);
var ch : char;
begin
   s := '';

   repeat
      Read(F, ch)
   until not ((ch in [#10,#13,#9,' ']));

   while not (ch in [#10,#13,#26]) do
   begin
      s := s + ch;
      Read(F, ch);
   end;
end;

constructor StringTable.Create;
begin
   SetLength(header, 0);
   SetLength(data, 0);
   valid := false;
end;

function StringTable.GetHighInt(col: integer): integer;
var i, len, value, hi : integer;
begin
   len := Length(data);
   hi := 1;

   if len > 0 then
   begin
     hi := atoi(data[0][col]);
     for i := 1 to len - 1 do
     begin
        value := atoi(data[i][col]);
        if hi < value then
           hi := value;
     end;
   end;

   GetHighInt := hi;
end;

function StringTable.GetHighValue(col: integer): real;
var i, len : integer;
    value, hi: real;
begin
   len := Length(data);
   hi := 1;

   if len > 0 then
   begin
     hi := atof(data[0][col]);
     for i := 1 to len - 1 do
     begin
        value := atof(data[i][col]);
        if hi < value then
           hi := value;
     end;
   end;

   GetHighValue := hi;
end;

function StringTable.GetLowInt(col: integer): integer;
var i, len : integer;
    value, lo: integer;
begin
   len := Length(data);
   lo := 1;

   if len > 0 then
   begin
     lo := atoi(data[0][col]);
     for i := 1 to len - 1 do
     begin
        value := atoi(data[i][col]);
        if lo > value then
           lo := value;
     end;
   end;

   GetLowInt := lo;
end;

function StringTable.GetLowValue(col: integer): real;
var i, len : integer;
    value, lo: real;
begin
   len := Length(data);
   lo := 1;

   if len > 0 then
   begin
     lo := atof(data[0][col]);
     for i := 1 to len - 1 do
     begin
        value := atof(data[i][col]);
        if lo > value then
           lo := value;
     end;
   end;

   GetLowValue := lo;
end;

procedure StringTable.Read(fileName : string; cols : integer);

var   F     : TextFile;
      temp  : StringTable;
      line  : string;
      msg   : string;
      msg1  : string;
      msg2  : string;
      i     : integer;
      size  : integer;
      error : boolean;

begin
   size := 0;
   i := 0;
   error := false;

   {$I-}
   Assign(F, fileName);
   if IOResult <> 0 then error := true;
   Reset(F);
   if IOResult <> 0 then error := true;
   {$I+}

   if error then
      MessageDlg('Error opening file "' + filename + '"' + #13 +
                 'Perhaps the file is being used by another program?',
                 mtError, [mbOk], 0)
   else
   begin
   try
      temp := StringTable.Create;

      SafeReadLn(F, line);

      ParseLine(temp.header, line);
      if (cols <> 0) and (Length(temp.header) <> cols) then
      begin
         error := true;
         Str(cols, msg);
         Str(Length(temp.header), msg1);
         MessageDlg('Error processing file "' + filename + '"' + #13 +
            'Expecting ' + msg + ' columns but header appears to have '
             + msg1 + ' columns!', mtError, [mbOk], 0);
      end
      else

         while (not Eof(F)) and (not error) do
         begin
            SafeReadLn(F, line);
            if line = '' then continue;

            // Grow table in blocks to reduce memory fragmentation
            if (i >= size) then
            begin
               size := max(size, 1) * 2;
               SetLength(temp.data, size);
            end;

            SetLength(temp.data[i], 0);
            ParseLine(temp.data[i], line);
            if Length(temp.data[i]) <> Length(temp.header) then
            begin
               Str(Length(temp.header), msg);
               Str(Length(temp.data[i]), msg1);
               Str(i + 1, msg2);
               MessageDlg('Error processing file "' + filename + '"' + #13 +
                  'Header defines ' + msg + ' columns but line ' + msg2 + ' ' +
                  'appears to have ' + msg1 + ' columns!', mtError, [mbOk], 0);
               error := true;
            end;
            Inc(i);
         end;

         if i = 0 then
         begin
            MessageDlg('The file "' + filename + '" has no data!',
                       mtError, [mbOk], 0);
            error := true;
         end;

         if not error then
         begin
            SetLength(temp.data, i);
            header := temp.header;
            data := temp.data;
            valid := true;
         end;
   finally
      CloseFile(F);
   end;
   end;
end;

var tokens_in_last_line : integer = 0;

procedure ParseLine(var tokens : TableLine; line : string);
var   i : integer;
      start : integer;
      finish : integer;
      len : integer;
      size : integer;
begin
   i := 0;
   start := 1;
   len := Length(line);

   // Hopefully this will reduce memory fragmentation
   size := tokens_in_last_line;
   SetLength(tokens, size);

   while IsDelimiter(separators, line, start) do
      if (start <= len) then Inc(start) else break;

   while (start <= len) do
   begin
      finish := start;
      while not IsDelimiter(separators, line, finish) do
         if (finish <= len) then Inc(finish) else break;

      if (i >= size) then
      begin
         size := max(size, 1) * 2;
         SetLength(tokens, size);
      end;

      tokens[i] := Copy(line, start, finish - start);
      Inc(i);

      start := finish;
      while IsDelimiter(separators, line, start) do
         if (start <= len) then Inc(start) else break;
   end;

   if ((i > 0) and (i <> tokens_in_last_line)) then
   begin
      SetLength(tokens, i);
      tokens_in_last_line := i;
   end;
end;

end.
