/*
 *  Copyright (C) 2010  Regents of the University of Michigan
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

//////////////////////////////////////////////////////////////////////////
// This file contains the processing for the executable option "readIndexedBam"
// which reads an indexed BAM file by chromosome and writes it into a new
// file sorted from reference id -1 to maxRefID.

#include "SamFile.h"
#include "Parameters.h"
#include "SamValidation.h"

void readIndexedBamDescription()
{
    std::cout << " readIndexedBam - Read Indexed BAM By Reference and write it from reference id -1 to maxRefId:" << std::endl;
    std::cout << "\t./bam readIndexedBam <inputFilename> <outputFile.sam/bam> <bamIndexFile>" << std::endl;
    std::cout << std::endl;
}


void readIndexedBamUsage()
{
    readIndexedBamDescription();
}


int readIndexedBam(const char* inputFilename,
                   const char* outputFilename,
                   const char* indexFilename)
{
    // Open the input file for reading.
    SamFile samIn;
    if(!samIn.OpenForRead(inputFilename))
    {
        fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        return(samIn.GetStatus());
    }

    // Open the bam index file for reading.
    if(!samIn.ReadBamIndex(indexFilename))
    {
        fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        return(samIn.GetStatus());
    }

    // Open the output file for writing.
    SamFile samOut;
    if(!samOut.OpenForWrite(outputFilename))
    {
        fprintf(stderr, "%s\n", samOut.GetStatusMessage());
        return(samOut.GetStatus());
    }

    // Read the sam header.
    SamFileHeader samHeader;
    if(!samIn.ReadHeader(samHeader))
    {
        fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        return(samIn.GetStatus());
    }

    // Write the sam header.
    if(!samOut.WriteHeader(samHeader))
    {
        fprintf(stderr, "%s\n", samOut.GetStatusMessage());
        return(samOut.GetStatus());     
    }

    SamRecord samRecord;
    SamValidationErrors samInvalidErrors;

    // Loop through each Reference.
    for(int i = -1; i < samHeader.referenceContigs.Length(); i++)
    {
        int numSectionRecords = 0;
        samIn.SetReadSection(i);
        // Keep reading records until they aren't more.
        while(samIn.ReadRecord(samHeader, samRecord))
        {
            numSectionRecords++;
            // Successfully read a record from the file, so write it.
            if(!samOut.WriteRecord(samHeader, samRecord))
            {
                // Failed to write a record.
                fprintf(stderr, "%s\n", samOut.GetStatusMessage());
            }
        }

        if(samIn.GetStatus() != SamStatus::NO_MORE_RECS)
        {
            // Failed to read a record.
            fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        }

        std::cout << "Reference ID " << i << " has " << numSectionRecords 
                  << " records" << std::endl;
    }
   
    std::cout << "Number of records = " << samIn.GetCurrentRecordCount() 
              << std::endl;
   
    return(0);
}


