/*
 *  Copyright (C) 2010  Regents of the University of Michigan
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <values.h>
#include <math.h>
#include <string>

#include "CigarRoller.h"
#include "Generic.h"
#include "GenomeSequence.h"
#include "SamFile.h"
#include "genmat/GenMatrix.h"
#include "base/logger.h"
#include "BgzfFileType.h"
#include "tclap/CmdLine.h"
#include "tclap/Arg.h"

//#define NUM_IBDS 101
//#define NUM_MIX_FRACS 51
#define MAX_Q 100

Logger* Logger::gLogger = NULL;

std::string bimChr2ncbiChr(std::string & bimChr) {
  int nChr = atoi(bimChr.c_str());
  std::string ncbiChr;
  if ( nChr == 0 ) {
    Logger::gLogger->error("Unrecognized chromosome %s",bimChr.c_str());
  }
  else if ( nChr < 23 ) {
    ncbiChr = bimChr;
  }
  else if ( nChr == 23 ) {
    ncbiChr = "X";
  }
  else if ( nChr == 24 ) {
    ncbiChr = "Y";
  }
  else if ( nChr == 25 ) {
    ncbiChr = "XY";
  }
  else if ( nChr == 26 ) {
    ncbiChr = "MT";
  }
  else {
    Logger::gLogger->error("Unrecognized chromosome %s",bimChr.c_str());
  }
  return ncbiChr;
}

int main(int argc, char** argv) {
  // configurable parameters
  std::string sRef, sInFile, sInPrefix, sInSuffix, sIndexFile, sOutFile, sLogFile, sBfile, sBimFile;
  bool bVerbose, bNoEOF, bSelfOnly, bBimAF, bMemoryMap;
  uint32_t minQ, maxQ, minMapQ, maxDepth;
  double genoError, minAF, mixUnit, ibdUnit;

  // parameters not set to be configuraable
  bool sameSMFlag = true;
  uint16_t includeSamFlag = 0x0000;
  uint16_t excludeSamFlag = 0x0704;
  
  try {
    TCLAP::CmdLine cmd("Command description message", ' ', "0.01");
    TCLAP::ValueArg<std::string> argReference("r","reference","Karma's reference sequence",true,"","string");
    TCLAP::ValueArg<std::string> argIn("i","in","Input BAM file. Must be sorted and indexed",false,"","string");
    TCLAP::ValueArg<std::string> argInprefix("p","inprefix","Prefix of input BAM file for multi-chromosome inputs",false,"","string");
    TCLAP::ValueArg<std::string> argInsuffix("s","insuffix","Suffix of input BAM file for multi-chromosome inputs",false,"","string");
    TCLAP::ValueArg<std::string> argIndex("I","index","Index of input BAM file - [inputBam].bai will be default value",false,"","string");
    TCLAP::ValueArg<std::string> argOut("o","out","Prefix of output files",true,"","string");
    TCLAP::ValueArg<std::string> argLog("l","log","Log file - default: [out].log",false,"","string");
    TCLAP::ValueArg<std::string> argBfile("b","bfile","Binary PLINK format genotype file prefix. Must be forward-stranded",false,"","string");
    TCLAP::ValueArg<std::string> argBimFile("B","bimpfile","PLINK format BIM file with allele frequency information at the last column",false,"","string");
    TCLAP::ValueArg<uint32_t> argMinQ("q","minQ","Minimum Phred-scale base quality value (default:20) - bases with lower quality will be ignored",false,20,"integer");
    TCLAP::ValueArg<uint32_t> argMaxQ("Q","maxQ","Maximum Phred-scale base quality value (default:40) - higher base quality will be enforced to be this value",false,40,"integer");
    TCLAP::ValueArg<uint32_t> argMinMapQ("m","minMapQ","Minimum mapping quality value (default:10) - reads with lower quality will be ignored",false,10,"integer");
    TCLAP::ValueArg<uint32_t> argMaxDepth("d","maxDepth","Maximum depth per site (default:20) - bases with higher depth will be ignored due to possible alignment artifacts. For deep coverage data, it is important to set this value to a sufficiently large number (e.g. 200)",false,20,"integer");
    TCLAP::ValueArg<double> argGenoError("g","genoError","Error rate in genotype data (default: 0.005)",false,5.0e-3,"double");
    TCLAP::ValueArg<double> argMinAF("f","minAF","Minimum allele frequency (default: 0.005). Markers with lower allele frequencies will be ignored",false,5.0e-3,"double");
    TCLAP::ValueArg<double> argMixUnit("","mixUnit","unit of % mixture (default:0.01)",false,0.01,"double");
    TCLAP::ValueArg<double> argIbdUnit("","ibdUnit","unit of IBD values (default: 0.01)",false,0.01,"double");
    TCLAP::SwitchArg switchVerbose("v","verbose","Toggle verbose mode (default:off)",cmd,false);
    TCLAP::SwitchArg switchNoeof("n","noeof","Do not check EOF marker for BAM file (default:off)",cmd,false);
    TCLAP::SwitchArg switchSelfOnly("S","selfonly","compare the genotypes with SELF (annotated sample) only (default:off)",cmd,false);
    TCLAP::SwitchArg switchBimAF("F","bimAF","use the allele frequency information by loading .bimp file instead of .bim file",cmd,false);
    TCLAP::SwitchArg switchMemoryMap("M","mmap","toggle whether to use memory map (default:true)",cmd,true);
    
    cmd.add(argReference);
    cmd.add(argIn);
    cmd.add(argInprefix);
    cmd.add(argInsuffix);
    cmd.add(argIndex);
    cmd.add(argOut);
    cmd.add(argLog);
    cmd.add(argBfile);
    cmd.add(argBimFile);
    cmd.add(argMinQ);
    cmd.add(argMaxQ);
    cmd.add(argMinMapQ);
    cmd.add(argMaxDepth);
    cmd.add(argGenoError);
    cmd.add(argMinAF);
    cmd.add(argMixUnit);
    cmd.add(argIbdUnit);

    cmd.parse(argc, argv);

    sRef = argReference.getValue();
    sInFile = argIn.getValue();
    sInPrefix = argInprefix.getValue();
    sInSuffix = argInsuffix.getValue();
    sIndexFile = argIndex.getValue();
    sOutFile = argOut.getValue();
    sLogFile = argLog.getValue();
    sBfile = argBfile.getValue();
    sBimFile = argBimFile.getValue();
    minQ = argMinQ.getValue();
    maxQ = argMaxQ.getValue();
    minMapQ = argMinMapQ.getValue();
    maxDepth = argMaxDepth.getValue();
    genoError = argGenoError.getValue();
    minAF = argMinAF.getValue();
    mixUnit = argMixUnit.getValue();
    ibdUnit = argIbdUnit.getValue();
    bVerbose = switchVerbose.getValue();
    bNoEOF = switchNoeof.getValue();
    bSelfOnly = switchSelfOnly.getValue();
    bBimAF = switchBimAF.getValue();
    bMemoryMap = switchMemoryMap.getValue();

    if ( sLogFile.empty() ) {
      sLogFile = sOutFile + ".log";
    }
    Logger::gLogger = new Logger(sLogFile.c_str(), bVerbose);

    Logger::gLogger->writeLog("Arguments in effect: \n%s",cmd.toString().c_str());
  }
  catch (TCLAP::ArgException &e)  { 
    std::cerr << "error: " << e.error() << " for arg " << e.argId() << std::endl; 
    abort();
  }

  uint32_t numIBDs = static_cast<uint32_t>(ceil(1./ibdUnit))+1;
  uint32_t numMixFracs = static_cast<uint32_t>(ceil(.5/mixUnit))+1;
  double* fIBDs = new double[numIBDs];
  double* fMixFracs = new double[numMixFracs];
  double fPhred2Err[MAX_Q+1];

  for(uint32_t i=0; i < numIBDs; ++i) {
    fIBDs[i] = ibdUnit * static_cast<double>(i);
    if ( fIBDs[i] > 1. ) fIBDs[i] = 1.;
  }
  for(uint32_t i=0; i < numMixFracs; ++i) {
    fMixFracs[i] = mixUnit * static_cast<double>(i);
    if ( fMixFracs[i] > .5 ) fMixFracs[i] = 0.5;
  }

  for(int32_t i=0; i < MAX_Q+1; ++i) {
    if ( i > static_cast<int32_t>(maxQ) ) {
      fPhred2Err[i] = fPhred2Err[maxQ]; // Phred>=40 is uninformative
    }
    else {
      fPhred2Err[i] = pow(10.,(0-i)/10.);
    }
  }

  if ( sBfile.empty() && sBimFile.empty() ) {
    Logger::gLogger->error("Either --bfile or --bimfile is required");
  }

  if ( sInFile.empty() && ( sInPrefix.empty() || sInSuffix.empty() ) ) {
    Logger::gLogger->error("--in or (--inprefix & --insuffix) are required arguments");
  }

  if ( ( !sInFile.empty() ) && ( !sInPrefix.empty() ) && ( !sInSuffix.empty() ) ) {
    Logger::gLogger->error("--in and (--inprefix & --insuffix) cannot be combined together");
  }

  std::map<std::string,uint32_t> msRGidx;
  std::vector<std::string> vsRGIDs;
  std::vector<std::string> vsSMIDs;
  std::string sBamSMID;

  GenomeSequence genomeSequence;
  genomeSequence.setReferenceName(sRef.c_str());
  genomeSequence.useMemoryMap(bMemoryMap);
  if ( genomeSequence.open() ) {
    Logger::gLogger->warning("Failed to open karma-type index of reference file %s. Creating a new one..",sRef.c_str());
    if ( genomeSequence.create(false)) {
      Logger::gLogger->error("Cannot crate karma-type index of reference file of %s",sRef.c_str());
    }
    if ( genomeSequence.open() ) {
      Logger::gLogger->error("Failed to open indexed karma-type reference file %s",sRef.c_str());
    }
  }

  BedMatrix* pBedMatrix = new BedMatrix;
  if ( !sBfile.empty() ) {
    if ( bBimAF ) {
      pBedMatrix->open((sBfile+".fam").c_str(),(sBfile+".bimp").c_str(),(sBfile+".bed").c_str());
    }
    else {
      pBedMatrix->open((sBfile+".fam").c_str(),(sBfile+".bim").c_str(),(sBfile+".bed").c_str());
    }
  }
  else if ( !sBimFile.empty() ) {
    bBimAF = true;
    pBedMatrix->openMarkerOnly(sBimFile.c_str());
  }
  else {
    Logger::gLogger->error("Neither --bfile nor --bimfile was specified");
  }

  // sanity check if the .bimp file loaded contains at least one positive allele frequency
  if ( bBimAF ) {
    uint32_t nZeros = 0;
    for(uint32_t i=0; i < pBedMatrix->vpGenMarkers.size(); ++i) {
      if ( pBedMatrix->vpGenMarkers[i]->dAlleleFrequency == 0 ) {
	++nZeros;
      }
    }
    if (nZeros == pBedMatrix->vpGenMarkers.size()) {
      Logger::gLogger->error("Allele frequencies are all zero. BIM file should contain additional column containing allele frequencies, and at least some of them should be nonzero");
    }
    else {
      Logger::gLogger->writeLog("Identified %u markers with zero minor allele frequencies and they are set to genotype error rate %lf",nZeros,genoError);
    }
  }

  uint32_t nMarkers = pBedMatrix->vpGenMarkers.size();
  uint32_t nInds = pBedMatrix->vpIndividuals.size();

  Logger::gLogger->writeLog("Total of %u markers over %u individuals exist in the genotype file",nMarkers,nInds);

  if ( bNoEOF ) {
    BgzfFileType::setRequireEofBlock(false);
  }

  std::string curChrom;
  if ( !sInPrefix.empty() ) {
    if ( !sIndexFile.empty() ) {
      Logger::gLogger->error("Cannot use --index option with --prefix and --suffix option");
    }

    // open current BAM file
    curChrom = pBedMatrix->vpGenMarkers[0]->sChrom;
    sInFile = sInPrefix + bimChr2ncbiChr(curChrom) + sInSuffix;
    sIndexFile = sInFile + ".bai";
  }

  SamFile inBam;
  SamFileHeader inHeader;
  if ( ! (inBam.OpenForRead(sInFile.c_str()))  ) {
    Logger::gLogger->error("Cannot open BAM file %s for reading - %s",sInFile.c_str(), SamStatus::getStatusString(inBam.GetStatus()) );
  }
  inBam.ReadHeader(inHeader);

  if ( sIndexFile.empty() ) {
    sIndexFile = sInFile + ".bai";
  }

  if ( ! inBam.ReadBamIndex( sIndexFile.c_str() ) ) {
    Logger::gLogger->error("Cannot open BAM file index %s for reading - %s", sIndexFile.c_str());
  }

  Logger::gLogger->writeLog("Reading header Records");

  SamHeaderRecord* pSamHeaderRecord;
  while( (pSamHeaderRecord = inHeader.getNextHeaderRecord()) != NULL ) {
    if ( pSamHeaderRecord->getType() == SamHeaderRecord::RG ) {
      std::string sRGID = pSamHeaderRecord->getTagValue("ID");
      std::string sSMID = pSamHeaderRecord->getTagValue("SM");
      if ( sRGID.empty() ) {
	Logger::gLogger->error("Readgroup ID is empty");
      }
      if ( sSMID.empty() ) {
	Logger::gLogger->warning("SM tag is missing in read group %s",sRGID.c_str());
      }
      vsRGIDs.push_back(sRGID);
      vsSMIDs.push_back(sSMID);
      if ( sBamSMID.empty() ) {
	sBamSMID = sSMID;
      }
      else if ( sBamSMID.compare(sSMID) != 0 ) {
	Logger::gLogger->warning("SM is not identical across the readGroups. Ignoring .bestSM/.selfSM outputs");
	sameSMFlag = false;
      }

      uint32_t idx = msRGidx.size();
      msRGidx[sRGID] = idx;
    }
  }

  uint32_t nCompInds = 0;
  int32_t nSelfIndex = -1;
  if ( bSelfOnly ) {
    if ( sameSMFlag ) {
      for(uint32_t i=0; i < nInds; ++i) {
	if ( sBamSMID.compare(pBedMatrix->vpIndividuals[i]->sIndID) == 0 ) {
	  if ( nSelfIndex < 0 ) {
	    nSelfIndex = static_cast<int32_t>(i);
	    pBedMatrix->vpIndividuals[0]->sIndID = pBedMatrix->vpIndividuals[i]->sIndID;
	  }
	  else {
	    Logger::gLogger->error("Multiple individuals with matching individual IDs (%s) in the PLINK file",sBamSMID.c_str());
	  }
	}
      }
      if ( nSelfIndex < 0 ) {
	Logger::gLogger->warning("--selfOnly was used but the individual in the BAM file cannot be found in the genotype file, skupping genotype comparison");
	nCompInds = 0;
      }
      else {
	nCompInds = 1;
      }
    }
    else {
      Logger::gLogger->error("The input BAM file consist of multiple individuals and cannot be run with --selfOnly option");
    }
  }
  else {
    nCompInds = nInds;
  }

  Logger::gLogger->writeLog("The following %u ReadGroup IDs are identified.",vsRGIDs.size());
  for(uint32_t i=0; i < vsRGIDs.size(); ++i) {
    Logger::gLogger->writeLog("\t%d: %s",i+1,vsRGIDs[i].c_str());
  }
  
  //SamStatus::Status returnStatus = SamStatus::SUCCESS;

  SamRecord samRecord;
  std::vector<uint32_t> nRGIndices;
  std::vector<char> cBases;
  std::vector<char> cQuals;
  CigarRoller cigarRoller;
  std::string cigar;
  uint32_t nRGs = vsRGIDs.size();

  // 1-dimensional (#ind) * (#geno) matrix of marker count (row-major)
  uint32_t* nIndGenoMarkers = new uint32_t[nCompInds * 3]();
  // (#ind) * (#geno) matrix of base count given marker genotypes
  uint32_t* nRGIndGenoMarkerBases = new uint32_t[nRGs * nCompInds * 3]();
  // (#ind) * (#geno) matrix of reference count given marker genotype
  uint32_t* nRGIndGenoMarkerRefs = new uint32_t[nRGs * nCompInds * 3]();
  // (#ind) * (#geno) matrix of non-reference count given marker genotype  
  uint32_t* nRGIndGenoMarkerAlts = new uint32_t[nRGs * nCompInds * 3]();

  // (#ind) * (#geno) matrix of base count given marker genotypes
  uint32_t* nSMIndGenoMarkerBases = new uint32_t[nCompInds * 3]();
  // (#ind) * (#geno) matrix of reference count given marker genotype
  uint32_t* nSMIndGenoMarkerRefs = new uint32_t[nCompInds * 3]();
  // (#ind) * (#geno) matrix of non-reference count given marker genotype  
  uint32_t* nSMIndGenoMarkerAlts = new uint32_t[nCompInds * 3]();

  // (#RG) * (#ind) * (#ind) * (#LLKs)
  double* fSumRGIndLLKs = new double[nRGs * nCompInds * numIBDs]();
  double* fRGGenoIBDLKs = new double[nRGs * 4]();
  double* fRGnonIBDLKs = new double[nRGs]();

  double* fSumSMIndLLKs = new double[nCompInds * numIBDs]();
  double* fSMGenoIBDLKs = new double[4]();
  double fSMnonIBDLK = 1.;

  double* fRGMixLLKs = new double[nRGs * numMixFracs]();
  double* fSMMixLLKs = new double[numMixFracs]();
  double* fRGMixMarkerLKs = new double[nRGs * 9];
  double* fSMMixMarkerLKs = new double[9];
  //double* fRGMixMarkerLKs = new double[nRGs * numMixFracs];
  //double* fSMMixMarkerLKs = new double[numMixFracs];

  uint32_t* nRGMarkerDepth = new uint32_t[nRGs * (maxDepth+1)]();
  uint32_t* nRGMarkerDepthRefOnly = new uint32_t[nRGs * (maxDepth+1)]();
  uint32_t* nRGMarkerDepthAltOnly = new uint32_t[nRGs * (maxDepth+1)]();
  uint32_t* nRGMarkerDepthRefAlt = new uint32_t[nRGs * (maxDepth+1)]();
  double* fRGSumEHetDepth = new double[nRGs * (maxDepth+1)]();

  uint32_t* nRGRefCnts = new uint32_t[nRGs]();
  uint32_t* nRGAltCnts = new uint32_t[nRGs]();
  uint32_t* nRGEtcCnts = new uint32_t[nRGs]();

  uint32_t* nSMMarkerDepth = new uint32_t[(maxDepth+1)]();
  uint32_t* nSMMarkerDepthRefOnly = new uint32_t[(maxDepth+1)]();
  uint32_t* nSMMarkerDepthAltOnly = new uint32_t[(maxDepth+1)]();
  uint32_t* nSMMarkerDepthRefAlt = new uint32_t[(maxDepth+1)]();
  double* fSMSumEHetDepth = new double[(maxDepth+1)]();

  uint32_t nSMRefCnt = 0;
  uint32_t nSMAltCnt = 0;
  uint32_t nSMEtcCnt = 0;

  Logger::gLogger->writeLog("Iterating over the genotypes..");
  //for( uint32_t markerCnt = 0; pBedMatrix->iterateMarker() && (markerCnt < 1000 ); ++markerCnt ) {

  // iterate each variable sites (marker)
  for( uint32_t markerCnt = 0; pBedMatrix->iterateMarker(); ++markerCnt ) {
    if ( (markerCnt+1) % 1000 == 0 )
      Logger::gLogger->writeLog("Processing %u / %u markers",markerCnt+1,nMarkers);
    
    // In 1000G files, one BAM file is splitted into each chromosome, this is the routine
    // to process those special cases
    if ( ( !sInPrefix.empty() ) 
	 && ( pBedMatrix->currentMarker()->sChrom.compare(curChrom) != 0 ) )  {
      inBam.Close();
      
      curChrom = pBedMatrix->currentMarker()->sChrom;
      std::string ncbiChr = bimChr2ncbiChr(pBedMatrix->currentMarker()->sChrom);
      Logger::gLogger->writeLog("Opening chromosome %s",ncbiChr.c_str());

      sInFile = sInPrefix + ncbiChr + sInSuffix;
      if ( ! (inBam.OpenForRead(sInFile.c_str())) ) {
	Logger::gLogger->error("Cannot open BAM file %s for reading - %s",sInFile.c_str(), SamStatus::getStatusString(inBam.GetStatus()) );	
      }
      inBam.ReadHeader(inHeader);
      inHeader.resetHeaderRecordIter();
      sIndexFile = sInFile + ".bai";

      if ( ! inBam.ReadBamIndex( sIndexFile.c_str() ) ) {
	Logger::gLogger->error("Cannot open BAM file index %s for reading - %s", sIndexFile.c_str());	
      }

      //Logger::gLogger->writeLog("foo");

      for( uint32_t k=0, rgCnt = 0; (pSamHeaderRecord = inHeader.getNextHeaderRecord()) != NULL; ++k ) {
	if ( pSamHeaderRecord->getType() == SamHeaderRecord::RG ) {
	  std::string sRGID = pSamHeaderRecord->getTagValue("ID");
	  std::string sSMID = pSamHeaderRecord->getTagValue("SM");
	  //Logger::gLogger->writeLog("bar %u %u %s %s",k,rgCnt,sRGID.c_str(),sSMID.c_str());

	  if ( sRGID.empty() ) {
	    Logger::gLogger->error("Readgroup ID is empty");
	  }
	  if ( sSMID.empty() ) {
	    Logger::gLogger->warning("SM tag is missing in read group %s",sRGID.c_str());
	  }

	  if ( vsRGIDs[rgCnt].compare(sRGID) != 0 ) {
	    Logger::gLogger->error("Nonidentical headers - ReadGroup ID is different at %u : %s vs %s",rgCnt,vsRGIDs[rgCnt].c_str(),sRGID.c_str());
	  }
	  if ( vsSMIDs[rgCnt].compare(sSMID) != 0 ) {
	    Logger::gLogger->error("Nonidentical headers - Readgroup SM is different at %u : %s vs %s",rgCnt,vsSMIDs[rgCnt].c_str(),sSMID.c_str());
	  }
	  ++rgCnt;
	}
      }

      //Logger::gLogger->writeLog("bar");
    }


    // Routine for processing each marker

    // Read the genotype file
    char* bedGenos = pBedMatrix->currentGenotypes();
    char a1 = pBedMatrix->currentMarker()->sAllele1[0];
    char a2 = pBedMatrix->currentMarker()->sAllele2[0];
    genomeIndex_t markerIndex = genomeSequence.getGenomePosition( pBedMatrix->currentMarker()->sChrom.c_str(), pBedMatrix->currentMarker()->nBasePosition );
    char refBase = genomeSequence[markerIndex];
    bool flipFlag = false;

    if ( refBase == a1 ) {
      // fine
    }
    else if ( refBase == a2 ) {
      // flip genotypes
      char tmp = a2;
      a2 = a1;
      a1 = tmp;
      for(uint32_t i=0; i < nInds; ++i) {
	if ( bedGenos[i] == 1 ) 
	  bedGenos[i] = 3;
	else if ( bedGenos[i] == 3 ) 
	  bedGenos[i] = 1;
      }
      flipFlag = true;
      //Logger::gLogger->writeLog("Marker alleles %s at %s:%u was flipped",pBedMatrix->currentMarker()->sMarkerID.c_str(), pBedMatrix->currentMarker()->sChrom.c_str(), pBedMatrix->currentMarker()->nBasePosition);

    }
    else {
      Logger::gLogger->error("Marker alleles %s at %s:%u do not match to reference allele",pBedMatrix->currentMarker()->sMarkerID.c_str(), pBedMatrix->currentMarker()->sChrom.c_str(), pBedMatrix->currentMarker()->nBasePosition);
    }

    // calculate non-reference allele frequency
    double alleleFreq = 0.0;
    uint32_t nMissing = 0;
    if ( !bBimAF ) {
      for(uint32_t i=0; i < nInds; ++i) {
	if ( bedGenos[i] == 0 ) {
	  ++nMissing;
	  alleleFreq *= static_cast<double>(nInds-nMissing+1)/static_cast<double>(nInds-nMissing);
	}
	else {
	  alleleFreq += static_cast<double>(bedGenos[i]-1)/static_cast<double>(2*(nInds-nMissing));
	}
      }
    }
    else {
      alleleFreq = pBedMatrix->currentMarker()->dAlleleFrequency;
      if ( !flipFlag ) {
	alleleFreq = 1. - alleleFreq;
      }
    }

    if ( ( alleleFreq < minAF ) || ( alleleFreq > 1.-minAF) ) {
      continue;
    }

    if ( alleleFreq < genoError ) alleleFreq = genoError;
    else if ( alleleFreq > 1.-genoError ) alleleFreq = 1.-genoError;

    //Logger::gLogger->writeLog("%c %c %c %lf",refBase,a1,a2,alleleFreq);

    // genotype-match likelihood
    for(uint32_t i=0; i < nRGs*4; ++i) {
      fRGGenoIBDLKs[i] = 1.;
    }
    for(uint32_t i=0; i < nRGs; ++i) {
      fRGnonIBDLKs[i] = 1.;
    }

    fSMGenoIBDLKs[0] = fSMGenoIBDLKs[1] = fSMGenoIBDLKs[2] = fSMGenoIBDLKs[3] = fSMnonIBDLK = 1.; 


    // retrieve the bases aligned to the marker position
    int refID = inHeader.GetReferenceID(pBedMatrix->currentMarker()->sChrom.c_str());
    uint32_t pos = pBedMatrix->currentMarker()->nBasePosition; // 1-based position
    int numSectionRecords = 0;
    
    nRGIndices.clear();
    cBases.clear();
    cQuals.clear();

    // **** Rouitine for reading each read
    if ( refID >= 0 ) {
      inBam.SetReadSection( refID, pos-1, pos ); // set bam file to retrieve the reads overlapping with the particular genomic position chr(refID):bp(pos) 

      // Keep reading records until they aren't anymore.
      while(inBam.ReadRecord(inHeader, samRecord)) {
	++numSectionRecords;

	// filtering step - mapQ
	if ( samRecord.getMapQuality() < minMapQ ) 
	  continue;

	// skip flagged reads
	uint16_t samFlags = samRecord.getFlag();
	if ( includeSamFlag && ( ( samFlags & includeSamFlag ) != includeSamFlag ) )
	  continue;
	if ( excludeSamFlag && ( samFlags & excludeSamFlag ) )
	  continue;

	// obtain readGroup info and store to rgIdx
	char tag[3];
	char vtype;
	void* value;
	bool found = false;
	uint32_t rgIdx;
	while( samRecord.getNextSamTag(tag, vtype, &value) != false ) {
	  if ( strcmp(tag, "RG") == 0 ) {
	    found = true;
	    if ( vtype == 'Z' ) {
	      std::string sValue = ((String)*(String*)value).c_str();
	      if ( msRGidx.find(sValue) != msRGidx.end() ) {
		rgIdx = msRGidx[sValue];
	      }
	      else {
		Logger::gLogger->error("ReadGroup ID %s cannot be found",sValue.c_str());
	      }
	    }
	    else {
	      Logger::gLogger->error("vtype of RG tag must be 'Z'");
	    }
	  }
	}
	if ( found == false ) {
	  Logger::gLogger->error("Cannot find RG tag for readName %s",samRecord.getReadName());
	}

	// access the base calls and qualities
	uint32_t readStartPosition = samRecord.get1BasedPosition();
	int32_t offset = pos - readStartPosition;
	const char* readQuality = samRecord.getQuality();
	const char* readSequence = samRecord.getSequence();

	cigar = samRecord.getCigar();
	cigarRoller.Set(cigar.c_str());

	if ( offset >= 0 ) {
	  int32_t readIndex = cigarRoller.getQueryIndex(offset);
	  if ( readIndex != CigarRoller::INDEX_NA ) {
	    if ( ( static_cast<uint32_t>(readQuality[readIndex]) >= minQ + 33 ) && ( readSequence[readIndex] != 'N' ) ) {
	      nRGIndices.push_back(rgIdx);
	      cBases.push_back(readSequence[readIndex]);
	      cQuals.push_back(readQuality[readIndex]);
	    }
	  }
	}
      }

      // at the end, we will have
      // nRGIndices - vector of readGroup indices
      // cBases - vector of bases
      // cQual - vector of base qualities

      if ( bSelfOnly ) {  
	if ( nSelfIndex >= 0 ) {
	  bedGenos[0] = bedGenos[nSelfIndex];
	}
      }

      for(uint32_t i=0; i < nCompInds; ++i) {
	if ( bedGenos[i] > 0 ) {
	  ++nIndGenoMarkers[3*i + bedGenos[i]-1];
	}
      }

      /*
      std::string sIndices;
      for(uint32_t j=0; j < nRGIndices.size(); ++j) {
	char buf[255];
	if ( j > 0 ) sIndices += " ";
	sprintf(buf,"%u:%c:%d",nRGIndices[j],cBases[j],static_cast<int>(cQuals[j])-33);
	sIndices += buf;
      }

      if ( !sIndices.empty() ) {
	Logger::gLogger->writeLog("Reading %s:%u (%c,%c)... %d records are found - %s", pBedMatrix->currentMarker()->sChrom.c_str(), pos, a1, a2, numSectionRecords, sIndices.c_str());
      }*/

      if ( nRGIndices.size() <= maxDepth ) {
	double probReadIBD[4], probReadnonIBD;
	for(uint32_t i=0; i < nRGs; ++i) {
	  nRGRefCnts[i] = 0;
	  nRGAltCnts[i] = 0;
	  nRGEtcCnts[i] = 0;
	}
	nSMRefCnt = nSMAltCnt = nSMEtcCnt = 0;

	for(uint32_t i=0; i < nRGIndices.size(); ++i) {
	  //double baseError = pow(10.,(33.-static_cast<double>(cQuals[i]))/10.);
	  //if ( baseError < 1e-4) { baseError = 1e-4; }
	  double baseError = fPhred2Err[cQuals[i]-33];

	  if ( cBases[i] == a1 ) {
	    probReadIBD[0] = probReadnonIBD = (1.-alleleFreq)*(1.-baseError) + alleleFreq * (baseError/3.);
	    probReadIBD[1] = 1.-baseError;
	    probReadIBD[2] = 0.5-baseError/3.;
	    probReadIBD[3] = baseError/3.;
	    ++nRGRefCnts[nRGIndices[i]];
	    ++nSMRefCnt;
	  }
	  else if ( cBases[i] == a2 ) {
	    probReadIBD[0] = probReadnonIBD = (alleleFreq)*(1.-baseError) + (1.-alleleFreq) * (baseError/3.);
	    probReadIBD[1] = baseError/3.;
	    probReadIBD[2] = 0.5-baseError/3.;
	    probReadIBD[3] = 1.-baseError;
	    ++nRGAltCnts[nRGIndices[i]];
	    ++nSMAltCnt;
	  }
	  else {
	    probReadIBD[0] = probReadIBD[1] = probReadIBD[2] = probReadIBD[3] = probReadnonIBD = baseError/3.;
	    ++nRGEtcCnts[nRGIndices[i]];
	    ++nSMEtcCnt;
	  }

	  for(uint32_t j=0; j < 4; ++j) {
	    fRGGenoIBDLKs[nRGIndices[i]*4+j] *= probReadIBD[j];
	    fSMGenoIBDLKs[j] *= probReadIBD[j];
	  }
	  fRGnonIBDLKs[nRGIndices[i]] *= probReadnonIBD;
	  fSMnonIBDLK *= probReadnonIBD;

	  for(uint32_t j=0; j < nCompInds; ++j) {
	    if ( bedGenos[j] > 0 ) {
	      ++nRGIndGenoMarkerBases[nRGIndices[i]*nCompInds*3 + j*3 + bedGenos[j]-1];
	      ++nSMIndGenoMarkerBases[j*3 + bedGenos[j]-1];
	      if ( cBases[i] == a1 ) {
		++nRGIndGenoMarkerRefs[nRGIndices[i]*nCompInds*3 + j*3 + bedGenos[j]-1];
		++nSMIndGenoMarkerRefs[j*3 + bedGenos[j]-1];
	      }
	      else if ( cBases[i] == a2 ) {
		++nRGIndGenoMarkerAlts[nRGIndices[i]*nCompInds*3 + j*3 + bedGenos[j]-1];
		++nSMIndGenoMarkerAlts[j*3 + bedGenos[j]-1];
	      }
	    }
	  }
	}

	for(uint32_t j=0; j < nRGs; ++j) {
	  uint32_t k = j*(maxDepth+1)+nRGRefCnts[j]+nRGAltCnts[j];
	  if ( nRGRefCnts[j] > 0 ) {
	    if ( nRGAltCnts[j] > 0 ) {
	      ++nRGMarkerDepthRefAlt[k];
	    }
	    else {
	      ++nRGMarkerDepthRefOnly[k];
	    }
	    ++nRGMarkerDepth[k];
	    fRGSumEHetDepth[k] += (2.*alleleFreq*(1.-alleleFreq));
	  }
	  else if ( nRGAltCnts[j] > 0 ) {
	    ++nRGMarkerDepth[k];
	    ++nRGMarkerDepthAltOnly[k];
	    fRGSumEHetDepth[k] += (2.*alleleFreq*(1.-alleleFreq));
	  }
	  else { // skip
	  }

	  //nSMRefCnt += nRGRefCnts[j];
	  //nSMAltCnt += nRGAltCnts[j];
	  //nSMEtcCnt += nRGEtcCnts[j];
	}

	if ( nSMRefCnt > 0 ) {
	  if ( nSMAltCnt > 0 ) {
	    ++nSMMarkerDepthRefAlt[nSMRefCnt+nSMAltCnt];
	  }
	  else {
	    ++nSMMarkerDepthRefOnly[nSMRefCnt+nSMAltCnt];
	  }
	  ++nSMMarkerDepth[nSMRefCnt+nSMAltCnt];
	  fSMSumEHetDepth[nSMRefCnt+nSMAltCnt] += (2.*alleleFreq*(1.-alleleFreq));
	}
	else if ( nSMAltCnt > 0 ) {
	  ++nSMMarkerDepth[nSMRefCnt+nSMAltCnt];
	  ++nSMMarkerDepthAltOnly[nSMRefCnt+nSMAltCnt];
	  fSMSumEHetDepth[nSMRefCnt+nSMAltCnt] += (2.*alleleFreq*(1.-alleleFreq));
	}

	// calculating per-RG LLK
	for(uint32_t i=0; i < nRGs; ++i) {
	  for(uint32_t j=0; j < nCompInds; ++j) {
	    double probReadsIBD = (bedGenos[j] > 0) ? ( (1.-1.5*genoError) * fRGGenoIBDLKs[ i*4 + bedGenos[j] ] + 0.5*genoError*(fRGGenoIBDLKs[i*4+1]+fRGGenoIBDLKs[i*4+2]+fRGGenoIBDLKs[i*4+3]) ) : fRGGenoIBDLKs[i*4+0];
	    for(uint32_t k=0; k < numIBDs; ++k) {
	      double llk = log( fIBDs[k]*probReadsIBD + (1.-fIBDs[k])*fRGnonIBDLKs[i] );

	      fSumRGIndLLKs[i*nCompInds*numIBDs+j*numIBDs+k] += llk;
	    }
	  }
	}

	// calcuating per-SM LLK
	for(uint32_t j=0; j < nCompInds; ++j) {
	  double probReadsIBD = (bedGenos[j] > 0) ? ( (1.-1.5*genoError) * fSMGenoIBDLKs[ bedGenos[j] ] + 0.5*genoError*(fSMGenoIBDLKs[1]+fSMGenoIBDLKs[2]+fSMGenoIBDLKs[3]) ) : fSMGenoIBDLKs[0];
	  for(uint32_t k=0; k < numIBDs; ++k) {
	    double llk = log( fIBDs[k]*probReadsIBD + (1.-fIBDs[k])*fSMnonIBDLK );
	    fSumSMIndLLKs[j*numIBDs+k] += llk;
	  }
	}


	// Goo's routine
	// Given : 
	//   - alleleFrequency
	//   - a1 : reference allele
	//   - a2 : alternative allele
	//   - nRGIndices
	//   - cBases
	//   - cQuals
	double genoFreq[3] = {(1.-alleleFreq)*(1.-alleleFreq), 2*alleleFreq*(1.-alleleFreq), alleleFreq*alleleFreq};

	for(uint32_t j=0; j < numMixFracs; ++j) {
	  double alpha = fMixFracs[j];
	  for(uint32_t i=0; i < 9; ++i) {
	    fSMMixMarkerLKs[i] = 1.;
	  }
	  for(uint32_t i=0; i < 9*nRGs; ++i) {
	    fRGMixMarkerLKs[i] = 1.;
	  }
	  //double valuesToMultiply[9] = {1.,1.,1.,1.,1.,1.,1.,1.,1.};
	  double lks[9];
	  for(uint32_t i=0; i < nRGIndices.size(); ++i) {
	    double baseError = fPhred2Err[cQuals[i]-33];
	    double baseMatch = 1.-baseError;
	    if ( cBases[i] == a1 ) {
	      lks[0*3+0] = baseMatch;
	      lks[0*3+1] = (0.5+0.5*alpha)*baseMatch + ((1.-alpha)/6.)*baseError;
	      lks[0*3+2] = alpha*baseMatch + ((1.-alpha)/3.)*baseError;
	      lks[1*3+0] = (1.-0.5*alpha)*baseMatch + (alpha/6.)*baseError;
	      lks[1*3+1] = 0.5*baseMatch + 1./6.*baseError;
	      lks[1*3+2] = (0.5*alpha)*baseMatch + (1./3.-alpha/6.)*baseError;
	      lks[2*3+0] = (1.-alpha)*baseMatch + (alpha/3.)*baseError;
	      lks[2*3+1] = (0.5*(1.-alpha))*baseMatch + (1./6.+alpha/6.)*baseError;
	      lks[2*3+2] = baseError/3.;
	    }
	    else if ( cBases[i] == a2 ) {
	      lks[2*3+2] = baseMatch;
	      lks[2*3+1] = (0.5+0.5*alpha)*baseMatch + ((1.-alpha)/6.)*baseError;
	      lks[2*3+0] = alpha*baseMatch + ((1.-alpha)/3.)*baseError;
	      lks[1*3+2] = (1.-0.5*alpha)*baseMatch + (alpha/6.)*baseError;
	      lks[1*3+1] = 0.5*baseMatch + 1./6.*baseError;
	      lks[1*3+0] = (0.5*alpha)*baseMatch + (1./3.-alpha/6.)*baseError;
	      lks[0*3+2] = (1.-alpha)*baseMatch + (alpha/3.)*baseError;
	      lks[0*3+1] = (0.5*(1.-alpha))*baseMatch + (1./6.+alpha/6.)*baseError;
	      lks[0*3+0] = baseError/3.;
	    }
	    else {
	      lks[0*3+0] = baseError/3.;
	      lks[0*3+1] = baseError/3.;
	      lks[0*3+2] = baseError/3.;
	      lks[1*3+0] = baseError/3.;
	      lks[1*3+1] = baseError/3.;
	      lks[1*3+2] = baseError/3.;
	      lks[2*3+0] = baseError/3.;
	      lks[2*3+1] = baseError/3.;
	      lks[2*3+2] = baseError/3.;
	    }

	    for(uint32_t k=0; k < 9; ++k) {
	      //valuesToMultiply[k] *= lks[k];
	      fSMMixMarkerLKs[k] *= lks[k];
	      fRGMixMarkerLKs[nRGIndices[i] * 9 + k] *= lks[k];
	    }
	  }
	  
	  double perMarkerProb = 0;
	  for(uint32_t k1 = 0; k1 < 3; ++k1) {
	    for(uint32_t k2 = 0; k2 < 3; ++k2) {
	      perMarkerProb += (fSMMixMarkerLKs[k1*3+k2]*genoFreq[k1]*genoFreq[k2]);
	    }
	  }
	  fSMMixLLKs[j] += log(perMarkerProb);

	  for(uint32_t k=0; k < nRGs; ++k) {
	    perMarkerProb = 0;
	    for(uint32_t k1 = 0; k1 < 3; ++k1) {
	      for(uint32_t k2 = 0; k2 < 3; ++k2) {
		perMarkerProb += (fRGMixMarkerLKs[k*9+k1*3+k2]*genoFreq[k1]*genoFreq[k2]);
	      }
	    }
	    fRGMixLLKs[k*numMixFracs + j] += log(perMarkerProb);
	  }
	}
      }
    }
    else {
      //abort();
    }
    //Logger::gLogger->writeLog("Finished Reading %u records",numSectionRecords);
  }

  Logger::gLogger->writeLog("Writing output file %s",sOutFile.c_str());

  double* bestRGAllIBDs = new double[nRGs]();
  uint32_t* bestRGAllSMIndices = new uint32_t[nRGs]();
  double* bestRGAllLLKs = new double[nRGs]();

  double* bestRGSelfIBDs = new double[nRGs]();
  double* bestRGSelfLLKs = new double[nRGs]();

  FILE* fpBest = NULL;
  double bestSMAllIBD = 0;
  uint32_t bestSMAllSMIndex = 0;
  double bestSMAllLLK = 0-DBL_MAX;
  double bestSMSelfLLK = 0-DBL_MAX;
  double bestSMSelfIBD = 0;
  
  if ( nCompInds > 0 ) {
    for(uint32_t i=0; i < nRGs; ++i) {
      bestRGAllLLKs[i] = 0-DBL_MAX;
      bestRGSelfLLKs[i] = 0-DBL_MAX;
      for(uint32_t j=0; j < nCompInds; ++j) {
	for(uint32_t k=0; k < numIBDs; ++k) {
	  if ( fSumRGIndLLKs[i*nCompInds*numIBDs+j*numIBDs+k] > bestRGAllLLKs[i] ) {
	    bestRGAllLLKs[i] = fSumRGIndLLKs[i*nCompInds*numIBDs+j*numIBDs+k];
	    bestRGAllSMIndices[i] = j;
	    bestRGAllIBDs[i] = fIBDs[k];
	  }
	}
      }
      
      int32_t matchIndex = -1;
      for(uint32_t k=0; k < nCompInds; ++k) {
	if ( vsSMIDs[i].compare(pBedMatrix->vpIndividuals[k]->sIndID) == 0 ) {
	  matchIndex = k;
	  break;
	}
      }
      if ( matchIndex < 0 ) {
	bestRGSelfLLKs[i] = 0-DBL_MAX;
	bestRGSelfIBDs[i] = 0-DBL_MAX;
      }
      else {
	for(uint32_t k=0; k < numIBDs; ++k) {
	  if ( fSumRGIndLLKs[i*nCompInds*numIBDs+matchIndex*numIBDs+k] > bestRGSelfLLKs[i] ) {
	    bestRGSelfLLKs[i] = fSumRGIndLLKs[i*nCompInds*numIBDs+matchIndex*numIBDs+k];
	    bestRGSelfIBDs[i] = fIBDs[k];
	  }
	}
      }
    }

    {
      for(uint32_t j=0; j < nCompInds; ++j) {
	for(uint32_t k=0; k < numIBDs; ++k) {
	  if ( fSumSMIndLLKs[j*numIBDs+k] > bestSMAllLLK ) {
	    bestSMAllLLK = fSumSMIndLLKs[j*numIBDs+k];
	    bestSMAllSMIndex = j;
	    bestSMAllIBD = fIBDs[k];
	  }
	}
      }
      
      int32_t matchIndex = -1;
      for(uint32_t k=0; k < nCompInds; ++k) {
	if ( vsSMIDs[0].compare(pBedMatrix->vpIndividuals[k]->sIndID) == 0 ) {
	  matchIndex = k;
	  break;
	}
      }
      if ( matchIndex < 0 ) {
	bestSMSelfLLK = 0-DBL_MAX;
	bestSMSelfIBD = 0-DBL_MAX;
      }
      else {
	for(uint32_t k=0; k < numIBDs; ++k) {
	  if ( fSumSMIndLLKs[matchIndex*numIBDs+k] > bestSMSelfLLK ) {
	    bestSMSelfLLK = fSumSMIndLLKs[matchIndex*numIBDs+k];
	    bestSMSelfIBD = fIBDs[k];
	  }
	}
      }
    }

    fpBest = fopen((sOutFile+".bestRG").c_str(),"w");
    if ( fpBest == NULL ) {
      Logger::gLogger->error("Cannot open output file %s",sOutFile.c_str());        
    }
    fprintf(fpBest,"SEQ_SM\tRG\tBEST_SM\tBESTIBD\tBESTIBDLLK\tBESTIBDLLK-\t#GENOS\t#BASES\t%%GENREF\t%%GENHET\t%%GENALT\tDPREF\tRDPHET\tRDPALT\tREF-A1%%\tREF-A2%%\tHET-A1%%\tHET-A2%%\tALT-A1%%\tALT-A2%%\t#DP>2\t%%HETAF\t%%HETSEQ\tEXHET\t%%MIX\tBESTMIXLLK\tBESTMIXLLK-\n");
  }

  FILE* fpSelf = fopen((sOutFile+".selfRG").c_str(),"w");
  if ( fpSelf == NULL ) {
    Logger::gLogger->error("Cannot open output file %s",sOutFile.c_str());        
  }
  fprintf(fpSelf,"SEQ_SM\tRG\tSELF_SM\tSELFIBD\tSELFIBDLLK\tSELFIBDLLK-\t#GENOS\t#BASES\t%%GENREF\t%%GENHET\t%%GENALT\tDPREF\tRDPHET\tRDPALT\tREF-A1%%\tREF-A2%%\tHET-A1%%\tHET-A2%%\tALT-A1%%\tALT-A2%%\t#DP>2\t%%HETAF\t%%HETSEQ\tEXHET\t%%MIX\tBESTMIXLLK\tBESTMIXLLK-\n");

  for(uint32_t i=0; i < nRGs; ++i) {
    // check if the SMIDs exist in the individuals
    int32_t matchIndex = -1;
    uint32_t j = bestRGAllSMIndices[i];
    for(uint32_t k=0; k < nCompInds; ++k) {
      if ( vsSMIDs[i].compare(pBedMatrix->vpIndividuals[k]->sIndID) == 0 ) {
	matchIndex = k;
	break;
      }
    }

    double bestHetLLK = 0-DBL_MAX;
    double pureHetLLK, bestMixFrac = 0;
    for(uint32_t k=0; k < numMixFracs; ++k) {
      if ( bestHetLLK < fRGMixLLKs[i*numMixFracs + k] ) {
	bestHetLLK = fRGMixLLKs[i*numMixFracs+k];
	bestMixFrac = fMixFracs[k];
      }
    }
    pureHetLLK = fRGMixLLKs[i*numMixFracs];

    if ( nCompInds > 0 ) {
      fprintf(fpBest,"%s",vsSMIDs[i].c_str());  // SEQ_SM
      fprintf(fpBest,"\t%s",vsRGIDs[i].c_str());  // RG
      fprintf(fpBest,"\t%s\t%.3lf\t%.3le",pBedMatrix->vpIndividuals[j]->sIndID.c_str(),bestRGAllIBDs[i],bestRGAllLLKs[i]); // BEST_SM, BESTIBD, BESTLLK
      if ( matchIndex < 0 ) {
	fprintf(fpBest,"\tN/A");
      }
      else {
	fprintf(fpBest,"\t%.3le",bestRGAllLLKs[i]-fSumRGIndLLKs[i*nCompInds*numIBDs+matchIndex*numIBDs+numIBDs-1]); // BESTLLK-
      }
    }
    
    fprintf(fpSelf,"%s",vsSMIDs[i].c_str());  // SEQ_SM
    fprintf(fpSelf,"\t%s",vsRGIDs[i].c_str());  // RG
    if ( matchIndex < 0 ) {
      fprintf(fpSelf,"\tN/A\tN/A\tN/A\tN/A");
    }
    else {
      fprintf(fpSelf,"\t%s\t%.3lf\t%.3le",vsSMIDs[i].c_str(),bestRGSelfIBDs[i],bestRGSelfLLKs[i]); // SELF_SM, SELFIBD, SELFLLK
      fprintf(fpSelf,"\t%.3le",bestRGSelfLLKs[i]-fSumRGIndLLKs[i*nCompInds*numIBDs+matchIndex*numIBDs+numIBDs-1]); // BESTLLK-
    }
      
    uint32_t nGenos, nBases;
    double refDepth, hetDepth, altDepth;
    if ( nCompInds > 0 ) {
      nGenos = nIndGenoMarkers[3*j+0]+nIndGenoMarkers[3*j+1]+nIndGenoMarkers[3*j+2];
      fprintf(fpBest,"\t%u",nGenos); // #GENOS
      nBases = nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0]+nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1]+nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2];
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%u",nBases); // #BASES
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+0])/static_cast<double>(nGenos)); // %GENREF
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+1])/static_cast<double>(nGenos)); // %GENHET
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+2])/static_cast<double>(nGenos)); // %GENALT
      refDepth = static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0])/static_cast<double>(nIndGenoMarkers[3*j+0]);
      hetDepth = static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1])/static_cast<double>(nIndGenoMarkers[3*j+1]);
      altDepth = static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2])/static_cast<double>(nIndGenoMarkers[3*j+2]);
      fprintf(fpBest,"\t%.4lf\t%.5lf\t%.5lf",refDepth,hetDepth/refDepth,altDepth/refDepth);
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nRGIndGenoMarkerRefs[i*nCompInds*3+j*3+0])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0])); //REF_A1%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nRGIndGenoMarkerAlts[i*nCompInds*3+j*3+0])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0])); //REF_A2%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nRGIndGenoMarkerRefs[i*nCompInds*3+j*3+1])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1])); //HET_A1%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nRGIndGenoMarkerAlts[i*nCompInds*3+j*3+1])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1])); //HET_A2%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nRGIndGenoMarkerRefs[i*nCompInds*3+j*3+2])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2])); //ALT_A1%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nRGIndGenoMarkerAlts[i*nCompInds*3+j*3+2])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2])); //ALT_A2%
    }
      
    if ( matchIndex < 0 ) {
      fprintf(fpSelf,"\tN/A"); // #GENOS
      fprintf(fpSelf,"\tN/A"); // #BASES
      fprintf(fpSelf,"\tN/A"); // %GENREF
      fprintf(fpSelf,"\tN/A"); // %GENHET
      fprintf(fpSelf,"\tN/A"); // %GENALT
      fprintf(fpSelf,"\tN/A"); // DPREF
      fprintf(fpSelf,"\tN/A"); // RDPHET
      fprintf(fpSelf,"\tN/A"); // RDPALT
      fprintf(fpSelf,"\tN/A"); // REF-A1%
      fprintf(fpSelf,"\tN/A"); // REF-A2%
      fprintf(fpSelf,"\tN/A"); // HET-A1%
      fprintf(fpSelf,"\tN/A"); // HET-A2%
      fprintf(fpSelf,"\tN/A"); // ALT-A1%
      fprintf(fpSelf,"\tN/A"); // ALT-A2%
    }
    else {
      j = matchIndex;
      nGenos = nIndGenoMarkers[3*j+0]+nIndGenoMarkers[3*j+1]+nIndGenoMarkers[3*j+2];
      fprintf(fpSelf,"\t%u",nGenos); // #GENOS
      nBases = nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0]+nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1]+nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2];
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%u",nBases); // #BASES
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+0])/static_cast<double>(nGenos)); // %GENREF
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+1])/static_cast<double>(nGenos)); // %GENHET
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+2])/static_cast<double>(nGenos)); // %GENALT

      refDepth = static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0])/static_cast<double>(nIndGenoMarkers[3*j+0]);
      hetDepth = static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1])/static_cast<double>(nIndGenoMarkers[3*j+1]);
      altDepth = static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2])/static_cast<double>(nIndGenoMarkers[3*j+2]);

      fprintf(fpSelf,"\t%.4lf\t%.5lf\t%.5lf",refDepth,hetDepth/refDepth,altDepth/refDepth);
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nRGIndGenoMarkerRefs[i*nCompInds*3+j*3+0])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0])); //REF_A1%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nRGIndGenoMarkerAlts[i*nCompInds*3+j*3+0])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+0])); //REF_A2%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nRGIndGenoMarkerRefs[i*nCompInds*3+j*3+1])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1])); //HET_A1%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nRGIndGenoMarkerAlts[i*nCompInds*3+j*3+1])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+1])); //HET_A2%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nRGIndGenoMarkerRefs[i*nCompInds*3+j*3+2])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2])); //ALT_A1%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nRGIndGenoMarkerAlts[i*nCompInds*3+j*3+2])/static_cast<double>(nRGIndGenoMarkerBases[i*nCompInds*3+j*3+2])); //ALT_A2%
    }
    
    uint32_t nDP2 = 0;
    double fEHetDP2AF = 0.;
    double fEHetDP2SEQ = 0.;
    for(uint32_t k=2; k <= maxDepth; ++k) {
      nDP2 += nRGMarkerDepth[i*(maxDepth+1)+k];
      fEHetDP2AF += fRGSumEHetDepth[i*(maxDepth+1)+k];
      fEHetDP2SEQ += (static_cast<double>(nRGMarkerDepthRefAlt[i*(maxDepth+1)+k])/(1.-pow(0.5,static_cast<double>(k-1))));
    }
    fEHetDP2AF /= static_cast<double>(nDP2);
    fEHetDP2SEQ /= static_cast<double>(nDP2);
    if ( nCompInds > 0 ) {
      fprintf(fpBest,"\t%u\t%.5lf\t%.5lf\t%.5lf",nDP2,fEHetDP2AF,fEHetDP2SEQ,fEHetDP2SEQ/fEHetDP2AF);
      fprintf(fpBest,"\t%.3lf\t%.3le\t%.3le",bestMixFrac,bestHetLLK,bestHetLLK-pureHetLLK);
      fprintf(fpBest,"\n");
    }

    fprintf(fpSelf,"\t%u\t%.5lf\t%.5lf\t%.5lf",nDP2,fEHetDP2AF,fEHetDP2SEQ,fEHetDP2SEQ/fEHetDP2AF);
    fprintf(fpSelf,"\t%.3lf\t%.3le\t%.3le",bestMixFrac,bestHetLLK,bestHetLLK-pureHetLLK);
    fprintf(fpSelf,"\n");
  }
  if ( nCompInds > 0 ) fclose(fpBest);
  fclose(fpSelf);

  if ( sameSMFlag ) {
    if ( nCompInds > 0 ) {
      fpBest = fopen((sOutFile+".bestSM").c_str(),"w");
      if ( fpBest == NULL ) {
	Logger::gLogger->error("Cannot open output file %s",sOutFile.c_str());    
      }
      fprintf(fpBest,"SEQ_SM\tRG\tBEST_SM\tBESTIBD\tBESTIBDLLK\tBESTIBDLLK-\t#GENOS\t#BASES\t%%GENREF\t%%GENHET\t%%GENALT\tDPREF\tRDPHET\tRDPALT\tREF-A1%%\tREF-A2%%\tHET-A1%%\tHET-A2%%\tALT-A1%%\tALT-A2%%\t#DP>2\t%%HETAF\t%%HETSEQ\tEXHET\t%%MIX\tBESTMIXLLK\tBESTMIXLLK-\n");
    }
    
    fpSelf = fopen((sOutFile+".selfSM").c_str(),"w");
    if ( fpSelf == NULL ) {
      Logger::gLogger->error("Cannot open output file %s",sOutFile.c_str());    
    }
    fprintf(fpSelf,"SEQ_SM\tRG\tSELF_SM\tSELFIBD\tSELFIBDLLK\tSELFIBDLLK-\t#GENOS\t#BASES\t%%GENREF\t%%GENHET\t%%GENALT\tDPREF\tRDPHET\tRDPALT\tREF-A1%%\tREF-A2%%\tHET-A1%%\tHET-A2%%\tALT-A1%%\tALT-A2%%\t#DP>2\t%%HETAF\t%%HETSEQ\tEXHET\t%%MIX\tBESTMIXLLK\tBESTMIXLLK-\n");

    // check if the SMIDs exist in the individuals
    int32_t matchIndex = -1;
    uint32_t j = bestSMAllSMIndex;
    for(uint32_t k=0; k < nCompInds; ++k) {
      if ( vsSMIDs[0].compare(pBedMatrix->vpIndividuals[k]->sIndID) == 0 ) {
	matchIndex = k;
	break;
      }
    }

    if ( nCompInds > 0 ) {
      fprintf(fpBest,"%s",vsSMIDs[0].c_str());  // SEQ_SM
      fprintf(fpBest,"\tN/A");  // RG = N/A
      fprintf(fpBest,"\t%s\t%.3lf\t%.3le",pBedMatrix->vpIndividuals[j]->sIndID.c_str(),bestSMAllIBD,bestSMAllLLK); // BEST_SM, BESTIBD, BESTLLK
      if ( matchIndex < 0 ) {
	fprintf(fpBest,"\tN/A");
      }
      else {
	fprintf(fpBest,"\t%.3le",bestSMAllLLK-fSumSMIndLLKs[matchIndex*numIBDs+numIBDs-1]); // BESTLLK-
      }
    }
    
    fprintf(fpSelf,"%s",vsSMIDs[0].c_str());  // SEQ_SM
    fprintf(fpSelf,"\tN/A");  // RG
    if ( matchIndex < 0 ) {
      fprintf(fpSelf,"\tN/A\tN/A\tN/A\tN/A");
    }
    else {
      fprintf(fpSelf,"\t%s\t%.3lf\t%.3le",vsSMIDs[0].c_str(),bestSMSelfIBD,bestSMSelfLLK); // SELF_SM, SELFIBD, SELFLLK
      fprintf(fpSelf,"\t%.3le",bestSMSelfLLK-fSumSMIndLLKs[matchIndex*numIBDs+numIBDs-1]); // BESTLLK-
    }
      
    uint32_t nGenos, nBases;
    double refDepth, hetDepth, altDepth;
    if ( nCompInds > 0 ) {
      nGenos = nIndGenoMarkers[3*j+0]+nIndGenoMarkers[3*j+1]+nIndGenoMarkers[3*j+2];
      fprintf(fpBest,"\t%u",nGenos); // #GENOS
      nBases = nSMIndGenoMarkerBases[j*3+0]+nSMIndGenoMarkerBases[j*3+1]+nSMIndGenoMarkerBases[j*3+2];
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%u",nBases); // #BASES
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+0])/static_cast<double>(nGenos)); // %GENREF
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+1])/static_cast<double>(nGenos)); // %GENHET
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+2])/static_cast<double>(nGenos)); // %GENALT
      refDepth = static_cast<double>(nSMIndGenoMarkerBases[j*3+0])/static_cast<double>(nIndGenoMarkers[3*j+0]);
      hetDepth = static_cast<double>(nSMIndGenoMarkerBases[j*3+1])/static_cast<double>(nIndGenoMarkers[3*j+1]);
      altDepth = static_cast<double>(nSMIndGenoMarkerBases[j*3+2])/static_cast<double>(nIndGenoMarkers[3*j+2]);
      fprintf(fpBest,"\t%.4lf\t%.5lf\t%.5lf",refDepth,hetDepth/refDepth,altDepth/refDepth);
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nSMIndGenoMarkerRefs[j*3+0])/static_cast<double>(nSMIndGenoMarkerBases[j*3+0])); //REF_A1%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nSMIndGenoMarkerAlts[j*3+0])/static_cast<double>(nSMIndGenoMarkerBases[j*3+0])); //REF_A2%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nSMIndGenoMarkerRefs[j*3+1])/static_cast<double>(nSMIndGenoMarkerBases[j*3+1])); //HET_A1%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nSMIndGenoMarkerAlts[j*3+1])/static_cast<double>(nSMIndGenoMarkerBases[j*3+1])); //HET_A2%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nSMIndGenoMarkerRefs[j*3+2])/static_cast<double>(nSMIndGenoMarkerBases[j*3+2])); //ALT_A1%
      fprintf(fpBest,"\t");
      fprintf(fpBest,"%.5lf",static_cast<double>(nSMIndGenoMarkerAlts[j*3+2])/static_cast<double>(nSMIndGenoMarkerBases[j*3+2])); //ALT_A2%
    }

    if ( matchIndex < 0 ) {
      fprintf(fpSelf,"\tN/A"); // #GENOS
      fprintf(fpSelf,"\tN/A"); // #BASES
      fprintf(fpSelf,"\tN/A"); // %GENREF
      fprintf(fpSelf,"\tN/A"); // %GENHET
      fprintf(fpSelf,"\tN/A"); // %GENALT
      fprintf(fpSelf,"\tN/A"); // DPREF
      fprintf(fpSelf,"\tN/A"); // RDPHET
      fprintf(fpSelf,"\tN/A"); // RDPALT
      fprintf(fpSelf,"\tN/A"); // REF-A1%
      fprintf(fpSelf,"\tN/A"); // REF-A2%
      fprintf(fpSelf,"\tN/A"); // HET-A1%
      fprintf(fpSelf,"\tN/A"); // HET-A2%
      fprintf(fpSelf,"\tN/A"); // ALT-A1%
      fprintf(fpSelf,"\tN/A"); // ALT-A2%
    }
    else {
      j = matchIndex;
      nGenos = nIndGenoMarkers[3*j+0]+nIndGenoMarkers[3*j+1]+nIndGenoMarkers[3*j+2];
      fprintf(fpSelf,"\t%u",nGenos); // #GENOS
      nBases = nSMIndGenoMarkerBases[j*3+0]+nSMIndGenoMarkerBases[j*3+1]+nSMIndGenoMarkerBases[j*3+2];
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%u",nBases); // #BASES
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+0])/static_cast<double>(nGenos)); // %GENREF
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+1])/static_cast<double>(nGenos)); // %GENHET
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nIndGenoMarkers[3*j+2])/static_cast<double>(nGenos)); // %GENALT
      refDepth = static_cast<double>(nSMIndGenoMarkerBases[j*3+0])/static_cast<double>(nIndGenoMarkers[3*j+0]);
      hetDepth = static_cast<double>(nSMIndGenoMarkerBases[j*3+1])/static_cast<double>(nIndGenoMarkers[3*j+1]);
      altDepth = static_cast<double>(nSMIndGenoMarkerBases[j*3+2])/static_cast<double>(nIndGenoMarkers[3*j+2]);
      fprintf(fpSelf,"\t%.4lf\t%.5lf\t%.5lf",refDepth,hetDepth/refDepth,altDepth/refDepth);
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nSMIndGenoMarkerRefs[j*3+0])/static_cast<double>(nSMIndGenoMarkerBases[j*3+0])); //REF_A1%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nSMIndGenoMarkerAlts[j*3+0])/static_cast<double>(nSMIndGenoMarkerBases[j*3+0])); //REF_A2%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nSMIndGenoMarkerRefs[j*3+1])/static_cast<double>(nSMIndGenoMarkerBases[j*3+1])); //HET_A1%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nSMIndGenoMarkerAlts[j*3+1])/static_cast<double>(nSMIndGenoMarkerBases[j*3+1])); //HET_A2%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nSMIndGenoMarkerRefs[j*3+2])/static_cast<double>(nSMIndGenoMarkerBases[j*3+2])); //ALT_A1%
      fprintf(fpSelf,"\t");
      fprintf(fpSelf,"%.5lf",static_cast<double>(nSMIndGenoMarkerAlts[j*3+2])/static_cast<double>(nSMIndGenoMarkerBases[j*3+2])); //ALT_A2%
    }
    
    uint32_t nDP2 = 0;
    double fEHetDP2AF = 0.;
    double fEHetDP2SEQ = 0.;
    for(uint32_t k=2; k <= maxDepth; ++k) {
      nDP2 += nSMMarkerDepth[k];
      fEHetDP2AF += fSMSumEHetDepth[k];
      fEHetDP2SEQ += (static_cast<double>(nSMMarkerDepthRefAlt[k])/(1.-pow(0.5,static_cast<double>(k-1))));
    }
    fEHetDP2AF /= static_cast<double>(nDP2);
    fEHetDP2SEQ /= static_cast<double>(nDP2);
    if ( nCompInds > 0 ) fprintf(fpBest,"\t%u\t%.5lf\t%.5lf\t%.5lf",nDP2,fEHetDP2AF,fEHetDP2SEQ,fEHetDP2SEQ/fEHetDP2AF);
    fprintf(fpSelf,"\t%u\t%.5lf\t%.5lf\t%.5lf",nDP2,fEHetDP2AF,fEHetDP2SEQ,fEHetDP2SEQ/fEHetDP2AF);

    double bestHetLLK = 0-DBL_MAX;
    double pureHetLLK, bestMixFrac = 0;
    for(uint32_t k=0; k < numMixFracs; ++k) {
      if ( bestHetLLK < fSMMixLLKs[k] ) {
	bestHetLLK = fSMMixLLKs[k];
	bestMixFrac = fMixFracs[k];
      }
    }
    pureHetLLK = fSMMixLLKs[0];    
    
    if ( nCompInds > 0 ) fprintf(fpBest,"\t%.3lf\t%.3le\t%.3le",bestMixFrac,bestHetLLK,bestHetLLK-pureHetLLK);
    fprintf(fpSelf,"\t%.3lf\t%.3le\t%.3le",bestMixFrac,bestHetLLK,bestHetLLK-pureHetLLK);

    if ( nCompInds > 0 ) fprintf(fpBest,"\n");
    fprintf(fpSelf,"\n");
  }
  if ( nCompInds > 0 ) fclose(fpBest);
  fclose(fpSelf);

  Logger::gLogger->writeLog("Finished writing output files %s.{bestRG,selfRG,bestSM,selfSM}",sOutFile.c_str());

  /*
  delete [] nIndGenoMarkers;
  delete [] nRGIndGenoMarkerBases;
  delete [] nRGIndGenoMarkerRefs;
  delete [] nRGIndGenoMarkerAlts;
  delete [] fSumRGIndLLKs;
  delete [] fRGGenoIBDLKs;
  delete [] fRGnonIBDLKs;
  delete [] bestIBDs;
  delete [] bestSMIndices;
  delete [] bestLLKs;
  delete [] nRGMarkerDepth;
  delete [] nRGMarkerDepthRefOnly;
  delete [] nRGMarkerDepthRefAlt;
  delete [] nRGMarkerDepthAltOnly;
  delete [] nTotalMarkerDepth;
  delete [] nTotalMarkerDepthRefOnly;
  delete [] nTotalMarkerDepthRefAlt;
  delete [] nTotalMarkerDepthAltOnly;
  delete [] nRGRefCnts;
  delete [] nRGAltCnts;
  delete [] nRGEtcCnts;*/

  return 0;
}
