/*
 *  Copyright (C) 2010  Regents of the University of Michigan
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Validate.h"

void validateRead1(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 1
    // Create record structure for validating.
    int expectedBlockSize = 89;
    const char* expectedReferenceName = "1";
    const char* expectedMateReferenceName = "1";
    const char* expectedMateReferenceNameOrEqual = "=";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    const bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 0;
    expectedRecordPtr->myPosition = 1010;
    expectedRecordPtr->myReadNameLength = 23;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 2;
    expectedRecordPtr->myFlag = 73;
    expectedRecordPtr->myReadLength = 5;
    expectedRecordPtr->myMateReferenceID = 0;
    expectedRecordPtr->myMatePosition = 1010;
    expectedRecordPtr->myInsertSize = 0;
   
    assert(samRecord.getString("MD") == "37");
    assert(samRecord.getString("YZ") == "");
    assert(samRecord.getInteger("YZ") == -1);
    assert(samRecord.getDouble("YZ") == -1);
    // Check the alignment end
    assert(samRecord.get0BasedAlignmentEnd() == 1016);
    assert(samRecord.get1BasedAlignmentEnd() == 1017);
    assert(samRecord.getAlignmentLength() == 7);
    assert(samRecord.get0BasedUnclippedStart() == 1010);
    assert(samRecord.get1BasedUnclippedStart() == 1011);
    assert(samRecord.get0BasedUnclippedEnd() == 1016);
    assert(samRecord.get1BasedUnclippedEnd() == 1017);

    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() == 
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() == 
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(),
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() ==
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() == 
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "1:1011:F:255+17M15D20M") == 0);
    assert(strcmp(samRecord.getCigar(), "5M2D") == 0);
    assert(strcmp(samRecord.getSequence(), "CCGAA") == 0);
    assert(strcmp(samRecord.getQuality(), "6>6+4") == 0);

    // Check the tags.
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'A');
    assert(tag[1] == 'M');
    assert(type == 'i');
    assert(*(char*)value == 0);
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'M');
    assert(tag[1] == 'D');
    assert(type == 'Z');
    assert(*(String*)value == "37");
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'N');
    assert(tag[1] == 'M');
    assert(type == 'i');
    assert(*(char*)value == 0);
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'X');
    assert(tag[1] == 'T');
    assert(type == 'A');
    assert(*(char*)value == 'R');
    // No more tags, should return false.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (const bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID == 
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // Validate the cigar.
    // The First cigar is 5M which is 5 << 4 | 0 = 80
    assert(*(unsigned int*)varPtr == 80);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // The 2nd cigar is 2D which is 2 << 4 | 2 = 34
    assert(*(unsigned int*)varPtr == 34);
    // Increment the varptr the size of an int.
    varPtr += 4;
   
    // Validate the sequence.
    // CC = 0x22
    assert(*varPtr == 0x22);
    varPtr++;
    // GA = 0x41
    assert(*varPtr == 0x41);
    varPtr++;
    // A  = 0x10
    assert(*varPtr == 0x10);
    varPtr++;
  
    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == samRecord.getQuality()[i] - 33);
        varPtr++;
    }

    // Validate the tags.  
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'D');
    varPtr++;
    assert(*varPtr == 'Z');
    varPtr++;
    assert(*varPtr == '3');
    varPtr++;
    assert(*varPtr == '7');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
    assert(*varPtr == 'N');
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
    assert(*varPtr == 'X');
    varPtr++;
    assert(*varPtr == 'T');
    varPtr++;
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == 'R');
    varPtr++;
}


void validateRead2(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 2
    // Create record structure for validating.
    int expectedBlockSize = 61;
    const char* expectedReferenceName = "1";
    const char* expectedMateReferenceName = "1";
    const char* expectedMateReferenceNameOrEqual = "=";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 0;
    expectedRecordPtr->myPosition = 1011;
    expectedRecordPtr->myReadNameLength = 23;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 0;
    expectedRecordPtr->myFlag = 133;
    expectedRecordPtr->myReadLength = 4;
    expectedRecordPtr->myMateReferenceID = 0;
    expectedRecordPtr->myMatePosition = 1010;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the fields.
    bamRecordStruct retrieveRecord;
    String retrieveReadName;
    String retrieveCigar;
    String retrieveSequence;
    String retrieveQuality;

    assert(samRecord.getFields(retrieveRecord, retrieveReadName, 
                               retrieveCigar, retrieveSequence,
                               retrieveQuality) == true);
    assert(retrieveRecord.myBlockSize == expectedRecordPtr->myBlockSize);
    assert(retrieveRecord.myReferenceID == expectedRecordPtr->myReferenceID);
    assert(retrieveRecord.myPosition == expectedRecordPtr->myPosition);
    assert(retrieveRecord.myReadNameLength == 
           expectedRecordPtr->myReadNameLength);
    assert(retrieveRecord.myMapQuality == expectedRecordPtr->myMapQuality);
    assert(retrieveRecord.myBin == expectedRecordPtr->myBin);
    assert(retrieveRecord.myCigarLength == expectedRecordPtr->myCigarLength);
    assert(retrieveRecord.myFlag == expectedRecordPtr->myFlag);
    assert(retrieveRecord.myReadLength == expectedRecordPtr->myReadLength);
    assert(retrieveRecord.myMateReferenceID == 
           expectedRecordPtr->myMateReferenceID);
    assert(retrieveRecord.myMatePosition == expectedRecordPtr->myMatePosition);
    assert(retrieveRecord.myInsertSize == expectedRecordPtr->myInsertSize);

    // Check the alignment end
    assert(samRecord.getAlignmentLength() == 0);
    assert(samRecord.get0BasedAlignmentEnd() == 1011);
    assert(samRecord.get1BasedAlignmentEnd() == 1012);
    assert(samRecord.get0BasedUnclippedStart() == 1011);
    assert(samRecord.get1BasedUnclippedStart() == 1012);
    assert(samRecord.get0BasedUnclippedEnd() == 1011);
    assert(samRecord.get1BasedUnclippedEnd() == 1012);

    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() == 
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() == 
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(), 
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(), 
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() ==
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() == 
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "1:1011:F:255+17M15D20M") == 0);
    assert(strcmp(samRecord.getCigar(), "*") == 0);
    assert(strcmp(samRecord.getSequence(), "CTGT") == 0);
    assert(strcmp(samRecord.getQuality(), ">>9>") == 0);

    // No Tags to check, should return false.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID == 
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // No cigar to validate. 
    // Validate the sequence.
    // CT = 0x28
    assert(*varPtr == 0x28);
    varPtr++;
    // GT = 0x48
    assert(*varPtr == 0x48);
    varPtr++;

    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == samRecord.getQuality()[i] - 33);
        varPtr++;
    }

    // No tags.  
}


void validateRead3(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 3
    // Create record structure for validating.
    int expectedBlockSize = 87;
    const char* expectedReferenceName = "1";
    const char* expectedMateReferenceName = "18";
    const char* expectedMateReferenceNameOrEqual = "18";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 0;
    expectedRecordPtr->myPosition = 74;
    expectedRecordPtr->myReadNameLength = 21;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 1;
    expectedRecordPtr->myFlag = 97;
    expectedRecordPtr->myReadLength = 5;
    expectedRecordPtr->myMateReferenceID = 17;
    expectedRecordPtr->myMatePosition = 756;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() == 
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() == 
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(), 
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() == 
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() ==
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "18:462+29M5I3M:F:295") == 0);
    assert(strcmp(samRecord.getCigar(), "5M") == 0);
    assert(strcmp(samRecord.getSequence(), "ACGTN") == 0);
    assert(strcmp(samRecord.getQuality(), ";>>>>") == 0);

    // Check the alignment end
    assert(samRecord.get0BasedAlignmentEnd() == 78);
    assert(samRecord.get1BasedAlignmentEnd() == 79);
    assert(samRecord.getAlignmentLength() == 5);
    assert(samRecord.get0BasedUnclippedStart() == 74);
    assert(samRecord.get1BasedUnclippedStart() == 75);
    assert(samRecord.get0BasedUnclippedEnd() == 78);
    assert(samRecord.get1BasedUnclippedEnd() == 79);


    // Check the tags.
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'A');
    assert(tag[1] == 'M');
    assert(type == 'i');
    assert(*(char*)value == 0);
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'M');
    assert(tag[1] == 'D');
    assert(type == 'Z');
    assert(*(String*)value == "30A0C5");
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'N');
    assert(tag[1] == 'M');
    assert(type == 'i');
    assert(*(char*)value == 2);
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'X');
    assert(tag[1] == 'T');
    assert(type == 'A');
    assert(*(char*)value == 'R');
    // No more tags, should return false.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID == 
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // Validate the cigar.
    // The cigar is 5M which is 5 << 4 | 0 = 80
    assert(*(unsigned int*)varPtr == 80);
    // Increment the varptr the size of an int.
    varPtr += 4;
   
    // Validate the sequence.
    // AC = 0x12
    assert(*varPtr == 0x12);
    varPtr++;
    // GT = 0x48
    assert(*varPtr == 0x48);
    varPtr++;
    // N  = 0xF0
    assert(*varPtr == 0xF0);
    varPtr++;

    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == samRecord.getQuality()[i] - 33);
        varPtr++;
    }

    // Validate the tags.  
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'D');
    varPtr++;
    assert(*varPtr == 'Z');
    varPtr++;
    assert(*varPtr == '3');
    varPtr++;
    assert(*varPtr == '0');
    varPtr++;
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == '0');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == '5');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
    assert(*varPtr == 'N');
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == 2);
    varPtr++;
    assert(*varPtr == 'X');
    varPtr++;
    assert(*varPtr == 'T');
    varPtr++;
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == 'R');
    varPtr++;
}


void validateRead4(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 4
    // Create record structure for validating.
    int expectedBlockSize = 57;
    const char* expectedReferenceName = "1";
    const char* expectedMateReferenceName = "18";
    const char* expectedMateReferenceNameOrEqual = "18";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 0;
    expectedRecordPtr->myPosition = 74;
    expectedRecordPtr->myReadNameLength = 21;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 0;
    expectedRecordPtr->myFlag = 97;
    expectedRecordPtr->myReadLength = 0;
    expectedRecordPtr->myMateReferenceID = 17;
    expectedRecordPtr->myMatePosition = 756;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the alignment end
    assert(samRecord.get1BasedUnclippedEnd() == 75);
    assert(samRecord.get0BasedUnclippedEnd() == 74);
    assert(samRecord.get0BasedUnclippedStart() == 74);
    assert(samRecord.get1BasedUnclippedStart() == 75);
    assert(samRecord.get1BasedAlignmentEnd() == 75);
    assert(samRecord.get0BasedAlignmentEnd() == 74);
    assert(samRecord.getAlignmentLength() == 0);

    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() ==
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() ==
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(),
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() ==
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() ==
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "18:462+29M5I3M:F:295") == 0);
    assert(strcmp(samRecord.getCigar(), "*") == 0);
    assert(strcmp(samRecord.getSequence(), "*") == 0);
    assert(strcmp(samRecord.getQuality(), "*") == 0);

    // Check the tag.
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'A');
    assert(tag[1] == 'M');
    assert(type == 'i');
    assert(*(char*)value == 0);
    // No more Tags to check, should return false.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID ==
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // No cigar to validate. 
    // Validate the sequence.
    // No sequence.
    // No Quality.

    // Validate the tags.  
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
}


void validateRead5(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 5
    int expectedBlockSize = 87;
    const char* expectedReferenceName = "2";
    const char* expectedMateReferenceName = "18";
    const char* expectedMateReferenceNameOrEqual = "18";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 1;
    expectedRecordPtr->myPosition = 74;
    expectedRecordPtr->myReadNameLength = 21;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 1;
    expectedRecordPtr->myFlag = 97;
    expectedRecordPtr->myReadLength = 5;
    expectedRecordPtr->myMateReferenceID = 17;
    expectedRecordPtr->myMatePosition = 756;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() ==
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() ==
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(),
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() == 
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() ==
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "18:462+29M5I3M:F:295") == 0);
    assert(strcmp(samRecord.getCigar(), "5M") == 0);
    assert(strcmp(samRecord.getSequence(), "ACGTN") == 0);
    assert(strcmp(samRecord.getQuality(), "*") == 0);

    // Check the tags.
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'A');
    assert(tag[1] == 'M');
    assert(type == 'i');
    assert(*(char*)value == 0);
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'M');
    assert(tag[1] == 'D');
    assert(type == 'Z');
    assert(*(String*)value == "30A0C5");
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'N');
    assert(tag[1] == 'M');
    assert(type == 'i');
    assert(*(char*)value == 2);
    assert(samRecord.getNextSamTag(tag, type, &value) == true);
    assert(tag[0] == 'X');
    assert(tag[1] == 'T');
    assert(type == 'A');
    assert(*(char*)value == 'R');
    // No more tags, should return false.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID ==
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // Validate the cigar.
    // The cigar is 5M which is 5 << 4 | 0 = 80
    assert(*(unsigned int*)varPtr == 80);
    // Increment the varptr the size of an int.
    varPtr += 4;
   
    // Validate the sequence.
    // AC = 0x12
    assert(*varPtr == 0x12);
    varPtr++;
    // GT = 0x48
    assert(*varPtr == 0x48);
    varPtr++;
    // N  = 0xF0
    assert(*varPtr == 0xF0);
    varPtr++;

    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == 0xFF);
        varPtr++;
    }

    // Validate the tags.  
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'D');
    varPtr++;
    assert(*varPtr == 'Z');
    varPtr++;
    assert(*varPtr == '3');
    varPtr++;
    assert(*varPtr == '0');
    varPtr++;
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == '0');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == '5');
    varPtr++;
    assert(*varPtr == 0);
    varPtr++;
    assert(*varPtr == 'N');
    varPtr++;
    assert(*varPtr == 'M');
    varPtr++;
    assert(*varPtr == 'C');
    varPtr++;
    assert(*varPtr == 2);
    varPtr++;
    assert(*varPtr == 'X');
    varPtr++;
    assert(*varPtr == 'T');
    varPtr++;
    assert(*varPtr == 'A');
    varPtr++;
    assert(*varPtr == 'R');
    varPtr++;
}


void validateRead6(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 6
    // Create record structure for validating.
    int expectedBlockSize = 77;
    const char* expectedReferenceName = "1";
    const char* expectedMateReferenceName = "18";
    const char* expectedMateReferenceNameOrEqual = "18";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 0;
    expectedRecordPtr->myPosition = 1750;
    expectedRecordPtr->myReadNameLength = 21;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 3;
    expectedRecordPtr->myFlag = 97;
    expectedRecordPtr->myReadLength = 8;
    expectedRecordPtr->myMateReferenceID = 17;
    expectedRecordPtr->myMatePosition = 756;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() == 
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() ==
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(),
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() ==
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() ==
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "18:462+29M5I3M:F:296") == 0);
    assert(strcmp(samRecord.getCigar(), "3S2H5M") == 0);
    assert(strcmp(samRecord.getSequence(), "TGCACGTN") == 0);
    assert(strcmp(samRecord.getQuality(), "453;>>>>") == 0);

    // Check the alignment end
    assert(samRecord.get0BasedAlignmentEnd() == 1754);
    assert(samRecord.get1BasedAlignmentEnd() == 1755);
    assert(samRecord.getAlignmentLength() == 5);
    assert(samRecord.get0BasedUnclippedStart() == 1745);
    assert(samRecord.get1BasedUnclippedStart() == 1746);
    assert(samRecord.get0BasedUnclippedEnd() == 1754);
    assert(samRecord.get1BasedUnclippedEnd() == 1755);

    // No tags.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID ==
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // Validate the cigar.
    // The cigar is 3S2H5M which is:
    // 3S: 3 << 4 | 4 = 0x34
    assert(*(unsigned int*)varPtr == 0x34);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // 2H: 2 << 4 | 5 = 0x25
    assert(*(unsigned int*)varPtr == 0x25);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // 5M: 5 << 4 | 0 = 0x50
    assert(*(unsigned int*)varPtr == 0x50);
    // Increment the varptr the size of an int.
    varPtr += 4;
   
    // Validate the sequence.
    // TG = 0x84
    assert(*varPtr == 0x84);
    varPtr++;
    // CA = 0x21
    assert(*varPtr == 0x21);
    varPtr++;
    // CG = 0x24
    assert(*varPtr == 0x24);
    varPtr++;
    // TN = 0x8F
    assert(*varPtr == 0x8F);
    varPtr++;

    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == samRecord.getQuality()[i] - 33);
        varPtr++;
    }
}


void validateRead7(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 7
    // Create record structure for validating.
    int expectedBlockSize = 83;
    const char* expectedReferenceName = "2";
    const char* expectedMateReferenceName = "18";
    const char* expectedMateReferenceNameOrEqual = "18";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 1;
    expectedRecordPtr->myPosition = 1750;
    expectedRecordPtr->myReadNameLength = 21;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 4;
    expectedRecordPtr->myFlag = 97;
    expectedRecordPtr->myReadLength = 9;
    expectedRecordPtr->myMateReferenceID = 17;
    expectedRecordPtr->myMatePosition = 756;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() ==
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() ==
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(),
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() ==
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() ==
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "18:462+29M5I3M:F:297") == 0);
    assert(strcmp(samRecord.getCigar(), "3S5M1S3H") == 0);
    assert(strcmp(samRecord.getSequence(), "TGCACGTNG") == 0);
    assert(strcmp(samRecord.getQuality(), "453;>>>>5") == 0);

    // Check the alignment end
    assert(samRecord.get0BasedAlignmentEnd() == 1754);
    assert(samRecord.get1BasedAlignmentEnd() == 1755);
    assert(samRecord.getAlignmentLength() == 5);
    assert(samRecord.get0BasedUnclippedStart() == 1747);
    assert(samRecord.get1BasedUnclippedStart() == 1748);
    assert(samRecord.get0BasedUnclippedEnd() == 1758);
    assert(samRecord.get1BasedUnclippedEnd() == 1759);

    // No tags.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID ==
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // Validate the cigar.
    // The cigar is 3S5M1S3H which is:
    // 3S: 3 << 4 | 4 = 0x34
    assert(*(unsigned int*)varPtr == 0x34);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // 5M: 5 << 4 | 0 = 0x50
    assert(*(unsigned int*)varPtr == 0x50);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // 1S: 1 << 4 | 4 = 0x14
    assert(*(unsigned int*)varPtr == 0x14);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // 3H: 3 << 4 | 5 = 0x35
    assert(*(unsigned int*)varPtr == 0x35);
    // Increment the varptr the size of an int.
    varPtr += 4;
   
    // Validate the sequence.
    // TG = 0x84
    assert(*varPtr == 0x84);
    varPtr++;
    // CA = 0x21
    assert(*varPtr == 0x21);
    varPtr++;
    // CG = 0x24
    assert(*varPtr == 0x24);
    varPtr++;
    // TN = 0x8F
    assert(*varPtr == 0x8F);
    varPtr++;
    // G  = 0x40
    assert(*varPtr == 0x40);
    varPtr++;

    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == samRecord.getQuality()[i] - 33);
        varPtr++;
    }
}


void validateRead8(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 8
    // Create record structure for validating.
    int expectedBlockSize = 65;
    const char* expectedReferenceName = "*";
    const char* expectedMateReferenceName = "*";
    const char* expectedMateReferenceNameOrEqual = "*";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = -1;
    expectedRecordPtr->myPosition = -1;
    expectedRecordPtr->myReadNameLength = 27;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4680;
    expectedRecordPtr->myCigarLength = 0;
    expectedRecordPtr->myFlag = 141;
    expectedRecordPtr->myReadLength = 4;
    expectedRecordPtr->myMateReferenceID = -1;
    expectedRecordPtr->myMatePosition = -1;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the alignment end
    assert(samRecord.get0BasedAlignmentEnd() == -1);
    assert(samRecord.get1BasedAlignmentEnd() == 0);
    assert(samRecord.getAlignmentLength() == 0);
    assert(samRecord.get0BasedUnclippedStart() == -1);
    assert(samRecord.get1BasedUnclippedStart() == 0);
    assert(samRecord.get0BasedUnclippedEnd() == -1);
    assert(samRecord.get1BasedUnclippedEnd() == 0);

    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() ==
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() ==
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(),
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() == 
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() ==
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "Y:16597235+13M13I11M:F:181") == 0);
    assert(strcmp(samRecord.getCigar(), "*") == 0);
    assert(strcmp(samRecord.getSequence(), "AACT") == 0);
    assert(strcmp(samRecord.getQuality(), "==;;") == 0);

    // No Tags to check, should return false.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID ==
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // No cigar to validate. 
    // Validate the sequence.
    // AA = 0x11
    assert(*varPtr == 0x11);
    varPtr++;
    // CT = 0x28
    assert(*varPtr == 0x28);
    varPtr++;

    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == samRecord.getQuality()[i] - 33);
        varPtr++;
    }

    // No tags.  
}


void validateRead9(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 9
    // Create record structure for validating.
    int expectedBlockSize = 77;
    const char* expectedReferenceName = "3";
    const char* expectedMateReferenceName = "18";
    const char* expectedMateReferenceNameOrEqual = "18";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = 2;
    expectedRecordPtr->myPosition = 74;
    expectedRecordPtr->myReadNameLength = 21;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4681;
    expectedRecordPtr->myCigarLength = 3;
    expectedRecordPtr->myFlag = 97;
    expectedRecordPtr->myReadLength = 8;
    expectedRecordPtr->myMateReferenceID = 17;
    expectedRecordPtr->myMatePosition = 756;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getStatus() == SamStatus::SUCCESS);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() ==
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() ==
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(),
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() ==
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() ==
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "18:462+29M5I3M:F:298") == 0);
    assert(strcmp(samRecord.getCigar(), "3S5M4H") == 0);
    assert((strcmp(samRecord.getSequence(), "TGCACGTN") == 0) || 
           (strcmp(samRecord.getSequence(), "tgcacgtn") == 0));
    assert(strcmp(samRecord.getQuality(), "453;>>>>") == 0);

    // Check the alignment end
    assert(samRecord.get0BasedAlignmentEnd() == 78);
    assert(samRecord.get1BasedAlignmentEnd() == 79);
    assert(samRecord.getAlignmentLength() == 5);
    assert(samRecord.get0BasedUnclippedStart() == 71);
    assert(samRecord.get1BasedUnclippedStart() == 72);
    assert(samRecord.get0BasedUnclippedEnd() == 82);
    assert(samRecord.get1BasedUnclippedEnd() == 83);

    // No tags.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    assert(bufferPtr != NULL);
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID == 
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // Validate the cigar.
    // The cigar is 3S5M1S3H which is:
    // 3S: 3 << 4 | 4 = 0x34
    assert(*(unsigned int*)varPtr == 0x34);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // 5M: 5 << 4 | 0 = 0x50
    assert(*(unsigned int*)varPtr == 0x50);
    // Increment the varptr the size of an int.
    varPtr += 4;
    // 4H: 4 << 4 | 5 = 0x45
    assert(*(unsigned int*)varPtr == 0x45);
    // Increment the varptr the size of an int.
    varPtr += 4;
   
    // Validate the sequence.
    // TG = 0x84
    assert(*varPtr == 0x84);
    varPtr++;
    // CA = 0x21
    assert(*varPtr == 0x21);
    varPtr++;
    // CG = 0x24
    assert(*varPtr == 0x24);
    varPtr++;
    // TN = 0x8F
    assert(*varPtr == 0x8F);
    varPtr++;

    // Validate the Quality
    for(int i = 0; i < expectedRecordPtr->myReadLength; i++)
    {
        assert(*varPtr == samRecord.getQuality()[i] - 33);
        varPtr++;
    }
}


void validateRead10(SamRecord& samRecord)
{
    //////////////////////////////////////////
    // Validate Record 10
    // Create record structure for validating.
    int expectedBlockSize = 59;
    const char* expectedReferenceName = "*";
    const char* expectedMateReferenceName = "*";
    const char* expectedMateReferenceNameOrEqual = "*";

    bamRecordStruct* expectedRecordPtr =
        (bamRecordStruct *) malloc(expectedBlockSize + sizeof(int));

    char tag[3];
    char type;
    void* value;
    bamRecordStruct* bufferPtr;
    unsigned char* varPtr;

    expectedRecordPtr->myBlockSize = expectedBlockSize;
    expectedRecordPtr->myReferenceID = -1;
    expectedRecordPtr->myPosition = -1;
    expectedRecordPtr->myReadNameLength = 27;
    expectedRecordPtr->myMapQuality = 0;
    expectedRecordPtr->myBin = 4680;
    expectedRecordPtr->myCigarLength = 0;
    expectedRecordPtr->myFlag = 141;
    expectedRecordPtr->myReadLength = 0;
    expectedRecordPtr->myMateReferenceID = -1;
    expectedRecordPtr->myMatePosition = -1;
    expectedRecordPtr->myInsertSize = 0;
   
    // Check the alignment end
    assert(samRecord.get0BasedUnclippedStart() == -1);
    assert(samRecord.get1BasedUnclippedStart() == 0);
    assert(samRecord.get0BasedUnclippedEnd() == -1);
    assert(samRecord.get1BasedUnclippedEnd() == 0);
    assert(samRecord.get1BasedAlignmentEnd() == 0);
    assert(samRecord.get0BasedAlignmentEnd() == -1);
    assert(samRecord.getAlignmentLength() == 0);

    // Check the accessors.
    assert(samRecord.getBlockSize() == expectedRecordPtr->myBlockSize);
    assert(samRecord.getReferenceID() == expectedRecordPtr->myReferenceID);
    assert(strcmp(samRecord.getReferenceName(), expectedReferenceName) == 0);
    assert(samRecord.get1BasedPosition() == expectedRecordPtr->myPosition + 1);
    assert(samRecord.get0BasedPosition() == expectedRecordPtr->myPosition);
    assert(samRecord.getReadNameLength() == 
           expectedRecordPtr->myReadNameLength);
    assert(samRecord.getMapQuality() == expectedRecordPtr->myMapQuality);
    assert(samRecord.getBin() == expectedRecordPtr->myBin);
    assert(samRecord.getCigarLength() == expectedRecordPtr->myCigarLength);
    assert(samRecord.getFlag() == expectedRecordPtr->myFlag);
    assert(samRecord.getReadLength() == expectedRecordPtr->myReadLength);
    assert(samRecord.getMateReferenceID() == 
           expectedRecordPtr->myMateReferenceID);
    assert(strcmp(samRecord.getMateReferenceName(), 
                  expectedMateReferenceName) == 0);
    assert(strcmp(samRecord.getMateReferenceNameOrEqual(),
                  expectedMateReferenceNameOrEqual) == 0);
    assert(samRecord.get1BasedMatePosition() == 
           expectedRecordPtr->myMatePosition + 1);
    assert(samRecord.get0BasedMatePosition() == 
           expectedRecordPtr->myMatePosition);
    assert(samRecord.getInsertSize() == expectedRecordPtr->myInsertSize);
    assert(strcmp(samRecord.getReadName(), "Y:16597235+13M13I11M:F:181") == 0);
    assert(strcmp(samRecord.getCigar(), "*") == 0);
    assert(strcmp(samRecord.getSequence(), "*") == 0);
    assert(strcmp(samRecord.getQuality(), "*") == 0);

    // No Tags to check, should return false.
    assert(samRecord.getNextSamTag(tag, type, &value) == false);
    assert(samRecord.getNextSamTag(tag, type, &value) == false);

    // Get the record ptr.   
    bufferPtr = (bamRecordStruct*)samRecord.getRecordBuffer();
    // Validate the buffers match.
    assert(bufferPtr->myBlockSize == expectedRecordPtr->myBlockSize);
    assert(bufferPtr->myReferenceID == expectedRecordPtr->myReferenceID);
    assert(bufferPtr->myPosition == expectedRecordPtr->myPosition);
    assert(bufferPtr->myReadNameLength == expectedRecordPtr->myReadNameLength);
    assert(bufferPtr->myMapQuality == expectedRecordPtr->myMapQuality);
    assert(bufferPtr->myBin == expectedRecordPtr->myBin);
    assert(bufferPtr->myCigarLength == expectedRecordPtr->myCigarLength);
    assert(bufferPtr->myFlag == expectedRecordPtr->myFlag);
    assert(bufferPtr->myReadLength == expectedRecordPtr->myReadLength);
    assert(bufferPtr->myMateReferenceID == 
           expectedRecordPtr->myMateReferenceID);
    assert(bufferPtr->myMatePosition == expectedRecordPtr->myMatePosition);
    assert(bufferPtr->myInsertSize == expectedRecordPtr->myInsertSize);

    // Validate the variable length fields in the buffer.
    // Set the pointer to the start of the variable fields.
    varPtr = (unsigned char*)(&(bufferPtr->myData[0]));

    // Validate the readname.
    for(int i = 0; i < expectedRecordPtr->myReadNameLength; i++)
    {
        assert(*varPtr == samRecord.getReadName()[i]);
        varPtr++;
    }

    // No cigar to validate. 
    // No sequence.
    // No Quality.
    // No Tags.
}


void validateHeader(SamFileHeader& samHeader)
{
    validateHeaderFields(samHeader);
    validateHeaderString(samHeader);
}


void validateHeaderFields(SamFileHeader& samHeader)
{
    const char* value;

    ////////////////////////////////////////////////////////
    // Test getting a specific HD Tag value from the header
    // that does not exist.
    value = samHeader.getHDTagValue("GO");
    assert(strcmp(value, "") == 0);

    ////////////////////////////////////////////////////////
    // Test getting a specific PG Tag value from the header
    // that does not exist.
    value = samHeader.getPGTagValue("CL", "1");
    assert(strcmp(value, "") == 0);

    ////////////////////////////////////////////////////////
    // Test getting a specific SQ Tag value from the header
    value = samHeader.getSQTagValue("LN", "1");
    assert(value != NULL);
    assert(strcmp(value, "247249719") == 0);
    value = samHeader.getSQTagValue("LN", "22");
    assert(value != NULL);
    assert(strcmp(value, "49691432") == 0);

    ////////////////////////////////////////////////////////
    // Test getting a specific SQ Tag value from the header
    // that does not exist.
    value = samHeader.getSQTagValue("LN", "1000");
    assert(strcmp(value, "") == 0);

    ////////////////////////////////////////////////////////
    // Test getting a specific SQ Tag value from the header
    // that does not exist - sq exists, but not with that tag.
    value = samHeader.getSQTagValue("AS", "1");
    assert(strcmp(value, "") == 0);

    ////////////////////////////////////////////////////////
    // Test getting a specific RG Tag value from the header
    value = samHeader.getRGTagValue("LB", "myID2");
    assert(value != NULL);
    assert(strcmp(value, "library2") == 0);
    value = samHeader.getRGTagValue("LB", "myID");
    assert(value != NULL);
    assert(strcmp(value, "library") == 0);

    ////////////////////////////////////////////////////////
    // Test getting a specific SQ from the header
    // Then pulling the tags out of it.
    SamHeaderSQ* sq = samHeader.getSQ("10");
    assert(strcmp(sq->getTagValue("SN"), "10") == 0);
    assert(strcmp(sq->getTagValue("LN"), "135374737") == 0);
   
    // Test pulling a tag that does not exist.
    assert(strcmp(sq->getTagValue("DD"), "") == 0);
   

    ////////////////////////////////////////////////////////
    // Test getting a specific RG from the header
    // Then pulling the tags out of it.
    const SamHeaderRG* rg = samHeader.getRG("myID");
    assert(strcmp(rg->getTagValue("ID"), "myID") == 0);
    assert(strcmp(rg->getTagValue("SM"), "sample") == 0);
    assert(strcmp(rg->getTagValue("LB"), "library") == 0);
   
    // Test pulling a tag that does not exist.
    assert(strcmp(rg->getTagValue("DD"), "") == 0);
   
    ////////////////////////////////////////////////////////
    // Test getting a specific RG from the header that does not exist.
    rg = samHeader.getRG("noExist");
    assert(rg == NULL);

    ////////////////////////////////////////////////////////
    // Test getting a specific SQ from the header that does not exist.
    sq = samHeader.getSQ("noExist");
    assert(sq == NULL);

    ////////////////////////////////////////////////////////
    // Test getting the reference ID.
    assert(samHeader.GetReferenceID("2") == 1);
    std::string refIDStdString = "X";
    assert(samHeader.GetReferenceID(refIDStdString.c_str()) == 22);
    String refIDString = "22";
    assert(samHeader.GetReferenceID(refIDString) == 21);
    assert(samHeader.GetReferenceID(refIDString.c_str()) == 21);
    assert(samHeader.GetReferenceID("Z") == 23);
    assert(samHeader.GetReferenceID("*") == -1);
    refIDString = "*";
    assert(samHeader.GetReferenceID(refIDString) == -1);
    assert(samHeader.GetReferenceID(refIDString.c_str()) == -1);
}

void validateHeaderString(SamFileHeader& samHeader)
{
    // Check the header line.
    std::string headerString = "";
    assert(samHeader.getHeaderString(headerString) == true);
    assert(headerString == "@SQ\tSN:1\tLN:247249719\n@SQ\tSN:2\tLN:242951149\n@SQ\tSN:3\tLN:199501827\n@SQ\tSN:4\tLN:191273063\n@SQ\tSN:5\tLN:180857866\n@SQ\tSN:6\tLN:170899992\n@SQ\tSN:7\tLN:158821424\n@SQ\tSN:8\tLN:146274826\n@SQ\tSN:9\tLN:140273252\n@SQ\tSN:10\tLN:135374737\n@SQ\tSN:11\tLN:134452384\n@SQ\tSN:12\tLN:132349534\n@SQ\tSN:13\tLN:114142980\n@SQ\tSN:14\tLN:106368585\n@SQ\tSN:15\tLN:100338915\n@SQ\tSN:16\tLN:88827254\n@SQ\tSN:17\tLN:78774742\n@SQ\tSN:18\tLN:76117153\n@SQ\tSN:19\tLN:63811651\n@SQ\tSN:20\tLN:62435964\n@SQ\tSN:21\tLN:46944323\n@SQ\tSN:22\tLN:49691432\n@SQ\tSN:X\tLN:154913754\n@RG\tID:myID\tLB:library\tSM:sample\n@RG\tID:myID2\tSM:sample2\tLB:library2\n@CO\tComment 1\n@CO\tComment 2\n");
}
