/*
 *  Copyright (C) 2010  Regents of the University of Michigan
 *
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

//////////////////////////////////////////////////////////////////////////
// This file contains the processing for the executable option "writeRegion"
// which writes a file with the reads in the specified region.

#include "SamFile.h"
#include "Parameters.h"
#include "BgzfFileType.h"

void writeRegionDescription()
{
    std::cout << " writeRegion - Write a file with reads in the specified region" << std::endl;
    std::cout << "\t./bam writeRegion --in <inputFilename>  --out <outputFilename> [--bamIndex <bamIndexFile>] [--noeof] [--refName <reference Name> | --refID <reference ID>] [--start <0-based start pos>] [--end <0-based end psoition>]"<< std::endl;
}

void writeRegionUsage()
{
    writeRegionDescription();
    std::cout << "\tRequired Parameters:" << std::endl;
    std::cout << "\t\t--in       : the BAM file to be read" << std::endl;
    std::cout << "\t\t--out      : the SAM/BAM file to write to" << std::endl;
    std::cout << "\tOptional Parameters:" << std::endl;
    std::cout << "\t\t--noeof  : do not expect an EOF block on a bam file." << std::endl;
    std::cout << "\t\t--bamIndex : the path/name of the bam index file" << std::endl;
    std::cout << "\t\t             (if not specified, uses the --in value + \".bai\")" << std::endl;
    std::cout << "\t\t--refName  : the BAM reference Name to read (either this or refID can be specified)" << std::endl;
    std::cout << "\t\t--refID    : the BAM reference ID to read (defaults to -1: unmapped)" << std::endl;
    std::cout << "\t\t--start    : inclusive 0-based start position (defaults to -1)" << std::endl;
    std::cout << "\t\t--end      : exclusive 0-based end position (defaults to -1: meaning til the end of the reference)" << std::endl;
    std::cout << std::endl;
}


int writeRegion(int argc, char **argv)
{
    // Extract command line arguments.
    static const int UNSPECIFIED_INT = -1;
    String inFile = "";
    String outFile = "";
    String indexFile = "";
    String refName = "";
    int refID = UNSPECIFIED_INT;
    int start = UNSPECIFIED_INT;
    int end = UNSPECIFIED_INT;
    bool noeof = false;

    ParameterList inputParameters;
    BEGIN_LONG_PARAMETERS(longParameterList)
        LONG_STRINGPARAMETER("in", &inFile)
        LONG_STRINGPARAMETER("out", &outFile)
        LONG_STRINGPARAMETER("bamIndex", &indexFile)
        LONG_STRINGPARAMETER("refName", &refName)
        LONG_INTPARAMETER("refID", &refID)
        LONG_INTPARAMETER("start", &start)
        LONG_INTPARAMETER("end", &end)
        LONG_PARAMETER("noeof", &noeof)
        END_LONG_PARAMETERS();
   
    inputParameters.Add(new LongParameters ("Input Parameters", 
                                            longParameterList));

    inputParameters.Read(argc-1, &(argv[1]));

    inputParameters.Status();
   
    // If no eof block is required for a bgzf file, set the bgzf file type to 
    // not look for it.
    if(noeof)
    {
        // Set that the eof block is not required.
        BgzfFileType::setRequireEofBlock(false);
    }

    // Check to see if the in file was specified, if not, report an error.
    if(inFile == "")
    {
        writeRegionUsage();
        // mandatory argument was not specified.
        std::cerr << "Missing mandatory argument: --in" << std::endl;
        return(-1);
    }
    if(outFile == "")
    {
        writeRegionUsage();
        // mandatory argument was not specified.
        std::cerr << "Missing mandatory argument: --out" << std::endl;
        return(-1);
    }
    
    if(indexFile == "")
    {
        // In file was not specified, so set it to the in file
        // + ".bai"
        indexFile = inFile + ".bai";
    }

    if(refID != -1 && refName.Length() != 0)
    {
        std::cerr << "Can't specify both refID and refName" << std::endl;
        return(-1);
    }

    SamFile samIn;
    // Open the file for reading.   
    if(!samIn.OpenForRead(inFile))
    {
        fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        return(samIn.GetStatus());
    }

    // If refName is set, use that.
    if(refName.Length() != 0)
    {
        // Use Reference Name.
        if(!samIn.SetReadSection(refName.c_str(), start, end))
        {
            fprintf(stderr, "%s\n", samIn.GetStatusMessage());
            return(samIn.GetStatus());
        }
    }
    else
    {
        // Use Reference ID
        if(!samIn.SetReadSection(refID, start, end))
        {
            fprintf(stderr, "%s\n", samIn.GetStatusMessage());
            return(samIn.GetStatus());
         }
    }

    // Open the output file for writing.
    SamFile samOut;
    if(!samOut.OpenForWrite(outFile))
    {
        fprintf(stderr, "%s\n", samOut.GetStatusMessage());
        return(samOut.GetStatus());
    }

    // Open the bam index file for reading.
    if(!samIn.ReadBamIndex(indexFile))
    {
        fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        return(samIn.GetStatus());
    }

    // Read the sam header.
    SamFileHeader samHeader;
    if(!samIn.ReadHeader(samHeader))
    {
        fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        return(samIn.GetStatus());
    }
    
    // Write the sam header.
    if(!samOut.WriteHeader(samHeader))
    {
        fprintf(stderr, "%s\n", samOut.GetStatusMessage());
        return(samOut.GetStatus());     
    }

    // Read the sam records.
    SamRecord samRecord;
    // Track the status.
    int numSectionRecords = 0;

    // Set returnStatus to success.  It will be changed
    // to the failure reason if any of the writes fail.
    SamStatus::Status returnStatus = SamStatus::SUCCESS;

    // Keep reading records until they aren't anymore.
    while(samIn.ReadRecord(samHeader, samRecord))
    {
        // Successfully read a record from the file, so write it.
        if(!samOut.WriteRecord(samHeader, samRecord))
        {
            // Failed to write a record.
            fprintf(stderr, "%s\n", samOut.GetStatusMessage());
            returnStatus = samOut.GetStatus();
        }
        ++numSectionRecords;
    }

    if(samIn.GetStatus() != SamStatus::NO_MORE_RECS)
    {
        // Failed to read a record.
        fprintf(stderr, "%s\n", samIn.GetStatusMessage());
        // Set the return status to the reason why
        // the read failed.
        returnStatus = samIn.GetStatus();
    }

    std::cout << "Wrote " << outFile << " with " << numSectionRecords << " records.\n";
    return(returnStatus);
}
