#ifndef _TRIMSEQUENCE_H
#define _TRIMSEQUENCE_H

#include <stdint.h>

//
// trimSequence is a templated function to find bases
// which are below a certain moving mean threshold,
// and can be applied to either end of the sequence string.
//
// meanValue is the value below which we wish to trim.
// 
// trimFromLeft is a bool indicating which direction we wish
// to trim.  true -> left to right, false is right to left.
//
// When trimFromLeft is true:
//    result == sequence.begin() implies no trimming
//    result == sequence.end() implies all values are trimmed
//
// When trimFromLeft is false:
//    result == sequence.begin() implies all values are trimmed
//    result == sequence.end() no values are trimmed
//
// result will always be in the range [sequence.begin() , sequence.end())
// (begin is inclusive, end is exclusive).
//

template<typename sequenceType, typename meanValueType>
typename sequenceType::iterator trimSequence(sequenceType &sequence, meanValueType meanValue, bool trimFromLeft)
{
    const int howManyValues = 4;
    typename sequenceType::iterator it;
    int64_t sumOf4Values = 0;
    int  counter = 0;

    typename sequenceType::iterator sequenceBegin = (trimFromLeft ? sequence.begin() : sequence.end() - 1);
    typename sequenceType::iterator sequenceEnd = (trimFromLeft ? sequence.end() : sequence.begin() - 1);
    int sequenceDirection = (trimFromLeft ? +1 : -1);

    if(sequence.size() == 0) return sequence.end();

    for(it=sequenceBegin; it!=sequenceEnd && counter < howManyValues; counter++, it += sequenceDirection) {
        sumOf4Values += *it;
    }
    
    //
    // continue on until mean last 4 values is >= meanValue
    //
    for( ; it!=sequenceEnd; it += sequenceDirection) {
        if( ((double) sumOf4Values / howManyValues) >= meanValue) break;
        sumOf4Values -= *(it - howManyValues * sequenceDirection);
        sumOf4Values += *it;
    }

    // we need to backtrack slightly to find the last value
    // which is not less than the desired meanValue
    for( ; it != (sequenceBegin - sequenceDirection) ; it -= sequenceDirection) {
        if(*it < meanValue) {
            if(trimFromLeft) it += sequenceDirection;
            break;
        }
    }

    // when we are backtracking on the left end, we get underrun,
    // and need to adjust the it to the begin of the sequence.
    //
    // This is not a problem on backtracking on the right end, because
    // it iterator is already correctly one after the last good base,
    // which is what we want to return.
    //
    if(trimFromLeft && it == (sequenceBegin - sequenceDirection)) it = sequenceBegin;

    return it;
}

#endif
