// FamilyLikelihood.cpp
// 3/13/2006 Wei-Min Chen

#include "FamilyLikelihood.h"
#include "Error.h"
#include "MapFunction.h"

#include <math.h>

// ********************** FamilyLikelihood ***************************

FamilyLikelihood::FamilyLikelihood()
{
   phase = NULL;
   family = NULL;
   esPerson = NULL;
   genotypeArray = NULL;
}

FamilyLikelihood::~FamilyLikelihood()
{
   if(esPerson) delete []esPerson;
   if(phase) delete []phase;
   if(genotypeArray) delete []genotypeArray;
}

void FamilyLikelihood::SelectFamily(Family * ptr)
{
   family = ptr;

   if(es.Items) delete []es.Items;
   if(es.States) delete []es.States;
   es.Items = new Vector[es.Index.Length()];
   es.States = new IntArray[es.Index.Length()];

   for(int i = 0; i < family->count; i++)
      ListStates(family->ped[family->path[i]], es.States[i]);
   for(int i = family->count; i < es.Index.Length(); i++){
      int mo = es.Index[i] / family->count;
      int fa = es.Index[i] % family->count;
      es.States[i].Dimension(es.States[mo].Length() * es.States[fa].Length());
      es.States[i].Set(-1);
   }
   for(int i = 0; i < es.Index.Length(); i++)
      es.Items[i].Dimension(es.States[i].Length());
   ResetES();
}

void FamilyLikelihoodHAP::ConvertNumber(long int number, IntArray & out, int bit)
{
   out.Dimension(bit);
   out.Zero();
   if( (number < 0) || (number > BinaryBase[30]) ) {
      printf("Failed to convert number %ld\n", number);
      return;
   }
   for(int i = 0; i < bit; i++)
      if(number & BinaryBase[i]) out[i] = 1;
}

void FamilyLikelihood::ResetES()
{
   for(int i = 0; i < es.Index.Length(); i++)
      es.Items[i].Set(1.0);
   for(int i = 0; i < family->count; i++)
      if(family->ped[family->path[i]].isFounder())
         for(int j = 0; j < es.States[i].Length(); j++)
            es.Items[i][j] = ProbabilityOfState(es.States[i][j]);
}

double FamilyLikelihood::Likelihood()
{
   ResetES();
   for(int i = 0; i < family->count; i++)
      for(int j = 0; j < es.Items[i].Length(); j++)
         es.Items[i][j] *= ProbabilityOfPhenotype(family->ped[family->path[i]], es.States[i][j]);
   return LikelihoodOfGenotype();
}

double FamilyLikelihood::score()
{
   ResetES();
   double lik = _NAN_;
   AlleleCount.Dimension(family->count);
   for(int i = 0; i < family->count; i++)
      if(family->ped[family->path[i]].markers[order[markerOrder]].isKnown())
         AlleleCount[i] = GenotypeMeasure(es.States[i][0]);
      else{
         if(!family->ped[family->path[i]].isFullyPhenotyped()) {
            AlleleCount[i] = 0;
            continue;
         }
         if(lik==_NAN_) lik = LikelihoodOfGenotype();
         ResetES();
         for(int j = 0; j < es.Items[i].Length(); j++)
            es.Items[i][j] *= GenotypeMeasure(es.States[i][j]);
         AlleleCount[i] = LikelihoodOfGenotype() / lik;
      }
   return lik;
}

void FamilyLikelihood::GetCovariance()
{
   ResetES();
   double lik = LikelihoodOfGenotype();
   AlleleCovariance.Dimension(family->count, family->count);
   AlleleCovariance.Zero();
   for(int i = 0; i < family->count; i++){
      if(family->ped[family->path[i]].markers[order[markerOrder]].isKnown())
         continue;
      ResetES();
      for(int k = 0; k < es.Items[i].Length(); k++){
         double temp = GenotypeMeasure(es.States[i][k]);
         es.Items[i][k] *= temp * temp;
      }
      AlleleCovariance[i][i] = LikelihoodOfGenotype() / lik - AlleleCount[i] * AlleleCount[i];
      for(int j = 0; j < i; j++){
         if(family->ped[family->path[j]].markers[order[markerOrder]].isKnown())
            continue;
         ResetES();
         for(int k = 0; k < es.Items[i].Length(); k++)
            es.Items[i][k] *= GenotypeMeasure(es.States[i][k]);
         for(int k = 0; k < es.Items[j].Length(); k++)
            es.Items[j][k] *= GenotypeMeasure(es.States[j][k]);
         AlleleCovariance[i][j] = AlleleCovariance[j][i] =
            LikelihoodOfGenotype() / lik - AlleleCount[i] * AlleleCount[j];
      }
   }
}

void FamilyLikelihood::GetVariance()
{
   // SavedLikelihood have been assigned the value;
   AlleleVariance.Dimension(family->count);
   AlleleVariance.Zero();
   for(int i = 0; i < 3; i++) AlleleDistribution[i].Dimension(family->count);
   if(SavedLikelihood == _NAN_){
      ResetES();
      SavedLikelihood = LikelihoodOfGenotype();
   }
   for(int i = 0; i < family->count; i++){
      if(family->ped[family->path[i]].markers[order[markerOrder]].isKnown())
         continue;
      ResetES();
      for(int k = 0; k < es.Items[i].Length(); k++){
         double temp = GenotypeMeasure(es.States[i][k]);
         es.Items[i][k] *= (temp * temp);
      }
      AlleleVariance[i] =  LikelihoodOfGenotype() / SavedLikelihood - AlleleCount[i] * AlleleCount[i];
   }
   for(int i = 0; i < family->count; i++){
      AlleleDistribution[1][i] = 2 * AlleleCount[i] - AlleleCount[i]*AlleleCount[i] - AlleleVariance[i];
      AlleleDistribution[2][i] = 0.5 * (-AlleleCount[i] + AlleleCount[i]*AlleleCount[i] + AlleleVariance[i]);
      AlleleDistribution[0][i] = 1 - AlleleDistribution[1][i] - AlleleDistribution[2][i];
   }
}

double FamilyLikelihood::nuclearFamilyLikelihood()
{
   double likelihood = 0.0;
   for(int u = 0; u < es.States[0].Length(); u++){
      double mterm = es.Items[0][u];
      if (mterm == 0.0) continue;
      for(int v = 0; v < es.States[1].Length(); v++){
         double pterm = es.Items[1][v];
         if(pterm == 0.0) continue;
         for(int w = 0; w < family->nonFounders; w++){
            double term = 0.0;
            for(int s = 0; s < es.States[w+2].Length(); s++)
               term += ProbabilityOfTransmission(es.States[w+2][s],
                  es.States[0][u], es.States[1][v]) * es.Items[w+2][s];
            pterm *= term;
            if(pterm == 0.0) break;
         }
         likelihood += mterm * pterm;
      }
   }
   return likelihood;
}

double FamilyLikelihood::LikelihoodOfGenotype()
{
   double likelihood = 0.0;
   double term;
   int m, s;

   for(int i = 0; i < es.Type.Length(); i++){
      if(es.Type[i] == 'N'){ // Nuclear family
         likelihood = nuclearFamilyLikelihood();
      }else if(es.Type[i] == 'E') // Ending
         for(int j = 0; j < es.States[es.From[i]].Length(); j++)
            likelihood += es.Items[es.From[i]][j];
      else if(es.Type[i] == 'T');
      else{
         int MarriageNode = es.From[i] < family->count ? es.To[i]: es.From[i];
         int OtherNode = es.From[i] < family->count ? es.From[i]: es.To[i];
         int mo = es.Index[MarriageNode] / family->count;
         int fa = es.Index[MarriageNode] % family->count;
         int faLength = es.States[fa].Length();
         if(es.Type[i] == 'O' && es.From[i] < family->count){  // Offspring -> Marriage
            for(m = 0; m < es.States[MarriageNode].Length(); m++){
               for(term = 0.0, s = 0; s < es.States[OtherNode].Length(); s++)
                  term += ProbabilityOfTransmission(es.States[OtherNode][s],
                     es.States[mo][m / faLength], es.States[fa][m % faLength]) * es.Items[OtherNode][s];
               es.Items[MarriageNode][m] *= term;
            }
         }else if(es.Type[i] == 'O'){ // Marriage -> Offspring
            for(int s = 0; s < es.States[OtherNode].Length(); s++){
               for(term = 0.0, m = 0; m < es.States[MarriageNode].Length(); m++)
                  term += ProbabilityOfTransmission(es.States[OtherNode][s],
                     es.States[mo][m / faLength], es.States[fa][m % faLength]) * es.Items[MarriageNode][m];
               es.Items[OtherNode][s] *= term;
            }
         }else if(es.From[i] < family->count){  // Spouse -> Marriage
            for(m = 0; m < es.States[MarriageNode].Length(); m++){
               if(family->ped[family->path[OtherNode]].sex == 1)
                  es.Items[MarriageNode][m] *= es.Items[fa][m % faLength];
               else
                  es.Items[MarriageNode][m] *= es.Items[mo][m / faLength];
            }
         }else{   // Marriage -> Spouse
            for(s = 0; s < es.States[OtherNode].Length(); s++){
               if(family->ped[family->path[OtherNode]].sex == 1)// s is father
                  for(term = 0.0, m = 0; m < es.States[mo].Length(); m++)
                     term += es.Items[MarriageNode][m * faLength + s];
               else  // summation over father
                  for(term = 0.0, m = 0; m < es.States[fa].Length(); m++)
                     term += es.Items[MarriageNode][s * faLength + m];
               es.Items[OtherNode][s] *= term;
            }
         }
      }
   }
   return likelihood;
}

// ********************* FamilyLikelihoodNULL *********************** //

FamilyLikelihoodNull::FamilyLikelihoodNull()
{
   frequencies.Dimension(1);
   frequencies[0] = 0.5;
   trait = 0;
   order.Dimension(1);
   order[0] = 0;
   markerOrder = 0;
   AllSNP = 1;

   IntToVector = new IntArray[16];
   for(int i = 0; i < 16; i++){
      IntToVector[i].Dimension(0);
      if(i&1) IntToVector[i].Push(0);
      if((i&2) || (i&4)) IntToVector[i].Push(1);
      if(i&8) IntToVector[i].Push(2);
   }
   static double transmission_matrix[] =
      {
      /* 0 x 0, AA x AA */   1.0, 0.0, 0.0,
      /* 0 x 1, AA x Aa */   0.5, 0.5, 0.0,
      /* 0 x 2, AA x aa */   0.0, 1.0, 0.0,
      /* 1 x 0, Aa x AA */   0.5, 0.5, 0.0,
      /* 1 x 1, Aa x Aa */   0.25, 0.5, 0.25,
      /* 1 x 2, Aa x aa */   0.0, 0.5, 0.5,
      /* 2 x 0, aa x AA */   0.0, 1.0, 0.0,
      /* 2 x 1, aa x Aa */   0.0, 0.5, 0.5,
      /* 2 x 2, aa x aa */   0.0, 0.0, 1.0,
      };

  stateTransmission.Dimension(27);
  for(int i = 0; i < 27; i++)
      stateTransmission[i] = transmission_matrix[i];
}

void FamilyLikelihoodNull::SelectLocus()
{
   stateCount = 3;
   stateProb.Dimension(stateCount);
   UpdateFrequency();
}

// ******************** SNP with Flanking Markers ***************

void ConvertNumber(long int number, IntArray & out, int bit);
int ConvertBinary(IntArray &number){
   int r = 0;
   for(int i = 0; i < number.Length(); i++)
      if(number[i])  r |= (1<<i);
   return r;
}

FamilyLikelihoodHAP::FamilyLikelihoodHAP()
{
   IntToArray = new IntArray[16];
   for(int i = 0; i < 16; i++){
      IntToArray[i].Dimension(0);
      for(int j = 0; j < 4; j++)
         if(i & (1<<j))
            IntToArray[i].Push(j);
   }
   recombinations.Dimension(1);
   recombinations[0] = 0.5;
   BinaryBase.Dimension(31);
   for(int i = 0; i < 31; i++)
      BinaryBase[i] = (1 << i);

   MaxDimension = 14;
   binary = new IntArray[1<<MaxDimension];
   for(int i = 0; i < BinaryBase[MaxDimension]; i++)
      ConvertNumber(i, binary[i], MaxDimension);

   binaryDouble.Dimension(BinaryBase[13]);
   for(int i = 0; i<binaryDouble.Length(); i++){
      int temp = 0;
      for(int j = 12; j >= 0; j--){
         if(i & BinaryBase[j]) {
            temp |= BinaryBase[j*2];
            temp |= BinaryBase[j*2+1];
         }
      }
      binaryDouble[i] = temp;
   }
   AllOne.Dimension(30);
   for(int i = 0; i < AllOne.Length(); i++)
      AllOne[i] = BinaryBase[i]-1;
}

FamilyLikelihoodHAP::~FamilyLikelihoodHAP()
{
   if(binary) delete []binary;
   if(IntToArray) delete []IntToArray;
}

int FamilyLikelihoodHAP::GenotypeMeasure(int state)
{
   if(AllSNP)
      return ((state & BinaryBase[markerOrder*2]) != 0) +
         ((state & BinaryBase[markerOrder*2+1]) != 0);
   else
      return Locus[haploid[0][state] * stateHapLength + markerOrder] +
         Locus[haploid[1][state] * stateHapLength + markerOrder];
}

void FamilyLikelihoodHAP::UpdateFrequency()
{
   stateProb.Set(1.0);
   if(AllSNP)
      for(int i = 0; i < stateHapCount; i++)
         for(int j = 0; j < orderCount; j++)
            stateProb[i] *= ((binary[i][j] == 0)? frequencies[j] : 1-frequencies[j]);
   else
      for(int i = 0; i < stateHapCount; i++)
         for(int j = 0; j < orderCount; j++)
            stateProb[i] *= family->ped.GetMarkerInfo(order[j]) ->
               freq[Locus[i*stateHapLength+j]+1];
}

void FamilyLikelihoodHAP::SelectLocus()
{
   orderCount = order.Length();
   if(AllSNP) stateLength = orderCount * 2; //  else{}
   stateHapLength = stateLength / 2;

   stateHapCount = 1 << stateHapLength;
   stateCount = 1 << stateLength;
   stateProb.Dimension(stateHapCount);
   stateProb.Set(1.0);

   Locus.Dimension(stateHapCount*stateHapLength);
   if(AllSNP)
      for(int i = 0; i < stateHapCount; i++)
         for(int j = 0; j < orderCount; j++)
            Locus[i*stateHapLength+j] = ((i & BinaryBase[j]) != 0);
   else{
      for(int i = 0; i < stateHapCount; i++)
         for(int j = 0; j < orderCount; j++)
     //       Locus[i*stateHapLength + j] = ((i & ((BinaryBase[LocusBit[j+1]]-1)) >> LocusBit[j]);
            Locus[i*stateHapLength + j] = (i >> LocusBit[j]) & BinaryBase[BitsPerLocus[AllelesPerLocus[j]]];
   }

   if(AllSNP)
   for(int i = 0; i < stateHapCount; i++)
      for(int j = 0; j < orderCount; j++)
         stateProb[i] *= ((binary[i][j] == 0)? frequencies[j] : 1-frequencies[j]);
   else
       for(int i = 0; i < stateHapCount; i++)
         for(int j = 0; j < orderCount; j++)
            stateProb[i] *= family->ped.GetMarkerInfo(order[j]) ->
               freq[Locus[i*stateHapLength+j]+1];

   IntArray buf(orderCount * 2);
   if(AllSNP){
      haploid[0].Dimension(stateCount);
      haploid[1].Dimension(stateCount);
      for(int i = 0; i < stateHapCount; i++)
         for(int j = 0; j < stateHapCount; j++){
            for(int k = 0; k < orderCount; k++){
               buf[k*2] = binary[i][k];
               buf[k*2+1] = binary[j][k];
            }
            int index = ConvertBinary(buf);
            haploid[0][index] = i;
            haploid[1][index] = j;
         }
   }
   RecomProb.Dimension(stateCount);
   RecomProb.Zero();

   TransmissionHaploidProbability.Dimension(stateHapCount);
   TransmissionHaploidProbability.Set(0.5);
   for(int o = 0; o < stateHapCount; o++)
      for(int i = 1; i < orderCount; i++)
         TransmissionHaploidProbability[o] *=
            (binary[o][i]!=binary[o][i-1] ? recombinations[i]: 1-recombinations[i]);
   buf.Dimension(orderCount*2);
   IntArray buf2(orderCount*2);
   for(int i = 0; i < BinaryBase[orderCount]; i++){
      buf.Zero();
      for(int j = 0; j < orderCount; j++){
         if(binary[i][j] == 0)
            buf[j*2] = 1;
         else buf[j*2+1] = 1;
      }
      for(int k = 0; k < BinaryBase[orderCount]; k++){
         buf2 = buf;
         for(int j = 0; j < orderCount; j++){
            if(buf[j*2]==1)
               buf2[j*2+1] = binary[k][j];
            else buf2[j*2] = binary[k][j];
         }
         RecomProb[ConvertBinary(buf2)] += TransmissionHaploidProbability[i];
      }
   }
   IsAlleleOne.Dimension(stateCount);
   if(AllSNP)
      for(int i = 0; i < stateCount; i++)
         IsAlleleOne[i] = i ^ AllOne[stateLength];
}

void FamilyLikelihoodHAP::ListStates(Person & person, IntArray & states)
{
   if(genotypeArray==NULL){
      for(int i = 0; i < orderCount; i++)
         OneState[i].Dimension(0);
      for(int i = 0; i < orderCount; i++)
         if(person.markers[order[i]].isKnown()){
            if(person.markers[order[i]][0] != person.markers[order[i]][1]){
               OneState[i].Push(1);
               OneState[i].Push(2);
            }else if(person.markers[order[i]][0] == 1) OneState[i].Push(0);
            else if(person.markers[order[i]][0] == 2) OneState[i].Push(3);
         }else for(int j = 0; j < 4; j++) OneState[i].Push(j);
   }else  // genotype elimination
      for(int i = 0; i < orderCount; i++)
         OneState[i] = IntToArray[genotypeArray[person.serial][i]];

   states.Dimension(0);
   base.Dimension(orderCount);
   base[0] = OneState[0].Length();
   for(int i = 1; i < orderCount; i++)
      base[i] = base[i-1] * OneState[i].Length();
   for(int i = 0; i < base[orderCount-1]; i++){
      int temp = OneState[0][i%base[0]];
      for(int j = 1; j < orderCount; j++)
         temp |= (OneState[j][(i%base[j])/base[j-1]]<<(2*j));
      states.Push(temp);
   }
}

double FamilyLikelihoodHAP::LikelihoodOfGenotype()
{
   double likelihood = 0.0;
   double term;
   int m, p, s;
   for(int i = 0; i < es.Type.Length(); i++){
      if(es.Type[i] == 'N') // Nuclear family
         likelihood = nuclearFamilyLikelihood();
      else if(es.Type[i] == 'E') // Ending
         likelihood = es.Items[es.From[i]].Sum();
      else{
         int MarriageNode = es.From[i] < family->count ? es.To[i]: es.From[i];
         int OtherNode = es.From[i] < family->count ? es.From[i]: es.To[i];
         int mo = es.Index[MarriageNode] / family->count;
         int fa = es.Index[MarriageNode] % family->count;
         int faLength = es.States[fa].Length();
         if(es.Type[i] == 'T'){  // Parents -> only child
            for(s = 0; s < es.States[OtherNode].Length(); s++){
               for(term = 0.0, m = 0; m < es.States[mo].Length(); m++)
                  term += es.Items[mo][m] *
//                     RecomProb[es.States[mo][m]*stateHapCount+haploid[0][es.States[OtherNode][s]]];
                     RecomProb[IsAlleleOne[es.States[mo][m] ^ binaryDouble[haploid[0][es.States[OtherNode][s]]]]];
               es.Items[OtherNode][s] *= term;
               for(term = 0.0, p = 0; p < es.States[fa].Length(); p++)
                  term += es.Items[fa][p] *
//                     RecomProb[es.States[fa][p]*stateHapCount+haploid[1][es.States[OtherNode][s]]];
                  RecomProb[IsAlleleOne[es.States[fa][p] ^ binaryDouble[haploid[1][es.States[OtherNode][s]]]]];
               es.Items[OtherNode][s] *= term;
            }
            i+=3;
         }else if(es.Type[i] == 'O' && es.From[i] < family->count){  // Offspring -> Marriage
            for(m = 0; m < es.States[MarriageNode].Length(); m++){
               for(term = 0.0, s = 0; s < es.States[OtherNode].Length(); s++)
                  term += ProbabilityOfTransmission(es.States[OtherNode][s],
                     es.States[mo][m / faLength], es.States[fa][m % faLength]) * es.Items[OtherNode][s];
               es.Items[MarriageNode][m] *= term;
            }
         }else if(es.Type[i] == 'O'){ // Marriage -> Offspring
            for(int s = 0; s < es.States[OtherNode].Length(); s++){
               for(term = 0.0, m = 0; m < es.States[MarriageNode].Length(); m++)
                  term += ProbabilityOfTransmission(es.States[OtherNode][s],
                     es.States[mo][m / faLength], es.States[fa][m % faLength]) * es.Items[MarriageNode][m];
               es.Items[OtherNode][s] *= term;
            }
         }else if(es.From[i] < family->count){  // Spouse -> Marriage
            if(family->ped[family->path[OtherNode]].sex == 1)
               for(m = 0; m < es.States[MarriageNode].Length(); m++)
                  es.Items[MarriageNode][m] *= es.Items[fa][m % faLength];
            else
               for(m = 0; m < es.States[MarriageNode].Length(); m++)
                  es.Items[MarriageNode][m] *= es.Items[mo][m / faLength];
         }else{   // Marriage -> Spouse
            if(family->ped[family->path[OtherNode]].sex == 1)//
               for(p = 0; p < es.States[OtherNode].Length(); p++){
                  for(term = 0.0, m = 0; m < es.States[mo].Length(); m++)
                     term += es.Items[MarriageNode][m * faLength + p];
                  es.Items[OtherNode][p] *= term;
               }
           else
               for(m = 0; m < es.States[OtherNode].Length(); m++){
                  for(term = 0.0, p = 0; p < es.States[fa].Length(); p++)
                     term += es.Items[MarriageNode][m * faLength + p];
                  es.Items[OtherNode][m] *= term;
               }
         }
      }
   }
   return likelihood;
}

// *************************** FamilyLikelihoodEff *********************
FamilyLikelihoodEff::FamilyLikelihoodEff(){}

int FamilyLikelihoodEff::GenotypeMeasure(int state)
{
   int myState = state & 3;
   if(myState==0) return 0;
   else if(myState==3) return 2;
   else return 1;
}

void FamilyLikelihoodEff::CopyPhase(FamilyLikelihood * engine)
{
   if(phase == NULL)
      phase = new IntArray[engine->family->ped.count];
   for(int i = 0; i < engine->family->ped.count; i++)
      phase[i] = engine->phase[i];

   markerOrder = engine->markerOrder;
   orderCount = engine->order.Length() + 1;
   if(AllSNP) stateLength = orderCount * 2;
   else{}
   stateCount = 1 << stateLength;
   stateHapCount = 1 << (stateLength/2);
   convertState.Dimension(stateCount);
   IntArray temp;
   if(orderCount *2 > MaxDimension) {
      delete []binary;
      binary = new IntArray[stateCount];
      MaxDimension = orderCount * 2;
      for(int i = 0; i < stateCount; i++)
         ConvertNumber(i, binary[i], orderCount*2);
   }
   // ConvertState: move the last two digits to the middle
   for(int i = 0; i < stateCount; i++){
      temp = binary[i];
      for(int k = orderCount*2-1; k > markerOrder*2+1; k--)
         temp[k] = binary[i][k];
      temp[markerOrder*2+1] = binary[i][1];
      temp[markerOrder*2] = binary[i][0];
      for(int k = markerOrder*2-1; k > -1; k--)
         temp[k] = binary[i][k+2];
      convertState[i] = ConvertBinary(temp);
   }
}

void FamilyLikelihoodEff::init(FamilyLikelihood * engine)
{
   order = engine->order;

   if(AllSNP) stateLength = orderCount * 2; //   else{}

   stateHapCount = 1 << (stateLength/2);
   stateCount = 1 << stateLength;

   order.Dimension(engine->order.Length()+1);
   order[markerOrder] = positionSNP;
   for(int i = markerOrder; i< engine->order.Length(); i++)
      order[i+1] = engine->order[i];
   recombinations = engine->recombinations;
   recombinations.Dimension(engine->recombinations.Length()+1);
   if(markerOrder==0) recombinations[markerOrder] = 0.5;
   else recombinations[markerOrder] = DistanceToRecombination(
      family->ped.GetMarkerInfo(positionSNP)->position -
      family->ped.GetMarkerInfo(order[markerOrder-1])->position);

   if(markerOrder < engine->order.Length())
      recombinations[markerOrder+1] = DistanceToRecombination(
         family->ped.GetMarkerInfo(order[markerOrder+1])->position -
         family->ped.GetMarkerInfo(positionSNP)->position);
   for(int i = markerOrder+1; i < engine->order.Length(); i++)
      recombinations[i+1] = engine->recombinations[i];

   frequencies = engine->frequencies;
   frequencies.Dimension(engine->frequencies.Length()+1);
   frequencies[markerOrder] = family->ped.GetMarkerInfo(positionSNP)->freq[1];
   for(int i = markerOrder+1; i < engine->order.Length()+1; i++)
      frequencies[i] = engine->frequencies[i-1];
}

void FamilyLikelihoodEff::ListStates(Person & person, IntArray & states)
{
   IntArray temp = IntToArray[genotypeArray[person.serial][0]];
   if(temp.Length()==0) {states.Dimension(0); return;}
   states = phase[person.serial];
   if(temp[0]) states.Add(temp[0]);
   for(int i = 1; i < temp.Length(); i++)
      for(int j = 0; j < phase[person.serial].Length(); j++)
         states.Push(phase[person.serial][j] + temp[i]);
}
// ********************* FamilyLikelihoodIBD ***************************

FamilyLikelihoodIBD::FamilyLikelihoodIBD(){}

void FamilyLikelihoodIBD::init(FamilyLikelihood * engine)
{
   FamilyLikelihoodEff::init(engine);
   frequencies[markerOrder] = 0.999999;
}

void FamilyLikelihoodIBD::ListStates(Person & person, IntArray & states)
{
   states = phase[person.serial];
   if(person.traverse == FixedPerson)
      states.Add(3);
   else
      for(int i = 0; i < phase[person.serial].Length(); i++){
         int state = phase[person.serial][i];
         states.Push(state+1);
         states.Push(state+2);
         states.Push(state+3);
      }
}

double FamilyLikelihoodIBD::score()
{
   ResetES();
   double lik = _NAN_;
   AlleleCount.Dimension(family->count);

   int i = FixedPerson;
   kin.Setup(*family);
   for(int j = i+1; j < family->count; j++){
      double temp = kin(family->ped[family->path[i]], family->ped[family->path[j]]);
      if(temp==0) AlleleCount[j] = 0;
      else if(temp==0.5) AlleleCount[j] = 2.0;
      else if(family->ped[family->path[j]].father->traverse == i ||
         family->ped[family->path[j]].mother->traverse == i)
         AlleleCount[j] = 1.0;
      else{
         if(lik==_NAN_) lik = LikelihoodOfGenotype();
         ResetES();
         for(int k = 0; k < es.Items[j].Length(); k++)
            es.Items[j][k] *= GenotypeMeasure(es.States[j][k]);
         AlleleCount[j] = LikelihoodOfGenotype() / lik;
      }
   }
   return lik;
}

// ****************************** FamilyLikelihoodIBD2 ********************

FamilyLikelihoodIBD2::FamilyLikelihoodIBD2(){}

void FamilyLikelihoodIBD2::CopyPhase(FamilyLikelihood * engine)
{
   if(phase == NULL)
      phase = new IntArray[engine->family->ped.count];
   for(int i = 0; i < engine->family->ped.count; i++)
      phase[i] = engine->phase[i];

   markerOrder = engine->markerOrder;
   orderCount = engine->order.Length() + 2;
   stateHapCount = (1 << orderCount);
   if(orderCount == 0)
      stateCount = 16;
   else
      stateCount = (1<<(orderCount*2));
   convertState.Dimension(stateCount);
   IntArray temp;
   if(orderCount * 2 > MaxDimension) {
      delete []binary;
      binary = new IntArray[stateCount];
      MaxDimension = orderCount * 2;
      for(int i = 0; i < stateCount; i++)
         ConvertNumber(i, binary[i], orderCount*2);
   }
   for(int i = 0; i < stateCount; i++){
      temp = binary[i];
      for(int k = orderCount*2-1; k > markerOrder*2+3; k--)
         temp[k] = binary[i][k];
      temp[markerOrder*2+3] = binary[i][1];
      temp[markerOrder*2+2] = binary[i][0];
      temp[markerOrder*2+1] = binary[i][3];
      temp[markerOrder*2] = binary[i][2];
      for(int k = markerOrder*2-1; k > -1; k--)
         temp[k] = binary[i][k+4];
      convertState[i] = ConvertBinary(temp);
   }
}

void FamilyLikelihoodIBD2::init(FamilyLikelihood * engine)
{
   order = engine->order;
   order.Dimension(engine->order.Length()+2);
   order[markerOrder] = order[markerOrder+1] = positionSNP;
   for(int i = markerOrder; i< engine->order.Length(); i++)
      order[i+2] = engine->order[i];
   recombinations = engine->recombinations;
   recombinations.Dimension(engine->recombinations.Length()+2);
   if(markerOrder==0){
      recombinations[markerOrder] = 0.5;
      recombinations[markerOrder+1] = 0;
   }else {
      recombinations[markerOrder] = DistanceToRecombination(
      family->ped.GetMarkerInfo(positionSNP)->position -
      family->ped.GetMarkerInfo(order[markerOrder-1])->position);
      recombinations[markerOrder+1] = 0;
   }
   if(markerOrder < engine->order.Length())
      recombinations[markerOrder+2] = DistanceToRecombination(
         family->ped.GetMarkerInfo(order[markerOrder+2])->position -
         family->ped.GetMarkerInfo(positionSNP)->position);
   for(int i = markerOrder+1; i < engine->order.Length(); i++)
      recombinations[i+2] = engine->recombinations[i];

   frequencies = engine->frequencies;
   frequencies.Dimension(engine->frequencies.Length()+2);
   frequencies[markerOrder] = family->ped.GetMarkerInfo(positionSNP)->freq[1];
   frequencies[markerOrder+1] = 0.999999;
   for(int i = markerOrder; i < engine->order.Length(); i++)
      frequencies[i+2] = engine->frequencies[i];
}

void FamilyLikelihoodIBD2::ListStates(Person & person, IntArray & states)
{
   int k;
   states.Dimension(0);

   if(phase[person.serial].Length()){
      IntArray temp = IntToArray[genotypeArray[person.serial][0]];
      if(temp.Length()==0) return;
      if(person.traverse == FixedPerson)
         for(int i = 0; i < phase[person.serial].Length(); i++)
            for(int j = 0; j < temp.Length(); j++)
               states.Push(((phase[person.serial][i]+temp[j])<<2)+3);
      else
         for(int i = 0; i < phase[person.serial].Length(); i++)
            for(int j = 0; j < temp.Length(); j++){
               k = (phase[person.serial][i]+temp[j])<<2;
               states.Push(k);
               states.Push(k+1);
               states.Push(k+2);
               states.Push(k+3);
            }
   }else{   // non two-stage
      IntArray ChangeOrder(orderCount-1);
      for(int i = markerOrder+1; i < orderCount-1; i++)
         ChangeOrder[i] = i;
      ChangeOrder[markerOrder] = 0;
      for(int i = 0; i < markerOrder; i++)
         ChangeOrder[i] = i+1;

      for(int i = 0; i < orderCount-1; i++)
         OneState[i] = IntToArray[genotypeArray[person.serial][i]];
      IntArray temp(0);
      base.Dimension(orderCount-1);
      base[0] = OneState[0].Length();
      for(int i = 1; i < orderCount-1; i++)
         base[i] = base[i-1] * OneState[i].Length();
      for(int i = 0; i < base[orderCount-2]; i++){
         int t = 0;
         for(int j = 0; j < orderCount-1; j++)
            t |= (OneState[j][(i%base[j])/base[j-1]]<<ChangeOrder[j]);
         temp.Push(t);
      }
      if(person.traverse == FixedPerson)
         for(int j = 0; j < temp.Length(); j++)
            states.Push((temp[j]<<2) + 3);
      else
         for(int j = 0; j < temp.Length(); j++){
            k = temp[j]<<2;
            states.Push(k);
            states.Push(k+1);
            states.Push(k+2);
            states.Push(k+3);
         }
   }
}


double FamilyLikelihoodIBD2::score()
{
   ResetES();
   double lik = _NAN_;
   AlleleCount.Dimension(family->count);

   int i = FixedPerson;
   kin.Setup(*family);
   for(int j = i+1; j < family->count; j++){
      double temp = kin(family->ped[family->path[i]], family->ped[family->path[j]]);
      if(temp==0) AlleleCount[j] = 0;
      else if(temp==0.5) AlleleCount[j] = 2.0;
      else if(family->ped[family->path[j]].father->traverse == i ||
         family->ped[family->path[j]].mother->traverse == i)
         AlleleCount[j] = 1.0;
      else{
         if(lik==_NAN_) lik = LikelihoodOfGenotype();
         ResetES();
         for(int k = 0; k < es.Items[j].Length(); k++)
            es.Items[j][k] *= GenotypeMeasure(es.States[j][k]);
         AlleleCount[j] = LikelihoodOfGenotype() / lik;
      }
   }
   return lik;
}

// ********************** Elston-Stewart *************************
ES_TRAVERSE::ES_TRAVERSE()
{Items=NULL; States=NULL;}

ES_TRAVERSE::~ES_TRAVERSE()
{if(Items) delete[]Items; if(States) delete[]States;}

ES_TRAVERSE & ES_TRAVERSE::operator=(ES_TRAVERSE & es)
{
   Index = es.Index;
   From = es.From;
   To = es.To;
   Type = es.Type;
   return *this;
}

void ES_TRAVERSE::PrintOrder(Family *family)
{
   printf("Traverse of Elston-Stewart algorithm:\n");
   for(int i = 0; i < Type.Length(); i++){
      printf("%c:", Type[i]);
      if(From[i] < family->count)
         printf("\t%d", From[i]);
      else
         printf("\t(%d %d)", Index[From[i]] / family->count, Index[From[i]] % family->count);
      if(To[i] < family->count)
         printf("\t==>\t%d", To[i]);
      else
         printf("\t==>\t(%d %d)", Index[To[i]] / family->count, Index[To[i]] % family->count);
      printf("\n");
   }
   printf("\n");
}

void ES_TRAVERSE::GetOrder(Family * family)
{
   Index.Dimension(0);
   for(int i = 0; i < family->count; i++)
      Index.Push(i);
      /*
   if(family->isNuclear()){
      From.Dimension(0);
      To.Dimension(0);
      Type.Dimension(0);
      From.Push(0);
      To.Push(0);
      Type.Push('N');
      return;
   }    */
   // build connections
   IntArray *Connection = new IntArray[family->count * 2];
   IntArray *ConnectionType = new IntArray[family->count * 2];
   for(int i = 0; i < family->count * 2; i++){
      Connection[i].Dimension(0);
      ConnectionType[i].Dimension(0);
   }
   IntArray Parents(0);
   int nodeCount = family->count;
   for(int i = 0; i < family->count; i++){
      if(family->ped[family->path[i]].isFounder()) continue;
      int node = Parents.Find(family->ped[family->path[i]].mother->traverse *
         family->count + family->ped[family->path[i]].father->traverse);
      if(node != -1){
         Connection[node + family->count].Push(i);
         ConnectionType[node + family->count].Push('O');
         Connection[i].Push(node + family->count);
         ConnectionType[i].Push('O');
      }else{ // marriage node cannot be found
         Parents.Push(family->ped[family->path[i]].mother->traverse *
            family->count + family->ped[family->path[i]].father->traverse);
         Connection[nodeCount].Push(family->ped[family->path[i]].mother->traverse);
         ConnectionType[nodeCount].Push('S');
         Connection[nodeCount].Push(family->ped[family->path[i]].father->traverse);
         ConnectionType[nodeCount].Push('S');
         Connection[nodeCount].Push(i);
         ConnectionType[nodeCount].Push('O');
         Connection[i].Push(nodeCount);
         ConnectionType[i].Push('O');
         Connection[family->ped[family->path[i]].mother->traverse].Push(nodeCount);
         ConnectionType[family->ped[family->path[i]].mother->traverse].Push('S');
         Connection[family->ped[family->path[i]].father->traverse].Push(nodeCount);
         ConnectionType[family->ped[family->path[i]].father->traverse].Push('S');
         nodeCount ++;
      }
   }
   Index.Append(Parents);
   IntArray ZeroConnection(0);
   IntArray OneConnection(0);
   IntArray MultipleConnections(0);
   for(int i = 0; i < Index.Length(); i++){
      if(Connection[i].Length() == 1) OneConnection.Push(i);
      else if(Connection[i].Length() == 0) ZeroConnection.Push(i);
      else MultipleConnections.Push(i);
   }

   // start traverse
   From.Dimension(0);
   To.Dimension(0);
   Type.Dimension(0);
   int k;
   int Trio = 0;
   int Child, Spouse[2];
   int ChildIndex;
   int SpouseIndex[2];
   int MarriageNode;

   do{
      Trio = 0;
      for(k = 0; k < MultipleConnections.Length(); k++){
         if(Connection[MultipleConnections[k]].Length()==3 && MultipleConnections[k] >= family->count){
            MarriageNode = MultipleConnections[k];
            if(ConnectionType[MarriageNode][0]=='O') ChildIndex = 0;
            else if(ConnectionType[MarriageNode][1]=='O') ChildIndex = 1;
            else if(ConnectionType[MarriageNode][2]=='O') ChildIndex = 2;
            else continue;
            Spouse[0] = Connection[MarriageNode][(ChildIndex + 1) % 3];
            Spouse[1] = Connection[MarriageNode][(ChildIndex + 2) % 3];
            Child = Connection[MarriageNode][ChildIndex];
            if(ConnectionType[Spouse[0]][0]!='S' || ConnectionType[Spouse[1]][0]!='S') continue;
            SpouseIndex[0] = OneConnection.Find(Spouse[0]);
            SpouseIndex[1] = OneConnection.Find(Spouse[1]);
            if(SpouseIndex[0] == -1 || SpouseIndex[1] == -1) continue;
            Trio = 1;
            break;
         }
      }
      if(Trio){
            From.Push(MarriageNode);
            Type.Push('T');
            To.Push(Child);
            for(int i = 0; i < 2; i++) {
               From.Push(Spouse[i]);
               Type.Push('S');
               To.Push(MarriageNode);
            }
            From.Push(MarriageNode);
            Type.Push('O');
            To.Push(Child);
            MultipleConnections.Delete(k);
            if(SpouseIndex[1]>SpouseIndex[0]){
               OneConnection.Delete(SpouseIndex[1]);
               OneConnection.Delete(SpouseIndex[0]);
            }else{
               OneConnection.Delete(SpouseIndex[0]);
               OneConnection.Delete(SpouseIndex[1]);
            }
            if(Connection[Child].Length()==1){
               OneConnection.Delete(OneConnection.Find(Child));
            }else if(Connection[Child].Length()==2){
               MultipleConnections.Delete(MultipleConnections.Find(Child));
               OneConnection.Push(Child);
            }
            int j = Connection[Child].Find(MarriageNode);
            Connection[Child].Delete(j);
            ConnectionType[Child].Delete(j);
            if(OneConnection.Length()==0){
               Type.Push('E');
               From.Push(Child);
               To.Push(Child);
               break;
            }
      }
   }while(Trio);

   for(;OneConnection.Length()>0;){
      int from = OneConnection.Pop();
      From.Push(from);
      Type.Push(ConnectionType[from][0]);
      int to = Connection[from][0];
      To.Push(to);
      int j = Connection[to].Find(from);
      Connection[to].Delete(j);
      ConnectionType[to].Delete(j);
      if(Connection[to].Length() == 1){
         MultipleConnections.Delete(MultipleConnections.Find(to));
         OneConnection.Push(to);
      }else if(Connection[to].Length() == 0)
         break;
   }
   if(OneConnection.Length()>0){
      int from = OneConnection.Pop();
      From.Push(from);
      To.Push(from);
      Type.Push('E');
   }
   if(ZeroConnection.Length()>0){
      int from = ZeroConnection.Pop();
      From.Push(from);
      To.Push(from);
      Type.Push('E');
   }else
   if(OneConnection.Length() + MultipleConnections.Length()>0){
      for(int i = family->first; i <= family->last; i++){
         printf("%d %d %d %d %d\n", family->serial,  family->count,
            family->ped[i].traverse,
            family->ped[i].father ? family->ped[i].father->traverse : 0,
            family->ped[i].mother ? family->ped[i].mother->traverse : 0);
      }
      error("Loop exists in the pedigree. Current algorithm cannot handle loops.");
   }
   delete []Connection;
   delete []ConnectionType;
}

void ES_TRAVERSE::GetOrder(Family * family, int traverse)
{
   Index.Dimension(0);
   for(int i = 0; i < family->count; i++)
      Index.Push(i);
      /*
   if(family->isNuclear()){
      From.Dimension(0);
      To.Dimension(0);
      Type.Dimension(0);
      From.Push(traverse);
      To.Push(traverse);
      Type.Push('N');
      return;
   }    */
   // build connections
   IntArray *Connection = new IntArray[family->count * 2];
   IntArray *ConnectionType = new IntArray[family->count * 2];
   for(int i = 0; i < family->count * 2; i++){
      Connection[i].Dimension(0);
      ConnectionType[i].Dimension(0);
   }
   IntArray Parents(0);
   int nodeCount = family->count;
   for(int i = 0; i < family->count; i++){
      if(family->ped[family->path[i]].isFounder()) continue;
      int node = Parents.Find(family->ped[family->path[i]].mother->traverse *
         family->count + family->ped[family->path[i]].father->traverse);
      if(node != -1){
         Connection[node + family->count].Push(i);
         ConnectionType[node + family->count].Push('O');
         Connection[i].Push(node + family->count);
         ConnectionType[i].Push('O');
      }else{ // marriage node cannot be found
         Parents.Push(family->ped[family->path[i]].mother->traverse *
            family->count + family->ped[family->path[i]].father->traverse);
         Connection[nodeCount].Push(family->ped[family->path[i]].mother->traverse);
         ConnectionType[nodeCount].Push('S');
         Connection[nodeCount].Push(family->ped[family->path[i]].father->traverse);
         ConnectionType[nodeCount].Push('S');
         Connection[nodeCount].Push(i);
         ConnectionType[nodeCount].Push('O');
         Connection[i].Push(nodeCount);
         ConnectionType[i].Push('O');
         Connection[family->ped[family->path[i]].mother->traverse].Push(nodeCount);
         ConnectionType[family->ped[family->path[i]].mother->traverse].Push('S');
         Connection[family->ped[family->path[i]].father->traverse].Push(nodeCount);
         ConnectionType[family->ped[family->path[i]].father->traverse].Push('S');
         nodeCount ++;
      }
   }
   Index.Append(Parents);
   IntArray ZeroConnection(0);
   IntArray OneConnection(0);
   IntArray MultipleConnections(0);
   for(int i = 0; i < Index.Length(); i++){
      if(Connection[i].Length() == 1) OneConnection.Push(i);
      else if(Connection[i].Length() == 0) ZeroConnection.Push(i);
      else MultipleConnections.Push(i);
   }

   int k = OneConnection.Find(traverse);
   if(k > 0){
      OneConnection[k] = OneConnection[0];
      OneConnection[0] = traverse;
   }

   // start traverse
   From.Dimension(0);
   To.Dimension(0);
   Type.Dimension(0);
   int Trio = 0;
   int Child, Spouse[2];
   int ChildIndex;
   int SpouseIndex[2];
   int MarriageNode;

   do{
      Trio = 0;
      for(k = 0; k < MultipleConnections.Length(); k++){
         if(Connection[MultipleConnections[k]].Length()==3 && MultipleConnections[k] >= family->count){
            MarriageNode = MultipleConnections[k];
            if(ConnectionType[MarriageNode][0]=='O') ChildIndex = 0;
            else if(ConnectionType[MarriageNode][1]=='O') ChildIndex = 1;
            else if(ConnectionType[MarriageNode][2]=='O') ChildIndex = 2;
            else continue;
            Spouse[0] = Connection[MarriageNode][(ChildIndex + 1) % 3];
            Spouse[1] = Connection[MarriageNode][(ChildIndex + 2) % 3];
            Child = Connection[MarriageNode][ChildIndex];
            if(Spouse[0] == traverse || Spouse[1] == traverse) continue;
            if(ConnectionType[Spouse[0]][0]!='S' || ConnectionType[Spouse[1]][0]!='S') continue;
            SpouseIndex[0] = OneConnection.Find(Spouse[0]);
            SpouseIndex[1] = OneConnection.Find(Spouse[1]);
            if(SpouseIndex[0] == -1 || SpouseIndex[1] == -1) continue;
            Trio = 1;
            break;
         }
      }
      if(Trio){
            From.Push(MarriageNode);
            Type.Push('T');
            To.Push(Child);
            for(int i = 0; i < 2; i++) {
               From.Push(Spouse[i]);
               Type.Push('S');
               To.Push(MarriageNode);
            }
            From.Push(MarriageNode);
            Type.Push('O');
            To.Push(Child);
            MultipleConnections.Delete(k);
            if(SpouseIndex[1]>SpouseIndex[0]){
               OneConnection.Delete(SpouseIndex[1]);
               OneConnection.Delete(SpouseIndex[0]);
            }else{
               OneConnection.Delete(SpouseIndex[0]);
               OneConnection.Delete(SpouseIndex[1]);
            }
            if(Connection[Child].Length()==1){
               OneConnection.Delete(OneConnection.Find(Child));
            }else if(Connection[Child].Length()==2){
               MultipleConnections.Delete(MultipleConnections.Find(Child));
               if((Child != traverse)||(OneConnection.Length()==0))
                  OneConnection.Push(Child);
               else{
                  OneConnection.Push(OneConnection[0]);
                  OneConnection[0] = Child;
               }
            }
            int j = Connection[Child].Find(MarriageNode);
            Connection[Child].Delete(j);
            ConnectionType[Child].Delete(j);
            if(OneConnection.Length()==0){
               Type.Push('E');
               From.Push(Child);
               To.Push(Child);
               break;
            }
      }
   }while(Trio);

   for(;OneConnection.Length()>0;){
      int from = OneConnection.Pop();
      From.Push(from);
      Type.Push(ConnectionType[from][0]);
      int to = Connection[from][0];
      To.Push(to);
      int j = Connection[to].Find(from);
      Connection[to].Delete(j);
      ConnectionType[to].Delete(j);
      if(Connection[to].Length() == 1){
         MultipleConnections.Delete(MultipleConnections.Find(to));
         if((to != traverse)||(OneConnection.Length()==0))
            OneConnection.Push(to);
         else{
            OneConnection.Push(OneConnection[0]);
            OneConnection[0] = to;
         };
      }else if(Connection[to].Length() == 0)
         break;
   }
   if(OneConnection.Length()>0){
      int from = OneConnection.Pop();
      From.Push(from);
      To.Push(from);
      Type.Push('E');
   }
   if(ZeroConnection.Length()>0){
      int from = ZeroConnection.Pop();
      From.Push(from);
      To.Push(from);
      Type.Push('E');
   }else
   if(OneConnection.Length() + MultipleConnections.Length()>0){
      for(int i = family->first; i <= family->last; i++){
         printf("%d %d %d\n", family->ped[i].traverse,
            family->ped[i].father ? family->ped[i].father->traverse : 0,
            family->ped[i].mother ? family->ped[i].mother->traverse : 0);
      }
      error("Loop exists in the pedigree. Current algorithm cannot handle loops.");
   }
   delete []Connection;
   delete []ConnectionType;
}






// ***************** FamilyLikelihoodPerson *********************************
FamilyLikelihoodPerson::FamilyLikelihoodPerson()
{
//    printf("Person\n");
//   FamilyLikelihoodHAP::FamilyLikelihoodHAP();
   lastPerson = 0;
}

FamilyLikelihoodPerson::~FamilyLikelihoodPerson()
{}


void FamilyLikelihoodPerson::ResetESP(int person)
{
   for(int i = 0; i < esPerson[person].Index.Length(); i++)
      esPerson[person].Items[i].Set(1.0);
   for(int i = 0; i < family->count; i++)
      if(family->ped[family->path[i]].isFounder())
         for(int j = 0; j < esPerson[person].States[i].Length(); j++)
            esPerson[person].Items[i][j] = ProbabilityOfState(esPerson[person].States[i][j]);
}

void FamilyLikelihoodPerson::SelectFamily(Family * ptr)
{
   family = ptr;
   for(int p = 0; p < family->count; p++){
      int person = family->path[p];
      if(esPerson[person].Items==NULL)
         esPerson[person].Items = new Vector[esPerson[person].Index.Length()];
      if(esPerson[person].States==NULL)
         esPerson[person].States = new IntArray[esPerson[person].Index.Length()];
//      printf("%d",esPerson[person].Index.Length());
      for(int i = 0; i < family->count; i++)
         ListStates(family->ped[family->path[i]], esPerson[person].States[i]);
      for(int i = family->count; i < esPerson[person].Index.Length(); i++){
         int mo = esPerson[person].Index[i] / family->count;
         int fa = esPerson[person].Index[i] % family->count;
         esPerson[person].States[i].Dimension(esPerson[person].States[mo].Length() * esPerson[person].States[fa].Length());
         esPerson[person].States[i].Set(-1);
      }
      for(int i = 0; i < esPerson[person].Index.Length(); i++)
         esPerson[person].Items[i].Dimension(esPerson[person].States[i].Length());
      ResetESP(person);
   }
}


double FamilyLikelihoodPerson::nuclearFamilyLikelihood()
{
   double likelihood = 0.0;
   double term;
   int m, p, s;
   int person = family->path[lastPerson];
   if(lastPerson < 2){
      likelihood = 0;
   /*
      m = 0; p = 1;
      if(lastPerson == 1) {m = 1; p = 0;}
      for(int u = 0; u < esPerson[person].States[m].Length(); u++){
         double mterm = 0;
         for(int v = 0; v < esPerson[person].States[p].Length(); v++){
            double pterm = esPerson[person].Items[p][v];
            for(int w = 0; w < family->nonFounders; w++){
               for(term = 0.0, s = 0; s < esPerson[person].States[w+2].Length(); s++)
                   term += ProbabilityOfTransmission(esPerson[person].States[w+2][s],
                     esPerson[person].States[m][u], esPerson[person].States[p][v])
                     * esPerson[person].Items[w+2][s];
               pterm *= term;
               if(pterm == 0.0) break;
            }
            mterm += pterm;
         }
         esPerson[person].Items[m][u] *= mterm;
      }
      likelihood = esPerson[person].Items[m].Sum();
      */
   }else{
//      printf("<");
//      int parentLength = esPerson[person].States[0].Length() * esPerson[person].States[1].Length();
      int kidLength = 1;
      for(int i = 2; i < 2+family->nonFounders; i++)
         kidLength *= esPerson[person].States[i].Length();
//    if( (kidLength < parentLength) && (family->nonFounders<=10) ){
         int stop[10], S[10], myOrder[10];
         for(int w = 0; w < family->nonFounders; w++) myOrder[w] = w;
         myOrder[lastPerson-2] = 0;
         myOrder[0] = lastPerson-2;
         for(int w = 0; w < family->nonFounders; w++)
            stop[w] = esPerson[person].States[myOrder[w]+2].Length();
         for(int w = family->nonFounders; w < 10; w++)
            stop[w] = 1;
         for(S[0] = 0; S[0] < stop[0]; S[0]++){
            double cterm = 0.0;
            for(S[1] = 0; S[1] < stop[1]; S[1]++)
            for(S[2] = 0; S[2] < stop[2]; S[2]++)
            for(S[3] = 0; S[3] < stop[3]; S[3]++)
            for(S[4] = 0; S[4] < stop[4]; S[4]++)
            for(S[5] = 0; S[5] < stop[5]; S[5]++)
            for(S[6] = 0; S[6] < stop[6]; S[6]++)
            for(S[7] = 0; S[7] < stop[7]; S[7]++)
            for(S[8] = 0; S[8] < stop[8]; S[8]++)
            for(S[9] = 0; S[9] < stop[9]; S[9]++){
               m = 0; p = 1;
               if(family->ped[family->path[0]].sex == 1){
                  m = 1; p = 0;
               }
               double mterm = 0;
               for(s = 0; s < esPerson[person].States[m].Length(); s++){
                  term = esPerson[person].Items[m][s];
                  for(int w = 0; w < family->nonFounders; w++){
                     term *=
                     RecomProb[IsAlleleOne[esPerson[person].States[m][s] ^
                     binaryDouble[haploid[m][esPerson[person].States[myOrder[w]+2][S[w]]]]]];
                  }
                  mterm += term;
               }
               double pterm = 0;
               for(s = 0; s < esPerson[person].States[p].Length(); s++){
                  term = esPerson[person].Items[p][s];
                  for(int w = 0; w < family->nonFounders; w++){
                     term *=
                     RecomProb[IsAlleleOne[esPerson[person].States[p][s] ^
                     binaryDouble[haploid[p][esPerson[person].States[myOrder[w]+2][S[w]]]]]];
                  }
                  pterm += term;
               }
               cterm += pterm * mterm;
            }
            esPerson[person].Items[lastPerson][S[0]] *= cterm; 
         }
         likelihood = esPerson[person].Items[lastPerson].Sum();
//            }else{}
//      printf(">");
   }
   return likelihood;
}

double FamilyLikelihoodPerson::LikelihoodOfGenotype()
{
   double likelihood = 0.0;
   double term;
   int m, p, s;

   int person = family->path[lastPerson];
   for(int i = 0; i < esPerson[person].Type.Length(); i++){
      if(esPerson[person].Type[i] == 'N'){ // Nuclear family
         likelihood = nuclearFamilyLikelihood();
      }else if(esPerson[person].Type[i] == 'E'){ // Ending
         likelihood = esPerson[person].Items[esPerson[person].From[i]].Sum();
      }else{
         int MarriageNode = esPerson[person].From[i] < family->count ? esPerson[person].To[i]: esPerson[person].From[i];
         int OtherNode = esPerson[person].From[i] < family->count ? esPerson[person].From[i]: esPerson[person].To[i];
         int mo = esPerson[person].Index[MarriageNode] / family->count;
         int fa = esPerson[person].Index[MarriageNode] % family->count;
         int faLength = esPerson[person].States[fa].Length();
         if(esPerson[person].Type[i] == 'T'){  // Parents -> only child
            for(s = 0; s < esPerson[person].States[OtherNode].Length(); s++){
               for(term = 0.0, m = 0; m < esPerson[person].States[mo].Length(); m++)
                  term += esPerson[person].Items[mo][m] *
//                     RecomProb[esPerson[person].States[mo][m]*stateHapCount+haploid[0][esPerson[person].States[OtherNode][s]]];
                     RecomProb[IsAlleleOne[esPerson[person].States[mo][m] ^ binaryDouble[haploid[0][esPerson[person].States[OtherNode][s]]]]];
               esPerson[person].Items[OtherNode][s] *= term;
               for(term = 0.0, p = 0; p < esPerson[person].States[fa].Length(); p++)
                  term += esPerson[person].Items[fa][p] *
//                     RecomProb[esPerson[person].States[fa][p]*stateHapCount+haploid[1][esPerson[person].States[OtherNode][s]]];
                     RecomProb[IsAlleleOne[esPerson[person].States[fa][p] ^ binaryDouble[haploid[1][esPerson[person].States[OtherNode][s]]]]];
               esPerson[person].Items[OtherNode][s] *= term;
            }
            i+=3;
         }else if(esPerson[person].Type[i] == 'O' && esPerson[person].From[i] < family->count){  // Offspring -> Marriage
            for(m = 0; m < esPerson[person].States[MarriageNode].Length(); m++){
               for(term = 0.0, s = 0; s < esPerson[person].States[OtherNode].Length(); s++)
                  term += ProbabilityOfTransmission(esPerson[person].States[OtherNode][s],
                     esPerson[person].States[mo][m / faLength], esPerson[person].States[fa][m % faLength]) * esPerson[person].Items[OtherNode][s];
               esPerson[person].Items[MarriageNode][m] *= term;
            }
         }else if(esPerson[person].Type[i] == 'O'){ // Marriage -> Offspring
            for(int s = 0; s < esPerson[person].States[OtherNode].Length(); s++){
               for(term = 0.0, m = 0; m < esPerson[person].States[MarriageNode].Length(); m++)
                  term += ProbabilityOfTransmission(esPerson[person].States[OtherNode][s],
                     esPerson[person].States[mo][m / faLength], esPerson[person].States[fa][m % faLength]) * esPerson[person].Items[MarriageNode][m];
               esPerson[person].Items[OtherNode][s] *= term;
            }
         }else if(esPerson[person].From[i] < family->count){  // Spouse -> Marriage
            if(family->ped[family->path[OtherNode]].sex == 1)
               for(m = 0; m < esPerson[person].States[MarriageNode].Length(); m++)
                  esPerson[person].Items[MarriageNode][m] *= esPerson[person].Items[fa][m % faLength];
            else
               for(m = 0; m < esPerson[person].States[MarriageNode].Length(); m++)
                  esPerson[person].Items[MarriageNode][m] *= esPerson[person].Items[mo][m / faLength];
         }else{   // Marriage -> Spouse
            if(family->ped[family->path[OtherNode]].sex == 1)//
               for(p = 0; p < esPerson[person].States[OtherNode].Length(); p++){
                  for(term = 0.0, m = 0; m < esPerson[person].States[mo].Length(); m++)
                     term += esPerson[person].Items[MarriageNode][m * faLength + p];
                  esPerson[person].Items[OtherNode][p] *= term;
               }
           else
               for(m = 0; m < esPerson[person].States[OtherNode].Length(); m++){
                  for(term = 0.0, p = 0; p < esPerson[person].States[fa].Length(); p++)
                     term += esPerson[person].Items[MarriageNode][m * faLength + p];
                  esPerson[person].Items[OtherNode][m] *= term;
               }
         }
      }
   }
   return likelihood;
}



double FamilyLikelihoodHAP::nuclearFamilyLikelihood()
{
   double likelihood = 0.0;
   double term;
   int m, p, s;

   int parentLength = es.States[0].Length() * es.States[1].Length();
   int kidLength = 1;
   for(int i = 2; i < 2+family->nonFounders; i++)
      kidLength *= es.States[i].Length();
   if( (kidLength < parentLength) && (family->nonFounders<=10) ){
         int stop[10], S[10], myOrder[10];
         myOrder[0] = 1;
         myOrder[1] = 0;
         for(int w = 0; w < family->nonFounders; w++)
            stop[w] = es.States[myOrder[w]+2].Length();
         for(int w = family->nonFounders; w < 10; w++)
            stop[w] = 1;
         for(S[0] = 0; S[0] < stop[0]; S[0]++){
            double cterm = 0.0;
            for(S[1] = 0; S[1] < stop[1]; S[1]++)
/*            for(S[2] = 0; S[2] < stop[2]; S[2]++)
            for(S[3] = 0; S[3] < stop[3]; S[3]++)
            for(S[4] = 0; S[4] < stop[4]; S[4]++)
            for(S[5] = 0; S[5] < stop[5]; S[5]++)
            for(S[6] = 0; S[6] < stop[6]; S[6]++)
            for(S[7] = 0; S[7] < stop[7]; S[7]++)
            for(S[8] = 0; S[8] < stop[8]; S[8]++)
            for(S[9] = 0; S[9] < stop[9]; S[9]++)*/{
//               m = 0; p = 1;
//               if(family->ped[family->path[0]].sex == 1){ m = 1; p = 0;  }

               double pterm = 0, product = 1.0;
               for(p = 0; p < 2; p++){
                  pterm = 0.0;
                  for(s = 0; s < es.States[p].Length(); s++){
                     term = es.Items[p][s];
                     for(int w = 0; w < family->nonFounders; w++){
                        term *=
                        RecomProb[IsAlleleOne[es.States[p][s] ^
                     binaryDouble[haploid[p][es.States[myOrder[w]+2][S[w]]]]]];
                        if(term == 0.0) break;
                     }
                     pterm += term;
                  }
                  product *= pterm;
               }
               cterm += product;

               /*
               if(mterm != 0.0)
               for(s = 0; s < es.States[1].Length(); s++){
                  term = es.Items[1][s];
                  for(int w = 0; w < family->nonFounders; w++){
                     term *=
                     RecomProb[IsAlleleOne[es.States[1][s] ^
                     binaryDouble[haploid[1][es.States[myOrder[w]+2][S[w]]]]]];
                     if(term == 0.0) break;
                  }
                  pterm += term;
               } */
//               cterm += pterm * mterm;
            }
            likelihood += es.Items[myOrder[0]+2][S[0]] * cterm;
         }
//         likelihood = es.Items[myOrder[0]+2].Sum();
   }else{
//      printf(".");
      for(int u = 0; u < es.States[0].Length(); u++){
         double mterm = es.Items[0][u];
         if (mterm == 0.0) continue;
         for(int v = 0; v < es.States[1].Length(); v++){
            double pterm = es.Items[1][v];
            if(pterm == 0.0) continue;
            for(int w = 0; w < family->nonFounders; w++){
               double term = 0.0;
               for(int s = 0; s < es.States[w+2].Length(); s++)
                  term += ProbabilityOfTransmission(es.States[w+2][s],
                     es.States[0][u], es.States[1][v]) * es.Items[w+2][s];
                  pterm *= term;
                  if(pterm == 0.0) break;
            }
            likelihood += mterm * pterm;
         }
      }
   }
   return likelihood;

/*
            for(int u = 0; u < es.States[0].Length(); u++){
               double mterm = 0;
               for(int v = 0; v < es.States[1].Length(); v++){
                  double pterm = es.Items[1][v];
                  for(int w = 0; w < family->nonFounders; w++){
                     for(term = 0.0, s = 0; s < es.States[w+2].Length(); s++)
                        term += ProbabilityOfTransmission(es.States[w+2][s],
                           es.States[0][u], es.States[1][v]) * es.Items[w+2][s];
                     pterm *= term;
                     if(pterm == 0.0) break;
                  }
                  mterm += pterm;
               }
               es.Items[0][u] *= mterm;
            }
            likelihood = es.Items[0].Sum();
*/
}


// ****************************** BiallelicLocus ************************* //

BiallelicLocus::BiallelicLocus()
{
   FamilyLikelihoodNull::FamilyLikelihoodNull();
   genotype_mean[0] = genotype_mean[1] = genotype_mean[2] = 0.0;
   variance = 1.0;
   mu = a = d = 0;
}

double BiallelicLocus::ProbabilityOfPhenotype(Person & person, int state)
{
   if (person.traits[trait] == _NAN_) return 1.0;
   double delta = person.traits[trait] - genotype_mean[GenotypeMeasure(state)];
   return 1.0 / sqrt(2 * M_PI * variance) * exp(-delta*delta / (2 * variance));
}


void BiallelicLocus::UpdateSegregation()
{
   genotype_mean[0] = mu + a;
   genotype_mean[1] = mu + d;
   genotype_mean[2] = mu - a;
}














// ************************ PolygenicModel ******************************

/*
SimplePolygenicModel::SimplePolygenicModel()
   {
   frequency = 0.5;
   genotype_mean[0] = genotype_mean[1] = genotype_mean[2] = 0.0;
   residual_variance = 1.0;
   polygenic_variance = 1.0;

   trait = 0;
   marker = 0;
   grid_points = 10;
   }

double SimplePolygenicModel::ProbabilityOfPhenotype(Person & person, int state)
   {
//   int    genotype = state % grid_points;

   // We evaluate the polygenic effect at an arbitrary number of
   // grid_points between -3.0 * sqrt(polygenic_variance) and
   // +3.0 * sqrt(polygenic_variance) -- for simplicity we assume
   // more extreme effects are very unlikely.

   double polygene_index = state / grid_points;
   double polygene_effect = ((polygene_index / grid_points) * 6 - 3) * sqrt(polygenic_variance);

   if (person.traits[trait] == _NAN_)
      return 1.0;

   double delta = person.traits[trait] - genotype_mean[state] - polygene_effect;

   return 1.0 / sqrt(2 * M_PI * residual_variance) * exp(-delta*delta / (2 * residual_variance));
   }

double SimplePolygenicModel::ProbabilityOfState(int state)
   {
   int    genotype = state % grid_points;
   double polygene_index = state / grid_points;
   double polygene_effect = ((polygene_index / grid_points) * 6 - 3);

   double genotype_prob;
   switch (genotype)
      {
      case 0:
         genotype_prob = frequency * frequency;
      case 1:
         genotype_prob = frequency * (1.0 - frequency) * 2.0;
      case 2:
         genotype_prob = (1.0 - frequency) * (1.0 - frequency);
      }

   // Polygenic deviation should be distributed around mean of 0
   // and with standard deviation of 1.0

   return genotype_prob * 1.0 / sqrt(2 * M_PI) * exp(polygene_effect * polygene_effect / 2.0);
   }

double SimplePolygenicModel::ProbabilityOfTransmission(int state, int maternal_state, int paternal_state)
   {
//   int    genotype = state % grid_points;
   double polygene_index = state / grid_points;
   double polygene_effect = ((polygene_index / grid_points) * 6 - 3);

//   int    mgenotype = maternal_state % grid_points;
   double mpolygene_index = maternal_state / grid_points;
   double mpolygene_effect = ((mpolygene_index / grid_points) * 6 - 3);

//   int    pgenotype = paternal_state % grid_points;
   double ppolygene_index = paternal_state / grid_points;
   double ppolygene_effect = ((ppolygene_index / grid_points) * 6 - 3);

   static double transmission_matrix[] =
      {
      1.0, 0.0, 0.0,
      0.5, 0.5, 0.0,
      0.0, 1.0, 0.0,
      0.5, 0.5, 0.0,
      0.25, 0.5, 0.25,
      0.0, 0.5, 0.5,
      0.0, 1.0, 0.0,
      0.0, 0.5, 0.5,
      0.0, 0.0, 1.0,
      };

   double genotype_prob =
          transmission_matrix[maternal_state * 9 + paternal_state * 3 + state];

   if (genotype_prob == 0.0)
      return 0.0;

   // Polygenic deviation should be distributed as normal
   // around mean of parental polygenic deviations and
   // with a variance of 0.5

   double pdelta = polygene_effect - (mpolygene_effect + ppolygene_effect) / 2;

   return genotype_prob *
          1.0 / sqrt(M_PI) * exp(pdelta * pdelta);
   }

void SimplePolygenicModel::ListStates(Person & person, IntArray & states)
{
   // These are the genotype based states
   if (person.markers[marker].isKnown())
      {
      if (person.markers[marker].Hi() > 2)
         printf("SimplePolygenicModel::ListStates() can only handle markers with two alleles\n");

      states.Dimension(1);
      states[0] = person.markers[marker][0] + person.markers[marker][1] - 2;
      }
   else
      {
      states.Dimension(3);

      states[0] = 0;
      states[1] = 1;
      states[2] = 2;
      }

   int geno_states = states.Length();

   states.Dimension(geno_states * grid_points);

   // Here we expand them to include the polygenic effects too
   for (int i = 1; i < grid_points; i++)
      for (int j = 0; j < geno_states; j++)
         states[i * geno_states + j] = grid_points * i + states[j];
}

  */


  /*
double FamilyLikelihoodNull::DerivativeOfState(int state)
{
   switch (state) {
      case 0:
         return frequency * 2;
      case 1:
         return 2.0 - frequency * 4;
      case 2:
         return - 2 * (1.0 - frequency);
   }
   error("Inconsistent state evaluating BiallelicLocus::ProbabilityOfState()");
   return -1;
}   */
/*
double FamilyLikelihood::ScoreOfGenotype()
{
   double likelihood0 = 0;
   if(!LikelihoodOfGenotypeCalculated) LikelihoodOfGenotype();
   likelihood0 = LikGeno;
   double likelihood = 0.0;
   for (int i = 0; i < maternal_states.Length(); i++){
      double mterm = DerivativeOfState(maternal_states[i]);
      if (mterm == 0.0) continue;
      for (int j = 0; j < paternal_states.Length(); j++) {
         double pterm = ProbabilityOfState(paternal_states[j]);
         if (pterm == 0.0) continue;
         for (int k = 0; k < family->nonFounders; k++) {
            double cterm = 0.0;
            for (int m = 0; m < child_states[k].Length(); m++)
               cterm += ProbabilityOfTransmission(child_states[k][m], maternal_states[i], paternal_states[j]);
            pterm *= cterm;
            if (pterm == 0.0) break;
         }
         likelihood += mterm * pterm;
      }
   }
   for (int i = 0; i < maternal_states.Length(); i++){
      double mterm = ProbabilityOfState(maternal_states[i]);
      if (mterm == 0.0) continue;
      for (int j = 0; j < paternal_states.Length(); j++) {
         double pterm = DerivativeOfState(paternal_states[j]);
         if (pterm == 0.0) continue;
         for (int k = 0; k < family->nonFounders; k++) {
            double cterm = 0.0;
            for (int m = 0; m < child_states[k].Length(); m++)
               cterm += ProbabilityOfTransmission(child_states[k][m], maternal_states[i], paternal_states[j]);
            pterm *= cterm;
            if (pterm == 0.0) break;
         }
         likelihood += mterm * pterm;
      }
   }
   return likelihood / likelihood0;
//   return likelihood;
}
  */
                 /*
   for(int m = 0; m < stateCount; m++){
      for(int p = 0; p < stateCount; p++)
         printf("%4.2lf ", stateTransmission[m*stateCount*stateCount+p*stateCount]);
      printf("\n");
   }               */

   /*
void FamilyLikelihood::SelectNuclearFamily(Family * ptr)
{
   family = ptr;
   // Check that this is a nuclear family
   if (family->isNuclear() == false)
      error("I can only handle nuclear families, but family %s doesn't look like one\n",
            (const char *) family->famid);

   // Free memory if necessary
   if (child_states != NULL)
      delete [] child_states;

   // Allocate new array of child states
   child_states = new IntArray [family->nonFounders];

   // Find out who is the mother and father
   mother = &family->ped[family->path[0]];
   father = &family->ped[family->path[1]];

   // Check that we extracted parents correctly
   if (mother->sex == 1) {
      mother = &family->ped[family->path[1]];
      father = &family->ped[family->path[0]];
   }

   // List possible allelic states for each individual
   ListStates(*mother, maternal_states);
   ListStates(*father, paternal_states);
   for (int i = 2; i < family->count; i++)
      ListStates(family->ped[family->path[i]], child_states[i - 2]);
}

double FamilyLikelihood::NuclearLikelihood()
{
   double likelihood = 0.0;
   for (int i = 0; i < maternal_states.Length(); i++){
      double mterm = ProbabilityOfPhenotype(*mother, maternal_states[i]) *
                     ProbabilityOfState(maternal_states[i]);
      if (mterm == 0.0) continue;
      for (int j = 0; j < paternal_states.Length(); j++) {
         double pterm = ProbabilityOfPhenotype(*father, paternal_states[j]) *
                        ProbabilityOfState(paternal_states[j]);
         if (pterm == 0.0) continue;
         for (int k = 0; k < family->nonFounders; k++) {
            double cterm = 0.0;
            for (int m = 0; m < child_states[k].Length(); m++)
               cterm += ProbabilityOfPhenotype(family->ped[family->path[2+k]], child_states[k][m]) *
               ProbabilityOfTransmission(child_states[k][m], maternal_states[i], paternal_states[j]);
            pterm *= cterm;
            if (pterm == 0.0) break;
         }
         likelihood += mterm * pterm;
      }
   }
   return likelihood;
}

double FamilyLikelihood::NuclearLikelihoodOfGenotype()
{
   double likelihood = 0.0;
   for (int i = 0; i < maternal_states.Length(); i++){
      double mterm = ProbabilityOfState(maternal_states[i]);
      if (mterm == 0.0) continue;
      for (int j = 0; j < paternal_states.Length(); j++) {
         double pterm = ProbabilityOfState(paternal_states[j]);
         if (pterm == 0.0) continue;
         for (int k = 0; k < family->nonFounders; k++) {
            double cterm = 0.0;
            for (int m = 0; m < child_states[k].Length(); m++)
               cterm += ProbabilityOfTransmission(child_states[k][m], maternal_states[i], paternal_states[j]);
            pterm *= cterm;
            if (pterm == 0.0) break;
         }
         likelihood += mterm * pterm;
      }
   }
   return likelihood;
}

double FamilyLikelihood::NuclearScore()
{
   double score0 = 0;
   AlleleCount.Dimension(family->count);
   AlleleCount.Zero();
   Vector tempCount(family->nonFounders);
   for (int i = 0; i < maternal_states.Length(); i++) {
      double mprob = ProbabilityOfState(maternal_states[i]);
      if (mprob == 0.0) continue;
      for (int j = 0; j < paternal_states.Length(); j++) {
         double pprob = ProbabilityOfState(paternal_states[j]);
         if (pprob == 0.0) continue;
         for (int k = 0; k < family->nonFounders; k++) {
            double cprob = 0.0;
            double cterm = 0.0;
            for (int m = 0; m < child_states[k].Length(); m++){
               double temp = ProbabilityOfTransmission(child_states[k][m],
                                 maternal_states[i], paternal_states[j]);
               cprob += temp;
               cterm += temp * GenotypeMeasure(child_states[k][m]);
            }
            pprob *= cprob;
            if (pprob == 0.0) break;
            tempCount[k] = cterm / cprob;
         }
         if(pprob == 0.0) continue;
         pprob *= mprob;
         AlleleCount[0] += pprob * GenotypeMeasure(maternal_states[i]);
         AlleleCount[1] += pprob * GenotypeMeasure(paternal_states[j]);
         for(int k = 0; k < family->nonFounders; k++)
            AlleleCount[k+2] += pprob * tempCount[k];
         score0 += pprob;
      }
   }
   for(int k = 0; k < family->count; k++)
         AlleleCount[k] /= score0;
   return score0;
}
     */



//   return stateTransmission[maternal_state * stateCount * stateCount + paternal_state * stateCount + state];
/*
   double r = 0;
   int o, i;
   for(o = 0; o < stateCount; o++){
      for(i = 0; i < orderCount; i++)
         if( (binary[maternal_state][i*2+binary[o][i*2]] != binary[state][i*2]) ||
         (binary[paternal_state][i*2+binary[o][i*2+1]] != binary[state][i*2+1]) )
         break;
      if(i == orderCount) r += TransmissionProbability[o];
   }
   return r;
*/
/*
    Vector TransmissionProbability;
   stateTransmission.Dimension(1 << (order.Length() * 6));
   stateTransmission.Zero();
   IntArray buf(order.Length() * 2);
   buf.Zero();
   TransmissionProbability.Dimension(stateCount);
   TransmissionProbability.Set(0.25);
   for(int o = 0; o < stateCount; o++)
      for(int i = 1; i < order.Length(); i++)
         TransmissionProbability[o] *=
            (binary[o][i*2]!=binary[o][i*2-2] ? recombinations[i]: 1-recombinations[i]) *
            (binary[o][i*2+1]!=binary[o][i*2-1] ? recombinations[i]: 1-recombinations[i]);
   for(int m = 0; m < stateCount; m++)
      for(int p = 0; p < stateCount; p++)
         for(int o = 0; o < stateCount; o++){
            for(int i = 0; i < order.Length(); i++){
               buf[i*2] = binary[m][i*2+binary[o][i*2]];
               buf[i*2+1] = binary[p][i*2+binary[o][i*2+1]];
            }
            int index = m * stateCount * stateCount + p * stateCount + ConvertBinary(buf);
            stateTransmission[index] += TransmissionProbability[o];
         }
*/
   /*
   IntArray buf(order.Length()*2);
   IntArray temp;
   if(FlankingCount==0) states = state[0];
   else for(int i = 0; i < state[0].Length(); i++){
      ConvertNumber(state[0][i], temp, 2);
      buf[0] = temp[0];
      buf[1] = temp[1];
      if(FlankingCount > 0) for(int j = 0; j < state[1].Length(); j++){
         ConvertNumber(state[1][j], temp, 2);
         buf[2] = temp[0];
         buf[3] = temp[1];
         if(FlankingCount > 1) for(int k = 0; k < state[2].Length(); k++){
            ConvertNumber(state[2][k], temp, 2);
            buf[4] = temp[0];
            buf[5] = temp[1];
            states.Push(ConvertBinary(buf));
         }else states.Push(ConvertBinary(buf));
      }else states.Push(ConvertBinary(buf));
   } */
/*
   int temp;
   if(FlankingCount==0) states = state[0];
   else for(int i = 0; i < state[0].Length(); i++){
      temp = state[0][i];
      if(FlankingCount > 0) for(int j = 0; j < state[1].Length(); j++){
         temp = (state[1][j] << 2) + state[0][i];
         if(FlankingCount > 1) for(int k = 0; k < state[2].Length(); k++){
            temp = (state[2][k] << 4) + (state[1][j] << 2) + state[0][i];
            states.Push(temp);
         }else states.Push(temp);
      }else states.Push(temp);
   }
   */

/*   if(order[markerOrder] == 20 && person.traverse == 2){
      for(int i = 0; i < orderCount; i++) state[i].Print();
      base.Print();
      states.Print();
      printf("\n");
   }*/
/*
void FamilyLikelihoodHAP::SelectLocus(int m, int fm1, int fm2)
{
   FlankingCount = 0;
   if(fm1) FlankingCount ++;
   if(fm2) FlankingCount ++;
   order.Dimension(1+FlankingCount);
   if(FlankingCount == 0){
      order[0] = m;
      markerOrder = 0;
   }else if(FlankingCount == 1) {
      if(fm1 < 0){
         order[0] = m + fm1;
         order[1] = m;
         markerOrder = 1;
      }else if(fm1 > 0){
         order[0] = m;
         order[1] = m + fm1;
         markerOrder = 0;
      }else error("SelectLocus(m, fm1)");
   }else if(FlankingCount == 2) {
      if((fm1 < 0) && (fm2 > 0) ){
         order[0] = m + fm1;
         order[1] = m;
         order[2] = m + fm2;
         markerOrder = 1;
      }else error("Marker under test should be flanked by two markers.");
   }

   orderCount = order.Length();
   stateCount = 1 << (orderCount * 2);
   stateProb.Dimension(stateCount);
   stateProb.Set(1);
   if(binary) delete []binary;
   binary = new IntArray[stateCount];
   for(int i = 0; i < stateCount; i++){
      ConvertNumber(i, binary[i], orderCount * 2);
      for(int j = 0; j < orderCount * 2; j++)
         stateProb[i] *= ((binary[i][j] == 0)? frequencies[j/2] : 1-frequencies[j/2]);
   }
   IntArray buf(orderCount * 2);
   haploid[0].Dimension(stateCount);
   haploid[1].Dimension(stateCount);
   for(int i = 0; i < (1<<orderCount); i++)
      for(int j = 0; j < (1<<orderCount); j++){
         for(int k = 0; k < orderCount; k++){
            buf[k*2] = binary[i][k];
            buf[k*2+1] = binary[j][k];
         }
         int index = ConvertBinary(buf);
         haploid[0][index] = i;
         haploid[1][index] = j;
      }
   RecomProb.Dimension(stateCount, (1<<orderCount));
   RecomProb.Zero();

   Vector TransmissionHaploidProbability(1<<orderCount);
   TransmissionHaploidProbability.Set(0.5);
   for(int o = 0; o < 1<<orderCount; o++)
      for(int i = 1; i < orderCount; i++)
         TransmissionHaploidProbability[o] *=
            (binary[o][i]!=binary[o][i-1] ? recombinations[i]: 1-recombinations[i]);

   buf.Dimension(orderCount);
   for(int p = 0; p < stateCount; p++)
      for(int o = 0; o < 1<<orderCount; o++){
         for(int i = 0; i < orderCount; i++)
            buf[i] = binary[p][i*2+binary[o][i]];
         RecomProb[p][ConvertBinary(buf)] += TransmissionHaploidProbability[o];
       }
}
*/

/*
void FamilyLikelihoodNull::SelectFamily(Family * ptr)
{
   FamilyLikelihood::SelectFamily(ptr);
   path.Dimension(pheno.Length());
   int j;
   for(int i = 0; i < pheno.Length(); i++){
      for(j = 0; j < family->count; j++)
         if(family->path[j] == pheno[i]) break;
      if(j < family->count) path[i] = j;
   }  // index i for phenotype corresponds index path[i] for genotype
} */
//   return flankingStateProb[state>>2] * stateProb[state&3];
/*
void FamilyLikelihoodEff::SelectLocus()
{
   FamilyLikelihoodHAP::SelectLocus();
}
*/
/*
   if(binary) delete []binary;
   binary = new IntArray[stateCount];
   for(int i = 0; i < stateCount; i++)
      ConvertNumber(i, binary[i], orderCount * 2);

   for(int i = 0; i < 2; i++){
      tr_pos[i].Dimension(stateCount);
      tr_pos[i].Set(-1);
   }
   for(int i = 0; i < stateCount; i++){
      for(tr_pos[0][i] = markerOrder - 1; (tr_pos[0][i]>-1) &&
         (binary[i][tr_pos[0][i]*2] == binary[i][tr_pos[0][i]*2+1]); i--);
      for(tr_pos[1][i] = markerOrder; (tr_pos[1][i]<orderCount) &&
         (binary[i][tr_pos[1][i]*2] == binary[i][tr_pos[1][i]*2+1]); i++);
      if(tr_pos[1][i] == orderCount) tr_pos[1][i] = -1;
   }

   for(int i = 0; i < 2; i++){
      tr_chr[i].Dimension(stateCount * (1 << orderCount));
   }
   for(int i = 0; i < stateCount; i++)
      for(int j = 0; j < (1 << orderCount); j++){
         int index = i << orderCount + j;
         for(int k = 0; k < 2; k++){
         if(tr_pos[k][i] == -1) tr_chr[k][index] = -1;
         else{
            if(binary[i][tr_chr[k][i]*2] == binary[j][tr_chr[k][i]*2])
               tr_chr[k][index] = 0;
            else tr_chr[k][index] = 1;
         }
      }
   }
   for(int i = 0; i < 2; i++) distanceOfMarker[i].Dimension(orderCount);
   for(int i = 0; i < orderCount; i++){
      distanceOfMarker[0][i] = DistanceToRecombination(
      fabs(family->ped.GetMarkerInfo(order[i])->position - family->ped.GetMarkerInfo(positionSNP)->position));
      distanceOfMarker[1][i] = 1-distanceOfMarker[0][i];
   }
   stateProb.Dimension(4);
   stateProb[0] = frequencies[markerOrder] * frequencies[markerOrder];
   stateProb[1] = stateProb[2] = frequencies[markerOrder] * (1-frequencies[markerOrder]);
   stateProb[3] = (1-frequencies[markerOrder]) * (1-frequencies[markerOrder]);

   static double transmission_matrix[] = {
      1.0, 0.0, 0.0, 0.0,  // 00 x 00
      0.5, 0.0, 0.5, 0.0,  // 00 x 01
      0.5, 0.0, 0.5, 0.0,  // 00 x 10
      0.0, 0.0, 1.0, 0.0,  // 00 x 11
      0.5, 0.5, 0.0, 0.0,  // 01 x 00
      0.25,0.25,0.25,0.25, // 01 x 01
      0.25,0.25,0.25,0.25, // 01 x 10
      0.0, 0.0, 0.5, 0.5,  // 01 x 11
      0.5, 0.5, 0.0, 0.0,  // 10 x 00
      0.25,0.25,0.25,0.25, // 10 x 01
      0.25,0.25,0.25,0.25, // 10 x 10
      0.0, 0.0, 0.5, 0.5,  // 10 x 11
      0.0, 1.0, 0.0, 0.0,  // 11 x 00
      0.0, 0.5, 0.0, 0.5,  // 11 x 01
      0.0, 0.5, 0.0, 0.5,  // 11 x 10
      0.0, 0.0, 0.0, 1.0   // 11 x 11
   };
   for(int i = 0; i < 64; i++)
      stateTransmission[i] = transmission_matrix[i];

   static int temp_matrix[] = {
      -1, -1, // 00 ->
      0, 1,   // 01 ->
      1, 0,   // 10 ->
      -1, -1   // 11 ->
   };
   tr_m_chr.Dimension(8);
   for(int i = 0; i < 8; i++) tr_m_chr[i] = temp_matrix[i];

//   Vector Transmission_m(8);
  */

    /*
   flankingStateProb.Dimension(1);
   flankingStateProb[0] = 1;
   RecomProb.Dimension(1,1);
   RecomProb[0][0] = 1;
   for(int i = 0; i < 2; i++){
      haploid[i].Dimension(1);
      haploid[i][0] = 0;
   } */

   /*
   int parent[2];
   int child[2];
   parent[0] = maternal_state>>2;
   parent[1] = paternal_state>>2;
   child[0] = haploid[0][state>>2];
   child[1] = haploid[1][state>>2];
   int parent_m[2];
   int child_m[2];
   parent_m[0] = maternal_state & 3;
   parent_m[1] = paternal_state & 3;
   child_m[0] = state & 1;
   child_m[1] = (state>>1) & 1;
   double r = stateTransmission[parent_m[0] * 16 + parent_m[1] * 4 + state & 3];
   for(int k = 0; k < 2; k++){
      if(tr_m_chr[(parent_m[k] << 1) + child_m[k]] == -1)
         r *= RecomProb[parent[k]][child[k]];
      else if(tr_pos[0][parent[k]] == -1 && tr_pos[1][parent[k]] == -1);
      else if(tr_pos[0][parent[k]] == -1)
         r *= RecomProb[parent[k]][child[k]] *
            distanceOfMarker[tr_m_chr[(parent_m[k]<<1)+child_m[k]] != tr_chr[1][parent[k]*orderCount+child[k]]]
               [tr_pos[1][parent[k]]];
      else if(tr_pos[1][parent[k]] == -1)
         r *= RecomProb[parent[k]][child[k]] *
            distanceOfMarker[tr_m_chr[(parent_m[k]<<1)+child_m[k]] != tr_chr[0][parent[k]*orderCount+child[k]]]
               [tr_pos[0][parent[k]]];
      else{
         double d0 =
            distanceOfMarker[tr_m_chr[(parent_m[k]<<1)+child_m[k]] != tr_chr[0][parent[k]*orderCount+child[k]]]
               [tr_pos[0][parent[k]]];
         double d1 =
            distanceOfMarker[tr_m_chr[(parent_m[k]<<1)+child_m[k]] != tr_chr[1][parent[k]*orderCount+child[k]]]
               [tr_pos[1][parent[k]]];
         r *= RecomProb[parent[k]][child[k]] * d0 * d1 /
            DistanceToRecombination(RecombinationToDistance(d0) + RecombinationToDistance(d1));
      }
   }

      */





//   double r = RecomProb[maternal_state>>2][haploid[0][state>>2]>>1]
//      * RecomProb[paternal_state>>2][haploid[1][state>>2]>>1];
//   r *= stateTransmission[maternal_state * 16 + paternal_state * 4 + state];





   /*
   // heterozygousPosition[0][maternal_state>>2]
   // heterozygousScore[0][maternal_state>>2]
   // heterozygousScore[0][maternal_state>>2]
   // stateScore[0][state>>2] && 1

   int chr_f1 = -1;
   int chr_f2 = -1;
   int chr_m = state & 1;
   if( (maternal_state & 3) == 1 || (maternal_state & 3) == 2 ){
      // Get recombination of chr_f1, chr_f2 here
      if(chr_f1!=-1){
         double theta = ped.GetMarkerInfo(
         r *= (chr_f1 == chr_m)? 1-theta: theta;
      }
      // r *= (chr_f2 == chr_m)? 1-theta: theta;
   }
   chr_m = (state>>1) & 1;
   if( (paternal_state & 3) == 1 || (paternal_state & 3) == 2 ){
      // Get recombination of chr_f1, chr_f2 here
      // r *= (chr_f1 == chr_m)? 1-theta: theta;
      // r *= (chr_f2 == chr_m)? 1-theta: theta;
   }
   return r;
   */



/*
   double freq[2];
   freq[0] = frequencies[markerOrder];
   freq[1] = 1-frequencies[markerOrder];
   return stateProb[state>>2] * freq[state&2] * freq[state&1];
   */
    /*
double FamilyLikelihoodPerson::score()
{
   lastPerson = 0;
   int person = family->path[lastPerson];
   ResetESP(person);

   double lik = LikelihoodOfGenotype();
   AlleleCount.Dimension(family->count);
   for(int i = 0; i < family->count; i++)
      if(family->ped[family->path[i]].markers[order[markerOrder]].isKnown())
         AlleleCount[i] = GenotypeMeasure(esPerson[person].States[i][0]);
      else{
         lastPerson = i;
         person = family->path[lastPerson];
         ResetESP(person);
         for(int j = 0; j < esPerson[person].Items[i].Length(); j++)
            esPerson[person].Items[i][j] *= GenotypeMeasure(esPerson[person].States[i][j]);
         AlleleCount[i] = LikelihoodOfGenotype() / lik;

      }
   return lik;
}     */
//   frequencies = engine->frequencies;
//   frequencies.Dimension(engine->frequencies.Length()+1);
//   for(int i = markerOrder+1; i < engine->orderCount+1; i++)
//      frequencies[i] = engine->frequencies[i-1];
/*
   for(int i = 0; i < stateCount; i++)
      for(int j = 0; j < order.Length() * 2; j++)
         stateProb[i] *= ((binary[i][j] == 0)? frequencies[j/2] : 1-frequencies[j/2]);
*/
/*
   for(int i = 0; i < stateCount; i++)
      for(int j = 0; j < orderCount * 2; j++)
         stateProb[i] *= ((binary[i][j] == 0)? frequencies[j/2] : 1-frequencies[j/2]);
*/

//   return stateProb[state];
/*
   IntArray buf;
   ConvertNumber(state, buf, order.Length()*2);
   int myState = buf[markerOrder*2+1] * 2 + buf[markerOrder*2];
   if(myState==0) return 0;
   else if(myState==3) return 2;
   else return 1;
   */
/*
//   if(orderCount <= 5) // 32K
   buf.Dimension(orderCount);
   for(int p = 0; p < stateCount; p++)
      for(int o = 0; o < stateHapCount; o++){
         for(int i = 0; i < orderCount; i++)
            buf[i] = binary[p][i*2+binary[o][i]];
         RecomProb[p*stateHapCount+ConvertBinary(buf)] += TransmissionHaploidProbability[o];
       }
      */
   /*
      for(int i = 0; i < order.Length(); i++){
         OneState[i].Dimension(0);
         int temp = genotypeArray[person.serial][i];
         if(temp == 0) OneState[i].Push(0);
         else
            for(; temp; temp >>= 2) OneState[i].Push(temp & 3);
      }  */
//   return RecomProb[convertState[maternal_state]*stateHapCount+haploid[0][convertState[state]]]
//     * RecomProb[convertState[paternal_state]*stateHapCount+haploid[1][convertState[state]]];
/*
void FamilyLikelihoodEff::ListStates(Person & person, IntArray & states)
{
   states.Dimension(0);
   for(int i = 0; i < phase[person.serial].Length(); i++){
      int state = (phase[person.serial][i]<<2);
      if(person.markers[positionSNP].isHomozygous()){
         states.Push(state + (person.markers[positionSNP][0]-1)*3);
      }else if(person.markers[positionSNP].isHeterozygous()){
         states.Push(state+1);
         states.Push(state+2);
      }else{
         states.Push(state);
         states.Push(state+1);
         states.Push(state+2);
         states.Push(state+3);
      }
   }
}
*/
/*
            for(int u = 0; u < es.States[0].Length(); u++){
               double mterm = 0;
               for(int v = 0; v < es.States[1].Length(); v++){
                  double pterm = es.Items[1][v];
                  for(int w = 0; w < family->nonFounders; w++){
                     for(term = 0.0, s = 0; s < es.States[w+2].Length(); s++)
                        term += ProbabilityOfTransmission(es.States[w+2][s],
                           es.States[0][u], es.States[1][v]) * es.Items[w+2][s];
                     pterm *= term;
                     if(pterm == 0.0) break;
                  }
                  mterm += pterm;
               }
               es.Items[0][u] *= mterm;
            }
            likelihood = es.Items[0].Sum();
*/
/*   for(int i = 0; i < phase[person.serial].Length(); i++){
      int state = phase[person.serial][i]<<2;
      for(int j = 0; j < temp.Length(); j++)
         states.Push(temp[j]|state);
   }
   */
/* {     int state = (phase[person.serial][i]<<2);
      for(int j = 0; j < 4; j++)
         if(genotypeArray[person.serial][0] & (1<<j))
            state + j);
     }
    */
/*
void FamilyLikelihoodIBD::ListStates(Person & person, IntArray & states)
{
   states.Dimension(0);
   for(int i = 0; i < phase[person.serial].Length(); i++){
      int state = phase[person.serial][i];
      if(person.traverse == FixedPerson){
         states.Push(state+3);
      }else{
         states.Push(state);
         states.Push(state+1);
         states.Push(state+2);
         states.Push(state+3);
      }
   }
} */
   // for microsatellite markers
   /*
   buf.Dimension(orderCount*2);
   IntArray buf2(orderCount*2);
   for(int i = 0; i < stateHapCount; i++){
      buf.Zero();
      for(int j = 0; j < orderCount; j++){
         if(binary[i][j] == 0)
            buf[j*2] = 1;
         else buf[j*2+1] = 1;
      }
      for(int k = 0; k < stateHapCount; k++){
         buf2 = buf;
         for(int j = 0; j < orderCount; j++){
            if(buf[j*2]==1)
               buf2[j*2+1] = binary[k][j];
            else buf2[j*2] = binary[k][j];
         }
         RecomProb[ConvertBinary(buf2)] += TransmissionHaploidProbability[i];
      }
   }
   */
/*
void FamilyLikelihoodIBD2::ListStates(Person & person, IntArray & states)
{
   states.Dimension(0);
   IntArray tempState(0);
   if(person.markers[positionSNP].isHeterozygous()){
      tempState.Push(1);
      tempState.Push(2);
   }else if(person.markers[positionSNP].isHomozygous()){
      if(person.markers[positionSNP][0]==1) tempState.Push(0);
      else tempState.Push(3);
   }else
      for(int i = 0; i < 4; i++) tempState.Push(i);
   if(phase[person.serial].Length()==0){
      for(int j = 0; j < tempState.Length(); j++){
         if(person.traverse == FixedPerson){
            states.Push((tempState[j]<<2)+3);
         }else{
            states.Push((tempState[j]<<2));
            states.Push((tempState[j]<<2)+1);
            states.Push((tempState[j]<<2)+2);
            states.Push((tempState[j]<<2)+3);
         }
      }
   }else
      for(int i = 0; i < phase[person.serial].Length(); i++){
         int state = phase[person.serial][i]<<2;
         for(int j = 0; j < tempState.Length(); j++){
            if(person.traverse == FixedPerson){
               states.Push(state+(tempState[j]<<2)+3);
            }else{
               states.Push(state+(tempState[j]<<2));
               states.Push(state+(tempState[j]<<2)+1);
               states.Push(state+(tempState[j]<<2)+2);
               states.Push(state+(tempState[j]<<2)+3);
            }
         }
      }
}
*/
/*
   if (person.markers[order[markerOrder]].isKnown()) {
      if (person.markers[order[markerOrder]].Hi() > 2)
         printf("BiallelicLocus::ListStates() can only handle markers with two alleles\n");
      states.Dimension(1);
      states[0] = person.markers[order[markerOrder]][0] + person.markers[order[markerOrder]][1] - 2;
   }else{
      states.Dimension(3);
      states[0] = 0;
      states[1] = 1;
      states[2] = 2;
   }
*/
/*   order[markerOrder] = m;
 stateProb[0] = frequencies[markerOrder] * frequencies[markerOrder];
   stateProb[1] = frequencies[markerOrder] * (1.0 - frequencies[markerOrder]) * 2.0;
   stateProb[2] = (1.0 - frequencies[markerOrder]) * (1.0 - frequencies[markerOrder]);

double FamilyLikelihoodNull::ProbabilityOfTransmission(int state, int maternal_state, int paternal_state)
{
   return stateTransmission[paternal_state * 9 + maternal_state * 3 + state];
}

void FamilyLikelihoodNull::ListStates(Person & person, IntArray & states)
{
   states = IntToVector[genotypeVector[person.serial]];
}
  */
/*
double FamilyLikelihoodHAP::ProbabilityOfState(int state)
{
   return stateProb[haploid[0][state]] * stateProb[haploid[1][state]];
}

double FamilyLikelihoodHAP::ProbabilityOfTransmission(int state, int maternal_state, int paternal_state)
{
   return RecomProb[IsAlleleOne[maternal_state ^ binaryDouble[haploid[0][state]]]]
   * RecomProb[IsAlleleOne[paternal_state ^ binaryDouble[haploid[1][state]]]];
}
*/
/*
double FamilyLikelihoodEff::ProbabilityOfState(int state)
{
   return stateProb[haploid[0][convertState[state]]] * stateProb[haploid[1][convertState[state]]];
}

double FamilyLikelihoodEff::ProbabilityOfTransmission(int state, int maternal_state, int paternal_state)
{
   return RecomProb[IsAlleleOne[convertState[maternal_state] ^ binaryDouble[haploid[0][convertState[state]]]]]
   * RecomProb[IsAlleleOne[convertState[paternal_state] ^ binaryDouble[haploid[1][convertState[state]]]]];
}
*/
/*
void FamilyLikelihoodNull::UpdateFrequency()
{
   stateProb[0] = frequencies[markerOrder] * frequencies[markerOrder];
   stateProb[1] = frequencies[markerOrder] * (1.0 - frequencies[markerOrder]) * 2.0;
   stateProb[2] = (1.0 - frequencies[markerOrder]) * (1.0 - frequencies[markerOrder]);
} */

