// analysis.cpp
// 9/26/2006 Wei-Min Chen

#include "analysis.h"
#include "MathGenMin.h"
#include "MathStats.h"
#include "MapFunction.h"
#include "QuickIndex.h"
#include "Kinship.h"
#include "OLS.h"
#include "haplotype.h"

void AssociationAnalysis::HaploAnalysis()
{
   HaplotypeAnalysis HapEngine(ped);
   HapEngine.ReadData(haplotypeFile);
   
   if(markerList.Length() == 0){
      if(start == _NAN_ && stop == _NAN_)
         error("Please specify SNPs or a region.");
      else{
         ped.SortMarkersInMapOrder(markers, -1);
         int markerStart = -1;
         int markerStop = -1;
         if(start == _NAN_) markerStart = 0;
         if(stop == _NAN_) markerStop = markers.Length()-1;
         if(markerStart == -1)
            for(start *= 0.01, markerStart = 0; (markerStart < markers.Length())
               && (ped.GetMarkerInfo(markers[markerStart])->position < start); markerStart++);
         if(markerStop == -1)
            for(stop *= 0.01, markerStop = markers.Length()-1; (markerStop >= markerStart)
               && (ped.GetMarkerInfo(markers[markerStop])->position > stop); markerStop--);

         if((markerStart >= markers.Length()) || (markerStart > markerStop) )
            error("Please specify SNPs or a region.");
         if(markerStart > 0)
            for(int i = markerStart; i <= markerStop; i++)
               markers[i-markerStart] = markers[i];
         markers.Dimension(markerStop - markerStart + 1);
         markerCount = markers.Length();
         markerList = markers;
      }
   }

   HapEngine.markerList = markerList;
   HapEngine.traits = traits;
   HapEngine.covariates = covariates;
   HapEngine.rareCutoff = rareCutoff;
   HapEngine.ExploreMarker();
   HapEngine.SNPforHAP.Dimension(markerList.Length());
   HapEngine.SNPforHAP.SetSequence();
   HapEngine.PrintSNP();

   HapEngine.loglik.Dimension(traits.Length());
   HapEngine.pvalue.Dimension(traits.Length());
   HapEngine.loglik0.Dimension(traits.Length());

   for(int k = 0; k < traits.Length(); k++)
      HapEngine.regression0(k);

   if(moreFlag) HapEngine.printFlag = 2;
//   int bestHap = -1;
   IntArray bestHap(traits.Length());
   bestHap.Set(-1);
   if(windowSize){
      IntArray *bestSequence = new IntArray[traits.Length()];
      HapEngine.printFlag = 0;
      if(moreFlag) HapEngine.printFlag = 2;
      for(int k = 0; k < traits.Length(); k++){
         double minPvalue = 1;
         for(int s = 1; (s <= windowSize) && (s <= markerList.Length()); s++){
            HapEngine.SNPforHAP.Dimension(s);
            HapEngine.SNPforHAP.SetSequence();
            do{
               if(HapEngine.printFlag) HapEngine.PrintSNP();
               HapEngine.PrepareHap();
               if(HapEngine.hapCount > 1)
                  HapEngine.regression(k);
               else HapEngine.pvalue[k] = 1;
               if(HapEngine.pvalue[k] < minPvalue){
                  minPvalue = HapEngine.pvalue[k];
                  bestSequence[k] = HapEngine.SNPforHAP;
               }
            }while(Increment(HapEngine.SNPforHAP, markerList.Length()-1));
         }
      }
      HapEngine.printFlag = 1;
      if(moreFlag) HapEngine.printFlag = 2;
      for(int k = 0; k < traits.Length(); k++){
         printf("\n\nBest Haplotype Model for trait %s (window size up to %d)\n",
            (const char*)ped.traitNames[traits[k]], windowSize);
         printf("========================================================\n\n");
         HapEngine.SNPforHAP = bestSequence[k];
         HapEngine.PrintSNP();
         HapEngine.PrepareHap();
         HapEngine.regression(k);

         HapEngine.printFlag = 0;
         HapEngine.pvalue2.Dimension(HapEngine.hapCount);
         double minPvalue = 1.0;
         for(int i = 0; i < HapEngine.hapCount; i++){
            HapEngine.regression1(k, i);
            if(HapEngine.pvalue2[i] < minPvalue) {
               minPvalue = HapEngine.pvalue2[i];
               bestHap[k] = i;
            }
         }
         HapEngine.printFlag = 1;
         if(moreFlag) HapEngine.printFlag = 2;
         HapEngine.regression1(k, bestHap[k]);
         if(writeTransformation)
            for(int p = 0; p < ped.count; p++)
               ped[p].covariates.Push(
                  (HapEngine.map[HapEngine.hapCode[0][HapEngine.PedToHap[p]]] == bestHap[k])
                 +(HapEngine.map[HapEngine.hapCode[1][HapEngine.PedToHap[p]]] == bestHap[k]) );
      }
      delete []bestSequence;
   }else{
      HapEngine.SNPforHAP.Dimension(markerList.Length());
      for(int i = 0; i < markerList.Length(); i++)
         HapEngine.SNPforHAP[i] = i;
      HapEngine.PrepareHap();
      for(int k = 0; k < traits.Length(); k++){
         HapEngine.regression(k);
         double minPvalue = 1.0;

         HapEngine.pvalue2.Dimension(HapEngine.hapCount);
         for(int i = 0; i < HapEngine.hapCount; i++){
            HapEngine.regression1(k, i);
            if(HapEngine.pvalue2[i] < minPvalue){
               minPvalue = HapEngine.pvalue2[i];
               bestHap[k] = i;
            }
         }
         if(writeTransformation)
            for(int p = 0; p < ped.count; p++)
               ped[p].covariates.Push(
                  (HapEngine.map[HapEngine.hapCode[0][HapEngine.PedToHap[p]]] == bestHap[k])
                 +(HapEngine.map[HapEngine.hapCode[1][HapEngine.PedToHap[p]]] == bestHap[k]) );
      }
   }
   if(writeTransformation){
      String datafile, pedfile;
      datafile.Copy("new");
      datafile.Add(prefix);
      datafile.Add(".dat");
      pedfile.Copy("new");
      pedfile.Add(prefix);
      pedfile.Add(".ped");
      if(traits.Length()==1){
         ped.covariateNames.Push("haplo");
         printf("Covariate haplo has been added in data %s and %s.\n",
         (const char*)datafile, (const char*)pedfile);
      }else{
         char temp[80];
         for(int k = 0; k < traits.Length(); k++){
            sprintf(temp, "haplo%d", k+1);
            ped.covariateNames.Push(temp);
            printf("Covariate %s (for %s) has been added in data %s and %s.\n",
               temp, (const char*)ped.traitNames[traits[k]],
               (const char*)datafile, (const char*)pedfile);
         }
      }
      ped.covariateCount += traits.Length();
      ped.WriteDataFile(datafile);
      ped.WritePedigreeFile(pedfile);
//      printf("Haplotypes have been added in data %s and %s.\n",
//         (const char*)datafile, (const char*)pedfile);
   }
}

AssociationAnalysis::AssociationAnalysis(Pedigree & pedigree):ped(pedigree)
{
   mean = 0;
   variance = 1;
   heritability = 0;
   freq = 0.3; // estimate by average
   trait = 0;
   Approach = SCORE1;
   markerCount = ped.markerCount;
//   markerFrequencies.Dimension(markerCount);
   pheno.Dimension(0);
//   pvalueLessThan = 1.0;
   pvalueLessThan = 0.0001;

   ToLOD = 0.2171472409516;

   analyticalPower = 0;
   IntervalMapping = 0;
   covariates.Dimension(0);
   MaxLikEngine = NULL;
   es = esPerson = NULL;
   debugPrint = 0;
   normalization = 0;
   printFreq = 0;
   GenoLik = 0;

   DifferenceInMissing = 0.15;
   toleranceVariance = _NAN_;
   WriteFile = 0;
   traits.Dimension(1);
   traits[0] = 0;
   start = stop = _NAN_;
   missingThreshold = 0.09999;

   performance_twoStage = 0;
   denseMap = NULL;
   debugFlag = 0;
   IBDcalculation = 0;
   ibd = NULL;

   GEEpoly = NULL;
   GEEvc = NULL;
   GEEassoc = NULL;
   IBDEngine = NULL;
   performance_fast = 1;
   writeTransformation = 0;
   noInfer = 0;
   GE = NULL;
   
   pvalueArray = NULL;
   pvaluePosition = NULL;
   statArray = NULL;
   alleleArray = NULL;
//   pvalueSummary = 0.2;
   permutationCount = 0;
   multivariateFlag = bivariateFlag = 0;
   IndexR = NULL;

   permutationMatrix.Dimension(0, 0);
//   index = NULL;
   scanFlag = 0;
//   next_chromosome = -1;
//   first_chromosome = last_chromosome = 1;
   no_chromosome = 0;
   many_chromosomes = 0;
   sequentialFlag = false;
   multiPoly = NULL;
   balanceFlag = false;
   windowSize = 0;
   moreFlag = false;
   rareCutoff = 0.01;
}

AssociationAnalysis::~AssociationAnalysis()
{
   if(MaxLikEngine) delete MaxLikEngine;
   if(es) delete []es;
   if(esPerson) delete []esPerson;
   if(denseMap) delete []denseMap;
   if(ibd) delete []ibd;
   if(GEEpoly) delete GEEpoly;
   if(GEEvc) delete GEEvc;
   if(multiPoly) delete multiPoly;
   if(GEEassoc) delete GEEassoc;
   if(IBDEngine) delete IBDEngine;
   if(GE) delete GE;

   if(statArray) delete []statArray;
   if(pvalueArray) delete []pvalueArray;
   if(pvaluePosition) delete []pvaluePosition;
   if(alleleArray) delete []alleleArray;
   if(IndexR) delete []IndexR;
}

void AssociationAnalysis::WritePermutation()
{
   if(permutationMatrix.cols==0) {// no permutations have been done
      printf("Permutation parameter is ignored.\n");
      return;
   }
   String filename;
   filename.catprintf("%s-pv.dat", (const char*)prefix);
   FILE *fp = fopen(filename, "wb");
   if(multivariateFlag)
      fprintf(fp, "Multivariate\n");
   else{
      for(int k = 0; k < traits.Length(); k++)
         fprintf(fp, "%s\t", (const char*)ped.traitNames[traits[k]]);
      fprintf(fp, "\n");
   }
   for(int j = 0; j < permutationMatrix.cols; j++){
      for(int i = 0; i < permutationMatrix.rows; i++)
         fprintf(fp, "%.3G\t", permutationMatrix[i][j]);
      fprintf(fp, "\n");
   }
   fclose(fp);
   printf("\nP-values from %d permuted data are saved in %s.\n",
      permutationCount, (const char*)filename);
}

void AssociationAnalysis::vcLinkage(int m)
{
   if(performance_fast && performance_twoStage && IntervalMapping);
   else{
      IBDEngine->positionSNP = markers[m];
      IBDEngine->CopyPhase(&flankingEngine);
      if(!performance_twoStage)
         for(int i = 0; i < ped.count; i++)
            IBDEngine->genotypeArray[i] = flankingEngine.genotypeArray[i];
      else
      for(int i = 0; i < ped.count; i++)
         IBDEngine->genotypeArray[i][0] = GE->genotypeArray[i][m];
   }

   IBDEngine->init(&flankingEngine);
   IBDEngine->SelectLocus();
   MultipointIBD();

   // GEEvc here.
   for(int i = 0; i < traits.Length(); i++){
      GEEvc->trait = traits[i];
      if(GEEvc->ibd==NULL) GEEvc->ibd = new Vector[ped.familyCount];
      for(int f = 0; f < ped.familyCount; f++)
         GEEvc->ibd[f] = ibd[f];
      GEEvc->variances.Dimension(3);
      GEEvc->variances[0] = variances[i] * (1-heritabilities[i]);
      GEEvc->variances[1] = variances[i] * heritabilities[i];
      GEEvc->variances[2] = 0;
      GEEvc->coef = means[i];
      GEEvc->AtBorder = 0;
      GEEvc->solve();
      if(GEEvc->AtBorder) GEEvc->solve();
      VCloglik[i] = GEEvc->loglik;
      double tempLOD = (VCloglik[i] - POLYloglik[i]) / log(10.0);
      if(tempLOD < 1E-100){
         tempLOD = 0;
         VCmeans[i] = means[i];
         VCvariances[i] = variances[i];
         VCheritabilities[i] = heritabilities[i];
         if(performance_fast && performance_twoStage && IntervalMapping){
            VCheritabilities1[i] = 0;
            VCLOD[i][sparseMap[m]] = tempLOD;
         }else{
            VCheritabilities2[i][m] = 0;
            VCLOD[i][m] = tempLOD;
         }
         VCloglik[i] = POLYloglik[i];
      }else{
         VCmeans[i] = GEEvc->coef;
         VCvariances[i] = GEEvc->totalVariance;
         VCheritabilities[i] = GEEvc->H2;
         if(performance_fast && performance_twoStage && IntervalMapping){
            VCheritabilities1[i] = GEEvc->h2;
            VCLOD[i][sparseMap[m]] = tempLOD;
         }else{
            VCheritabilities2[i][m] = GEEvc->h2;
            VCLOD[i][m] = tempLOD;
         }
      }
   }
}

void AssociationAnalysis::PhaseMarker(int sparseIndex)
{
   if(flankingEngine.phase==NULL) flankingEngine.phase = new IntArray[ped.count];
   flankingEngine.family = ped.families[0];

   for(int i = 0; i < ped.count; i++)
      flankingEngine.phase[i].Dimension(0);

   if(!IntervalMapping) {
      flankingEngine.frequencies.Dimension(0);
      flankingEngine.order.Dimension(0);
      flankingEngine.recombinations.Dimension(0);
      flankingEngine.markerOrder = 0;
      return;
   }

   if(performance_twoStage==0){
      DifferenceInMissing = -0.1;
      SelectLocus(sparseIndex, &flankingEngine);
      int count = flankingEngine.order.Length();
      for(int i = flankingEngine.markerOrder; i < count-1; i++){
         flankingEngine.frequencies[i] = flankingEngine.frequencies[i+1];
         flankingEngine.order[i] = flankingEngine.order[i+1];
         flankingEngine.recombinations[i] = flankingEngine.recombinations[i+1];
      }
      if(flankingEngine.markerOrder>0)
         flankingEngine.recombinations[flankingEngine.markerOrder] =
         DistanceToRecombination((ped.GetMarkerInfo(flankingEngine.order[flankingEngine.markerOrder])->position
            - ped.GetMarkerInfo(flankingEngine.order[flankingEngine.markerOrder-1])->position));
      else flankingEngine.recombinations[flankingEngine.markerOrder] = 0.5;

      flankingEngine.frequencies.Dimension(count-1);
      flankingEngine.order.Dimension(count-1);
      flankingEngine.recombinations.Dimension(count-1);
   }else{
      flankingEngine.frequencies.Dimension(flankingMap[1][sparseIndex]-flankingMap[0][sparseIndex]+1);
      flankingEngine.order.Dimension(flankingMap[1][sparseIndex]-flankingMap[0][sparseIndex]+1);
      flankingEngine.recombinations.Dimension(flankingMap[1][sparseIndex]-flankingMap[0][sparseIndex]+1);

      for(int i = 0; i < flankingEngine.order.Length(); i++){
         flankingEngine.order[i] = markers[sparseMap[flankingMap[0][sparseIndex]+i]];
         flankingEngine.frequencies[i] = ped.GetMarkerInfo(markers[sparseMap[flankingMap[0][sparseIndex]+i]])->freq[1];
         if(i > 0)
            flankingEngine.recombinations[i] = DistanceToRecombination(
               (ped.GetMarkerInfo(flankingEngine.order[i])->position
               - ped.GetMarkerInfo(flankingEngine.order[i-1])->position));
      }
      flankingEngine.markerOrder = sparseIndex - flankingMap[0][sparseIndex];

   int orderCount = flankingEngine.order.Length();
   IntArray *OneState = new IntArray[orderCount];
   IntArray base(orderCount);

   int temp;
   for(int p = 0; p < ped.count; p++){
      for(int i = 0; i < orderCount; i++)
         OneState[i].Dimension(0);
      for(int i = 0; i < orderCount; i++){
         temp = GE->genotypeArray[p][sparseMap[flankingMap[0][sparseIndex]+i]];
         for(int j = 0; j < 4; j++)
            if(temp & (1<<j))
               OneState[i].Push(j);
      }
      base[0] = OneState[0].Length();
      for(int i = 1; i < orderCount; i++)
         base[i] = base[i-1] * OneState[i].Length();
      for(int i = 0; i < base[orderCount-1]; i++){
         temp = OneState[0][i%base[0]];
         for(int j = 1; j < orderCount; j++)
            temp |= (OneState[j][(i%base[j])/base[j-1]]<<(2*j));
         flankingEngine.phase[p].Push(temp<<2);
      }
   }
   delete []OneState;
   }
}

void AssociationAnalysis::BuildFlankingMap()
{
   sparseMap.Dimension(0);
   for(int i = 0; i < markerCount; i++)
      if(markerMissing[i] < missingThreshold)
         sparseMap.Push(i);
   if(denseMap) delete[]denseMap;
   denseMap = new IntArray[sparseMap.Length()+1];
   for(int i = 0; i < sparseMap.Length()+1; i++) denseMap[i].Dimension(0);

   int interval = 0;
   for(int i = 0; i < markerCount; i++){
      if(markerMissing[i] < missingThreshold)
         interval ++;
      else if(markerMissing[i] < .999999999999)
         denseMap[interval].Push(i);
   }

   for(int i = 0; i < 2; i++)
      flankingMap[i].Dimension(sparseMap.Length()+1);
   int left = flankingMap[0][0] = 0;
   int right = flankingMap[1][0] = (IntervalMapping <= sparseMap.Length()) ?
      IntervalMapping-1: sparseMap.Length()-1;
   for(int i = 0; i < sparseMap.Length(); i++){
      double pos = ped.GetMarkerInfo(markers[sparseMap[i]])->position;
      if(i+1 < sparseMap.Length()) pos = (pos + ped.GetMarkerInfo(markers[sparseMap[i+1]])->position)/2;
      while(right + 1 < sparseMap.Length() &&
         fabs(ped.GetMarkerInfo(markers[sparseMap[right + 1]])->position - pos) <
            fabs(pos - ped.GetMarkerInfo(markers[sparseMap[left]])->position)) {
            left++;
            right++;
      }
      if(IntervalMapping>3){
         if((left>=i) && (i>0)){
            left--;
            right--;
         }else if((right<=i+1) && (i+2<sparseMap.Length()) ){
            left++;
            right++;
         }
      }
      flankingMap[0][i+1] = left;
      flankingMap[1][i+1] = right;
   }
   int sum = 0;
   for(int i = 0; i < sparseMap.Length()+1; i++) sum += denseMap[i].Length();

   printf("Missing genotypes are being inferred at %d loci, using %d of %d markers with missing rate < %3.2f...\n",
      sum, IntervalMapping, sparseMap.Length(), missingThreshold);
   printf("\n");
}

void AssociationAnalysis::SetupGlobals()
{
   peakStatistic.Set(-1.0);
   peakPvalue.Set(1.0);
   peakPosition.Set(0);

   if(start == _NAN_ && stop == _NAN_)
      markerCount = markers.Length();
   else{
      int markerStart = -1;
      int markerStop = -1;
      if(start == _NAN_) markerStart = 0;
      if(stop == _NAN_) markerStop = markers.Length()-1;
      if(markerStart == -1)
         for(start *= 0.01, markerStart = 0; (markerStart < markers.Length())
            && (ped.GetMarkerInfo(markers[markerStart])->position < start); markerStart++);
      if(markerStop == -1)
         for(stop *= 0.01, markerStop = markers.Length()-1; (markerStop >= markerStart)
            && (ped.GetMarkerInfo(markers[markerStop])->position > stop); markerStop--);
      if((markerStart >= markers.Length()) || (markerStart > markerStop) ){
         markers.Dimension(0);
         return;
      }
      if(markerStart > 0)
         for(int i = markerStart; i <= markerStop; i++)
            markers[i-markerStart] = markers[i];
      markers.Dimension(markerStop - markerStart + 1);
      markerCount = markers.Length();
   }
   if(IntervalMapping){
      flankingCount.Dimension(markerCount);
      flankingCount.Zero();
   }
   if(markerList.Length()){
      IntArray tempList(0);
      for(int i = 0; i < markerList.Length(); i++){
         int pos = 0;
         for(; (pos < markers.Length()) && (markers[pos] < markerList[i]); pos++);
         if(pos < markers.Length()) tempList.Push(pos);
      }
      markerList = tempList;
   }
}

void AssociationAnalysis::pre_genome()
{
   if(normalization)
      for(trait = 0; trait < traits.Length(); trait++)
         InvNorm(trait);

   peakStatistic.Dimension(traits.Length());
   peakPvalue.Dimension(traits.Length());
   peakPosition.Dimension(traits.Length());

   if(writeTransformation){
      String datafile, pedfile;
      datafile.Copy("new");
      datafile.Add(prefix);
      datafile.Add(".dat");
      pedfile.Copy("new");
      pedfile.Add(prefix);
      pedfile.Add(".ped");
      ped.WriteDataFile(datafile);
      ped.WritePedigreeFile(pedfile);
      printf("Transformed phenotypes are saved in %s and %s.\n",
         (const char*)datafile, (const char*)pedfile);
      no_chromosome = 1;
   }
}

void AssociationAnalysis::post_genome()
{}

void AssociationAnalysis::Analyze()
{
   if(GE == NULL) GE = new GenotypeElimination(ped);
   IntArray temp(0);
   for(int i = 0; i < markers.Length(); i++)
      temp.Push(markers[i]);
   GE->Run(ped, temp);

   GetTraverse();
   EstimateFrequency();

   if(debugFlag) {exit(0);}

   if(IntervalMapping && performance_twoStage) BuildFlankingMap();
   if(MaxLikEngine) delete MaxLikEngine;
   if(WriteFile){
      if(IntervalMapping && performance_twoStage)
         MaxLikEngine = new FamilyLikelihoodEff;
      else MaxLikEngine = new FamilyLikelihoodHAP;
      GenotypeInfer();
      return;
   }

   if(Approach == MLE1) {
      MaxLikEngine = new BiallelicLocus;
      EstimateMLE();
   }else{
      if(IntervalMapping){
         if(performance_twoStage)
            MaxLikEngine = new FamilyLikelihoodEff;
         else
           MaxLikEngine = new FamilyLikelihoodHAP;
      }else
         MaxLikEngine = new FamilyLikelihoodNull;
   }
   MaxLikEngine->trait = 0;
}

void AssociationAnalysis::MultiPoly()
{
   if(multiPoly==NULL) {
      if(balanceFlag){
         if(bivariateFlag || traits.Length()>=21)
            multiPoly = new POLY_Bivariate_Balance(ped);
         else
            multiPoly = new POLY_Multivariate_Balance(ped);
      }
      else multiPoly = new POLY_Multivariate(ped);
   }
   multiPoly->mCovariate = covariates;
   if( (!bivariateFlag) && (traits.Length()<21)){
      multiPoly->mTrait = traits;
      multiPoly->solve();
      multiPoly->print();
   }else{   // bivariate analysis for all pairs
      int traitCount = traits.Length();
      POLY **Model = new POLY *[traitCount];
      for(int r = 0; r < traitCount; r++){
         Model[r] = new POLY(ped);
         Model[r]->Epsilon = 0.0001;
         Model[r]->trait = traits[r];
         Model[r]->mCovariate = covariates;
         Model[r]->solve();
      }
      IntArray bTrait(2);
      Vector RhoG(0);
      Vector RhoE(0);
      Vector Rho(0);
      Vector H1(0);
      Vector H2(0);
      IntArray I(0);
      IntArray J(0);
      multiPoly->Model = new POLY *[2];
      multiPoly->ModelPreset = 1;
      for(int i = 0; i < traitCount; i++)
         for(int j = i+1; j < traitCount; j++){
            bTrait[0] = traits[i];
            bTrait[1] = traits[j];
            multiPoly->mTrait = bTrait;
            multiPoly->Model[0] = Model[i];
            multiPoly->Model[1] = Model[j];
            multiPoly->solve();
            multiPoly->print();
            RhoG.Push(multiPoly->Rho_g[0]);
            RhoE.Push(multiPoly->Rho_e[0]);
            Rho.Push(multiPoly->Rho[0]);
            H1.Push(multiPoly->H2[0]);
            H2.Push(multiPoly->H2[1]);
            I.Push(i);
            J.Push(j);
         }
      for(int r = 0; r < traitCount; r++) delete Model[r];
      printf("\nSummary of Bivariate Analysis\n");
      for(int i = 0; i < 75; i++) printf("=");
      printf("\n%15s%15s%9s%9s%9s%9s%9s\n",
         "Trait1", "Trait2", "RhoG", "RhoE", "Rho", "H2_T1", "H2_T2");
      for(int i = 0; i < traits.Length()*(traits.Length()-1)/2; i++)
            printf("%15s%15s%9.3lf%9.3lf%9.3lf%8.1lf%%%8.1lf%%\n",
               (const char*)ped.traitNames[traits[I[i]]],
               (const char*)ped.traitNames[traits[J[i]]],
               RhoG[i], RhoE[i], Rho[i], H1[i]*100, H2[i]*100);
   }
}

void AssociationAnalysis::polygenic()
{
   means.Dimension(traits.Length(), ped.covariateCount+1);
   variances.Dimension(traits.Length());
   heritabilities.Dimension(traits.Length());
   SampleSize.Dimension(traits.Length());
   POLYloglik.Dimension(traits.Length());
   if(GEEpoly==NULL) GEEpoly = new POLY(ped);
   GEEpoly->mCovariate = covariates;

   for(int i = 0; i < traits.Length(); i++){
      GEEpoly->Epsilon = 1E-7;
      GEEpoly->trait = traits[i];
      GEEpoly->solve();
      double oldLoglik = GEEpoly->loglik;
      double oldScale = GEEpoly->deltaScale;
      if(GEEpoly->AtBorder){
         GEEpoly->deltaScale *= 0.5;
         GEEpoly->solve();
         if(GEEpoly->loglik < oldLoglik - 0.001){
            GEEpoly->deltaScale = oldScale;
            GEEpoly->AtBorder = 0;
            GEEpoly->solve();
         }
         GEEpoly->deltaScale = oldScale;
         GEEpoly->AtBorder = 0;
      }
      means[i] = GEEpoly->coef;
      variances[i] = GEEpoly->totalVariance;
      heritabilities[i] = GEEpoly->H2;
      POLYloglik[i] = GEEpoly->loglik;
      SampleSize[i] = GEEpoly->ValidPersons;
   }
   PrintPolygenic();
}

void AssociationAnalysis::PrintPolygenic()
{
   printf("\nFitted Polygenic Models\n");
   for(int i = 0; i < 48 + 9*covariates.Length(); i++) printf("=");
   printf("\n");
   printf("%15s%9s%9s%6s%9s", "Trait", "Herit", "Variance", "N", "Mean");
   for(int i = 0; i < covariates.Length(); i++)
      printf("%9s", (const char*)ped.covariateNames[covariates[i]]);
   printf("\n");
   for(trait = 0; trait < traits.Length(); trait++){
      printf("%15s%8.1f%%%9.3f", (const char*)ped.traitNames[traits[trait]],
         heritabilities[trait]*100, variances[trait]);
      printf("%6d", SampleSize[trait]);
      for(int i = 0; i < means[trait].Length(); i++)
         printf("%9.3f", means[trait][i]);
      printf("\n");
   }
}

void AssociationAnalysis::MultipointIBD()
{
   if(ibd == NULL) {
      ibd = new Vector[ped.familyCount];
      for(int f = 0; f < ped.familyCount; f++)
         ibd[f].Dimension(ped.families[f]->count * ped.families[f]->count);
   }

   for(int f = 0; f < ped.familyCount; f++){
      int count = ped.families[f]->count;
      IBDEngine->es = es[f];
      for(int i = 0; i < count; i++){
         IBDEngine->FixedPerson = i;
         IBDEngine->SelectFamily(ped.families[f]);
         IBDEngine->score();
         ibd[f][i*count+i] = 1;
         for(int j = i+1; j < count; j++)
            ibd[f][i*count+j] = ibd[f][j*count+i] = IBDEngine->AlleleCount[j] / 2;
      }
   }
}

int ancestorsGenotyped(Person &p, int m)
{
   if(p.isFounder()) return 0;                 
   if(p.father->markers[m].isKnown() && p.mother->markers[m].isKnown()) return 1;
   if(p.father->markers[m].isKnown() && ancestorsGenotyped(*p.mother, m) ) return 1;
   if(p.mother->markers[m].isKnown() && ancestorsGenotyped(*p.father, m) ) return 1;
   return 0;
}

void AssociationAnalysis::GenotypeInfer()
{
   if(performance_twoStage && IntervalMapping){
      for(int k = 0; k < sparseMap.Length()+1; k++){
         double missing = 0;
         if(denseMap[k].Length()==0) continue;
            PhaseMarker(k);
            MaxLikEngine->CopyPhase(&flankingEngine);
         for(int u = 0; u < denseMap[k].Length(); u++){
            int m = denseMap[k][u];
//            if(markerMissing[m]==1) continue;
            if(ped.GetMarkerInfo(markers[m])->freq.Length()>3) continue;
            MaxLikEngine->positionSNP = markers[m];
            MaxLikEngine->init(&flankingEngine);
            MaxLikEngine->frequencies = flankingEngine.frequencies;
            MaxLikEngine->frequencies.Dimension(flankingEngine.frequencies.Length()+1);
            MaxLikEngine->frequencies[MaxLikEngine->markerOrder] = ped.GetMarkerInfo(markers[m])->freq[1];
            for(int i = MaxLikEngine->markerOrder+1; i < flankingEngine.orderCount+1; i++)
               MaxLikEngine->frequencies[i] = flankingEngine.frequencies[i-1];
            MaxLikEngine->SelectLocus();

            for(int f = 0; f < ped.familyCount; f++){
               double tolerance = toleranceVariance;
               if(tolerance == _NAN_){
                  double freq = ped.GetMarkerInfo(markers[m])->freq[1];
                  if(freq < 0.5) freq = 1-freq;
                  tolerance = 1/(1+(1-freq)*0.5/freq) + 0.0001;
               }else
                  for(int i = 1; i < ped.GetMarkerInfo(markers[m])->freq.Length(); i++)
                     if(ped.GetMarkerInfo(markers[m])->freq[i] > tolerance)
                        tolerance = ped.GetMarkerInfo(markers[m])->freq[i];
               int foundersAllGenotyped = 1;
               for(int i = 0; i < ped.families[f]->founders; i++)
                  if(!ped[ped.families[f]->path[i]].markers[markers[m]].isKnown()){
                     foundersAllGenotyped = 0;
                     break;
               }
               MaxLikEngine->es = es[f];
               MaxLikEngine->SelectFamily(ped.families[f]);
               MaxLikEngine->SavedLikelihood = MaxLikEngine->score();
               MaxLikEngine->GetVariance();
               for(int i = 0; i < ped.families[f]->count; i++){
                  if(toleranceVariance == _NAN_){
                     if(foundersAllGenotyped) tolerance = 0.5001;
                     else if(ancestorsGenotyped(ped[ped.families[f]->path[i]], markers[m]))
                        tolerance = 0.5001;
                  }
                  if(MaxLikEngine->AlleleDistribution[0][i] > tolerance)
                     ped[ped.families[f]->path[i]].markers[markers[m]][0] =
                        ped[ped.families[f]->path[i]].markers[markers[m]][1] = 1;
                  else if(MaxLikEngine->AlleleDistribution[2][i] > tolerance)
                     ped[ped.families[f]->path[i]].markers[markers[m]][0] =
                        ped[ped.families[f]->path[i]].markers[markers[m]][1] = 2;
                  else if(MaxLikEngine->AlleleDistribution[1][i] > tolerance){
                     ped[ped.families[f]->path[i]].markers[markers[m]][0] = 1;
                     ped[ped.families[f]->path[i]].markers[markers[m]][1] = 2;
                  }else; // cannot be inferred
               }
            }
         if(debugPrint){
            missing = (ped.count - missing) / ped.count;
            printf("Marker %s at %f cM has missing rate: %.3f\n",
            (const char*)ped.GetMarkerInfo(markers[m])->name,
            ped.GetMarkerInfo(markers[m])->position*100,
            missing);
         }}}
   }else
   for(int m = 0; m < markerCount; m++){
      double missing = 0;

      if(ped.GetMarkerInfo(markers[m])->freq.Length()>3) continue;
//      if(markerMissing[m]>.999) continue;
      SelectLocus(m, MaxLikEngine);

      for(int f = 0; f < ped.familyCount; f++){
         double tolerance = toleranceVariance;
         if(tolerance == _NAN_){
            double freq = ped.GetMarkerInfo(markers[m])->freq[1];
            if(freq < 0.5) freq = 1-freq;
            tolerance = 1/(1+(1-freq)*0.5/freq)+0.0001;
         }else
            for(int i = 1; i < ped.GetMarkerInfo(markers[m])->freq.Length(); i++)
               if(ped.GetMarkerInfo(markers[m])->freq[i] > tolerance)
                  tolerance = ped.GetMarkerInfo(markers[m])->freq[i];
         int foundersAllGenotyped = 1;
         for(int i = 0; i < ped.families[f]->founders; i++)
            if(!ped[ped.families[f]->path[i]].markers[markers[m]].isKnown()){
               foundersAllGenotyped = 0;
               break;
            }
         MaxLikEngine->es = es[f];
         MaxLikEngine->SelectFamily(ped.families[f]);
         MaxLikEngine->SavedLikelihood = MaxLikEngine->score();
         MaxLikEngine->GetVariance();
         for(int i = 0; i < ped.families[f]->count; i++){
            if(toleranceVariance == _NAN_){
               if(foundersAllGenotyped) tolerance = 0.5001;
               else if(ancestorsGenotyped(ped[ped.families[f]->path[i]], markers[m]))
                  tolerance = 0.5001;
            }
            if(MaxLikEngine->AlleleDistribution[0][i] > tolerance)
               ped[ped.families[f]->path[i]].markers[markers[m]][0] =
                  ped[ped.families[f]->path[i]].markers[markers[m]][1] = 1;
               else if(MaxLikEngine->AlleleDistribution[2][i] > tolerance)
                  ped[ped.families[f]->path[i]].markers[markers[m]][0] =
                  ped[ped.families[f]->path[i]].markers[markers[m]][1] = 2;
               else if(MaxLikEngine->AlleleDistribution[1][i] > tolerance){
                  ped[ped.families[f]->path[i]].markers[markers[m]][0] = 1;
                  ped[ped.families[f]->path[i]].markers[markers[m]][1] = 2;
               }else; // cannot be inferred
         }
      }

      if(debugPrint){
         missing = (ped.count - missing) / ped.count;
         printf("Marker %s at %f cM has missing rate: %.3f\n",
         (const char*)ped.GetMarkerInfo(markers[m])->name,
         ped.GetMarkerInfo(markers[m])->position*100,
         missing);
      }
   }
   String datfile;
   String pedfile;
   String temp = prefix;
   datfile.Copy(temp.Add("-infer.dat"));
   pedfile.Copy(prefix.Add("-infer.ped"));
   ped.WriteDataFile(datfile);
   ped.WritePedigreeFile(pedfile);
   printf("Inferred genotypes are saved in %s and %s.\n",
      (const char*)datfile, (const char*)pedfile);
}

void AssociationAnalysis::InvNorm(int tr)
{
   IntArray individuals(0);
   Vector myTraits(0);
   QuickIndex idx;
   for (int i = 0; i < ped.count; i++){
      int missing = 0;
      for(int k = 0; k < covariates.Length(); k++)
         if(!ped[i].isControlled(covariates[k])){
            missing = 1;
            break;
         }
      if(!missing && ped[i].isPhenotyped(traits[tr])){
         individuals.Push(i);
         myTraits.Push(ped[i].traits[traits[tr]]);
      }
   }
   idx.Index(myTraits);
   for(int i = 0; i < individuals.Length();){
      int start = i, end = i + 1;
      while(end < individuals.Length() &&
            ped[individuals[idx[end]]].traits[traits[tr]] ==
            ped[individuals[idx[start]]].traits[traits[tr]])
         end++;
      end --;
      double q = ninv((start + (end-start)/2.0 + 0.5)/individuals.Length());
      for(int j = start; j <= end; j++)
         ped[individuals[idx[j]]].traits[traits[tr]] = q;
      i = end + 1;
   }
}

void AssociationAnalysis::GetTraverse()
{
   if(es) delete []es;
   es = new ES_TRAVERSE[ped.familyCount];
   for(int f = 0; f < ped.familyCount; f++)
      es[f].GetOrder(ped.families[f]);
   if(debugPrint)
      es[0].PrintOrder(ped.families[0]);
}

void AssociationAnalysis::SelectLocus(int m, FamilyLikelihoodNull * Engine)
{
   IntArray fm(0);
   double current = ped.GetMarkerInfo(markers[m])->position;
   int left = m;
   int right = m;
   int leftStop = 1;
   int rightStop = 1;
   int direction = 0;
   for(int i = m-1; (i >= 0) && (current - ped.GetMarkerInfo(markers[i])->position < 0.1); i--)
      if(markerMissing[i] < markerMissing[m] - DifferenceInMissing) {
         left = i;
         leftStop = 0;
         break;
      }
   for(int i = m+1; (i < markerCount) && (ped.GetMarkerInfo(markers[i])->position - current < 0.1); i++)
      if(markerMissing[i] < markerMissing[m] - DifferenceInMissing) {
         right = i;
         rightStop = 0;
         break;
      }
   for(;(fm.Length()<IntervalMapping)&&(leftStop+rightStop<2);){
      if(leftStop){
         fm.Push(right-m);
         for(int i = right+1; (i < markerCount) && (fm.Length()<IntervalMapping)
            && (ped.GetMarkerInfo(markers[i])->position - current < 0.1); i++)
            if(markerMissing[i] < markerMissing[m] - DifferenceInMissing)
               fm.Push(i-m);
         break;
      }else if(rightStop){
         fm.Push(left-m);
         for(int i = left-1; (i >= 0) && (fm.Length()<IntervalMapping)
            && (current - ped.GetMarkerInfo(markers[i])->position < 0.1); i--)
            if(markerMissing[i] < markerMissing[m] - DifferenceInMissing)
               fm.Push(i-m);
         break;
      }else{
         if(direction == 2){
            rightStop = 1;
            fm.Push(right-m);
            for(int i = right+1; (i < markerCount) &&
               (ped.GetMarkerInfo(markers[i])->position - current < 0.1); i++)
               if(markerMissing[i] < markerMissing[m] - DifferenceInMissing) {
                  right = i;
                  rightStop = 0;
                  break;
               }
         }else if(direction == 1){
            leftStop = 1;
            fm.Push(left-m);
            for(int i = left-1; (i >= 0) &&
               (current - ped.GetMarkerInfo(markers[i])->position < 0.1); i--)
               if(markerMissing[i] < markerMissing[m] - DifferenceInMissing){
                  left = i;
                  leftStop = 0;
                  break;
               }
         }else if(direction==0){
            if(IntervalMapping == 1) {
               fm.Push(m-left < right-m? left-m: right-m);
               break;
            }else{
               leftStop = rightStop = 1;
               fm.Push(right-m);
               fm.Push(left-m);
               for(int i = right+1; (i < markerCount) &&
                  (ped.GetMarkerInfo(markers[i])->position - current < 0.1); i++)
                  if(markerMissing[i] < markerMissing[m] - DifferenceInMissing) {
                     right = i;
                     rightStop = 0;
                     break;
                  }
               for(int i = left-1; (i >= 0) &&
                  (current - ped.GetMarkerInfo(markers[i])->position < 0.1); i--)
                  if(markerMissing[i] < markerMissing[m] - DifferenceInMissing){
                     left = i;
                     leftStop = 0;
                     break;
                  }
            }
         }
         direction = (current-ped.GetMarkerInfo(markers[left])->position <
            ped.GetMarkerInfo(markers[right])->position-current) ? 1: 2;
      }
   }
   fm.Sort();
   Engine->frequencies.Dimension(fm.Length()+1);
   Engine->order.Dimension(fm.Length()+1);
   Engine->recombinations.Dimension(fm.Length()+1);
   for(Engine->markerOrder = 0; (Engine->markerOrder < fm.Length()) &&
      (fm[Engine->markerOrder] < 0); Engine->markerOrder++);
   for(int i = 0; i < Engine->markerOrder; i++){
      Engine->order[i] = markers[m + fm[i]];
      Engine->frequencies[i] = ped.GetMarkerInfo(markers[m+fm[i]])->freq[1];
   }
   Engine->order[Engine->markerOrder] = markers[m];
   Engine->frequencies[Engine->markerOrder] = ped.GetMarkerInfo(markers[m])->freq[1];
   for(int i = Engine->markerOrder+1; i < Engine->order.Length(); i++){
      Engine->order[i] = markers[m + fm[i-1]];
      Engine->frequencies[i] = ped.GetMarkerInfo(markers[m+fm[i-1]])->freq[1];
   }
   for(int i = 0; i < fm.Length(); i++)
      Engine->recombinations[i+1] = DistanceToRecombination(
         (ped.GetMarkerInfo(Engine->order[i+1])->position
            - ped.GetMarkerInfo(Engine->order[i])->position));
   Engine->SelectLocus();
   if(Engine->genotypeArray==NULL)
      Engine->genotypeArray = new IntArray[ped.count];
   for(int i = 0; i < ped.count; i++){
      Engine->genotypeArray[i].Dimension(GE->genotypeArray[i].Length());
      for(int j = 0; j < Engine->markerOrder; j++)
        Engine->genotypeArray[i][j] = GE->genotypeArray[i][m+fm[j]];
      Engine->genotypeArray[i][Engine->markerOrder] = GE->genotypeArray[i][m];
      for(int j = Engine->markerOrder+1; j < Engine->order.Length(); j++)
         Engine->genotypeArray[i][j] = GE->genotypeArray[i][m+fm[j-1]];
     }
   flankingCount[m] = fm.Length();
//   fm.Print();
}


void AssociationAnalysis::EstimateFrequency()
{
   LikelihoodFrequencySolver solver;
   GeneralMinimizer *minimizer = new AmoebaMinimizer();
//   GeneralMinimizer *minimizer = new PowellMinimizer();//SAMinimizer();
   solver.ped = &ped;
   if(solver.es) delete []solver.es;
   solver.es = new ES_TRAVERSE[ped.familyCount];
   for(int f = 0; f < ped.familyCount; f++)
      solver.es[f] = es[f];
   solver.MaxLikEngine.trait = traits[trait];
   minimizer->func = &solver;
   if(printFreq) printf("Allele frequency estimate ...\n");
   solver.MaxLikEngine.frequencies.Dimension(markerCount);
   markerMissing.Dimension(markerCount);
   markerMissing.Set(0.0);
   solver.MaxLikEngine.genotypeVector.Dimension(ped.count);
   for(int m = 0; m < markerCount; m++){
      int largestAllele = 0;
      for(int i = 0; i < ped.count; i++)
         if (ped[i].markers[markers[m]].Hi() > largestAllele)
            largestAllele = ped[i].markers[markers[m]].Hi();
      if(largestAllele != 2){
         if(largestAllele == 1)
            printf("Only one allele is found at marker %s.\n",
               (const char*)ped.GetMarkerInfo(markers[m])->name);
         else
            printf("Genotype inconsistency is found at marker %s.\n",
               (const char*)ped.GetMarkerInfo(markers[m])->name);
         markerMissing[m] = 1.0;
         continue;
      }
      ped.GetMarkerInfo(markers[m])->freq.Dimension(largestAllele+1);
      for(int i = 0; i < ped.count; i++)
         if(!ped[i].markers[markers[m]].isKnown()) markerMissing[m] ++;
      markerMissing[m] /= ped.count;
      if(noInfer == 0)
      for(int f = 0; f < ped.familyCount; f++)
         for(int i = ped.families[f]->founders; i < ped.families[f]->count; i++)
            if( (!ped[ped.families[f]->path[i]].markers[markers[m]].isKnown()) &&
               ped[ped.families[f]->path[i]].father->markers[markers[m]].isHomozygous() &&
               ped[ped.families[f]->path[i]].mother->markers[markers[m]].isHomozygous() ) {
               ped[ped.families[f]->path[i]].markers[markers[m]][0] =
                  ped[ped.families[f]->path[i]].mother->markers[markers[m]][0];
               ped[ped.families[f]->path[i]].markers[markers[m]][1] =
                  ped[ped.families[f]->path[i]].father->markers[markers[m]][0];
            }
//      printf("%5.2f\t", ped.GetMarkerInfo(markers[m])->position*100);
      solver.MaxLikEngine.order[0] = markers[m];
      solver.MaxLikEngine.SelectLocus();
      int MendelianError = 0;
      for(int i = 0; i < ped.count; i++){
         if(GE->genotypeArray[i][m] > 15) {MendelianError = 1; break;}
         solver.MaxLikEngine.genotypeVector[i] = GE->genotypeArray[i][m];
//         printf("%d ", GE->genotypeArray[i][m]);
      }
      if(MendelianError){
         printf("Genotype inconsistency is found at marker %s.\n",
               (const char*)ped.GetMarkerInfo(markers[m])->name);
         markerMissing[m] = 1.0;
         continue;
      }
      minimizer->Reset(1);
      minimizer->point[0] = 0.5;
      minimizer->Minimize(1e-8); // 1e-8
      ped.GetMarkerInfo(markers[m])->freq[1] = minimizer->point[0];
      ped.GetMarkerInfo(markers[m])->freq[2] = 1- minimizer->point[0];
      if(printFreq)
         printf("Frequency of allele 1 at marker %d (%5.1f cM) is %4.3f. Missing rate is %4.3f.\n", m+1,
            ped.GetMarkerInfo(markers[m])->position*100, ped.GetMarkerInfo(markers[m])->freq[1], markerMissing[m]);

      if(GenoLik){
         printf("lnLikelihood at marker %d for %d families = %.2f\n",
            m+1, ped.familyCount, -solver.Evaluate(minimizer->point));
      }
      if(ped.GetMarkerInfo(markers[m])->freq[1]<0.001 || ped.GetMarkerInfo(markers[m])->freq[1]>0.9999){
            printf("Genotype inconsistency (or only one allele) is found at marker %s.\n",
               (const char*)ped.GetMarkerInfo(markers[m])->name);
            markerMissing[m] = 1;
      }
   }

   if(printFreq) printf("\n");
   delete(minimizer);
}

double LikelihoodFrequencySolver::Evaluate(Vector & point)
{
   double sum = 0;
   MaxLikEngine.frequencies[MaxLikEngine.markerOrder] = point[0];
   MaxLikEngine.UpdateFrequency();
   if(MaxLikEngine.frequencies[MaxLikEngine.markerOrder] < 0 ||
      MaxLikEngine.frequencies[MaxLikEngine.markerOrder] > 1) return 1e100;

   for(int f = 0; f < ped->familyCount; f++){
      MaxLikEngine.es = es[f];
      MaxLikEngine.SelectFamily(ped->families[f]);
      double t = MaxLikEngine.LikelihoodOfGenotype();
      if(t <= 0.0) {
         sum = -1e100;
         break;
      }
      sum += log(t);
   }
   return -sum;
}

double LikelihoodSolver::Evaluate(Vector & point)
{
   double sum = 0;
   MaxLikEngine.frequencies[MaxLikEngine.markerOrder] = point[0];
   MaxLikEngine.variance = point[1];
   MaxLikEngine.mu = point[2];
   if(dim>3){
      MaxLikEngine.a = point[3];
      MaxLikEngine.d = point[4];
   }
   MaxLikEngine.UpdateFrequency();
   MaxLikEngine.UpdateSegregation();
   if(MaxLikEngine.frequencies[MaxLikEngine.markerOrder] < 0 ||
      MaxLikEngine.frequencies[MaxLikEngine.markerOrder] > 1) return 1e200;
   if(MaxLikEngine.variance < 0) return 1e200;

   for(int f = 0; f < ped->familyCount; f++){
      MaxLikEngine.es = es[f];
      MaxLikEngine.SelectFamily(ped->families[f]);
      double t = MaxLikEngine.Likelihood();
      if(t <= 1e-200) {
         sum = -1e200;
         break;
      }
      sum += log(t);
   }
   return -sum;
}

void AssociationAnalysis::EstimateMLE()
{
   LikelihoodSolver solver;
   GeneralMinimizer *minimizer = new AmoebaMinimizer();
   solver.ped = &ped;
   if(solver.es) delete []solver.es;
   solver.es = new ES_TRAVERSE[ped.familyCount];
   for(int f = 0; f < ped.familyCount; f++)
      solver.es[f] = es[f];

   minimizer->func = &solver;
   statistic.Dimension(traits.Length(), markerCount);

   for(trait = 0; trait < traits.Length(); trait++){
      solver.MaxLikEngine.trait = traits[trait];
      for(int m = 0; m < markerCount; m++){
         SelectLocus(m, &solver.MaxLikEngine);
         minimizer->Reset(3);
         solver.dim = 3;
         solver.MaxLikEngine.a = solver.MaxLikEngine.d = 0;
//         minimizer->point[0] = markerFrequencies[m];
         minimizer->point[0] = ped.GetMarkerInfo(markers[m])->freq[1];
         minimizer->point[1] = 1;
         minimizer->point[2] = 0;
         minimizer->Minimize(1e-8);
         double loglik0 = solver.Evaluate(minimizer->point);
         minimizer->Reset(5);
         solver.dim = 5;
         minimizer->point[3] = 0;
         minimizer->point[4] = 0;
         minimizer->Minimize(1e-8);
         double loglik1 = solver.Evaluate(minimizer->point);
         statistic[trait][m] = (loglik0 - loglik1) * 2;
      }
   }
   delete(minimizer);
}

void AssociationAnalysis::PrintScores(/*int trait*/)
{
   if(Approach==OLS) return;
   if(WriteFile) return;
   if(writeTransformation) return;
   printf("\nPedigree-Wide Association Analysis (%s)\n"
          "======================================================\n"
          "%10s%7s%10s%15s", (const char*)ped.traitNames[traits[trait]],

          "Position", "LOD", "pvalue", "Marker");

   printf("\n");
   for(trait = 0; trait < traits.Length(); trait++)
      for(int m = 0; m < markerCount; m++){
         pvalue[trait][m] = chidist(statistic[trait][m], 2);
         double LOD = statistic[trait][m] * 0.2171472409516;
         printf("%10f%7.2f%10.3G%15s\n", ped.GetMarkerInfo(markers[m])->position*100, LOD,
         pvalue[trait][m], (const char*)ped.GetMarkerInfo(markers[m])->name);
      }
   PrintSummary();
}

void AssociationAnalysis::PrintSummary()
{
   printf("\nProportion of significant tests\n");
   printf("=========================================================================\n");
   printf("%15s%8s%8s%8s%8s%8s%8s%8s\n",
      "Trait", "1E-2", "1E-3", "1E-4", "1E-5", "1E-6", "1E-7", "1E-8");

   for(trait = 0; trait < traits.Length(); trait++){
      Vector Level(0);
      double temp = 0.1;
      for(int i = 0; i < 7; i++){
         temp *= .1;
         if(temp < pvalueLessThan)
            Level.Push((pvalueArray[trait].Length()- pvalueArray[trait].CountIfGreater(temp)) / (double)markers.Length());
         else
            Level.Push(_NAN_);
      }
      printf("%15s", (const char*)ped.traitNames[traits[trait]]);
      for(int i = 0; i < Level.Length(); i++)
         if(Level[i]==_NAN_)
            printf("%8s", "");
         else
            printf("%8.2G", Level[i]);
      printf("\n");
   }

   printf("\nSummary of GWA results\n");
   printf("==================================================================\n");
   printf("%15s%10s%10s%15s%15s\n",
      "Trait_Name", "P-Value", "Chrom", "Position", "SNP_Name");
   for(trait = 0; trait < traits.Length(); trait++){
         int m = peakPosition[trait];
         printf("%15s%10.3G%10d%15f%15s\n", (const char*)ped.traitNames[traits[trait]],
         peakPvalue[trait], ped.GetMarkerInfo(markers[m])->chromosome, ped.GetMarkerInfo(markers[m])->position*100,
               (const char*)ped.GetMarkerInfo(markers[m])->name);
   }
}

/*      double min = 99;
      int t = -1;
      for(int i = 0; i < pvalueArray[trait].Length(); i++)
         if(pvalueArray[trait][i] < min){
            t = i;
            min = pvalueArray[trait][i];
         }
      if(min < pvalueSummary){
         t = pvaluePosition[trait][t];
         printf("%15s%10.3G%10d%15f%15s\n", (const char*)ped.traitNames[traits[trait]],
         min, ped.GetMarkerInfo(markers[t])->chromosome, ped.GetMarkerInfo(markers[t])->position*100,
               (const char*)ped.GetMarkerInfo(markers[t])->name);
      }else{
         printf("%15s%10s%10s%15s%15s\n", (const char*)ped.traitNames[traits[trait]],
            "NA", "NA", "NA", "NA");
      }
*/

/*
void AssociationAnalysis::PrintSummary()
{
   printf("\nProportion of significant tests\n");
   printf("=========================================================================\n");
   printf("%15s%8s%8s%8s%8s%8s%8s%8s\n",
      "Trait", "1E-2", "1E-3", "1E-4", "1E-5", "1E-6", "1E-7", "1E-8");
   for(trait = 0; trait < traits.Length(); trait++){
      Vector Level(0);
      double temp = 0.1;
      for(int i = 0; i < 7; i++){
         temp /= 10.0;
         Level.Push(1 - (double)pvalue[trait].CountIfGreater(temp) / pvalue[trait].CountIfGreater(-0.5));
      }
      printf("%15s", (const char*)ped.traitNames[traits[trait]]);
      for(int i = 0; i < Level.Length(); i++)
         printf("%8.2G", Level[i]);
      printf("\n");
   }

   printf("\nSummary of GWA results\n");
   printf("==================================================================\n");
   printf("%15s%10s%10s%15s%15s\n",
      "Trait_Name", "P-Value", "Chrom", "Location", "SNP_Name");
   for(trait = 0; trait < traits.Length(); trait++){
      double min = 99;
      int t = -1;
      for(int i = 0; i < pvalue[trait].Length(); i++)
         if(pvalue[trait][i] < min){
            t = i;
            min = pvalue[trait][i];
         }
      printf("%15s%10.3G%10d%15f%15s\n", (const char*)ped.traitNames[traits[trait]],
      min, ped.GetMarkerInfo(markers[t])->chromosome, ped.GetMarkerInfo(markers[t])->position*100,
            (const char*)ped.GetMarkerInfo(markers[t])->name);
   }
}
*/

               /*
                  if(MaxLikEngine->AlleleVariance[i] < toleranceVariance){
                     missing++;
                     double score = MaxLikEngine->AlleleCount[i];
                     if(score < 0.5)
                        ped[ped.families[f]->path[i]].markers[markers[m]][0] =
                           ped[ped.families[f]->path[i]].markers[markers[m]][1] = 1;
                     else if(score > 1.5)
                        ped[ped.families[f]->path[i]].markers[markers[m]][0] =
                           ped[ped.families[f]->path[i]].markers[markers[m]][1] = 2;
                     else{
                        ped[ped.families[f]->path[i]].markers[markers[m]][0] = 1;
                        ped[ped.families[f]->path[i]].markers[markers[m]][1] = 2;
                     }
                  }*/
/*
void AssociationAnalysis::PhaseMarker(int sparseIndex)
{
   if(flankingEngine.phase==NULL) flankingEngine.phase = new IntArray[ped.count];
   for(int i = 0; i < ped.count; i++)
      flankingEngine.phase[i].Dimension(0);

   if(!IntervalMapping) {
      flankingEngine.frequencies.Dimension(0);
      flankingEngine.order.Dimension(0);
      flankingEngine.recombinations.Dimension(0);
      flankingEngine.markerOrder = 0;
      flankingEngine.family = ped.families[0];
      return;
   }

   if(flankingEngine.esPerson==NULL) flankingEngine.esPerson = new ES_TRAVERSE[ped.count];
   for(int i = 0; i < ped.count; i++)
      flankingEngine.esPerson[i] = esPerson[i];

   if(performance_twoStage==0){
      DifferenceInMissing = -0.1;
//      flankingEngine.markerOrder = sparseIndex;
      SelectLocus(sparseIndex, &flankingEngine);
      int count = flankingEngine.order.Length();
      for(int i = flankingEngine.markerOrder; i < count-1; i++){
         flankingEngine.frequencies[i] = flankingEngine.frequencies[i+1];
         flankingEngine.order[i] = flankingEngine.order[i+1];
         flankingEngine.recombinations[i] = flankingEngine.recombinations[i+1];
      }
      if(flankingEngine.markerOrder>0)
         flankingEngine.recombinations[flankingEngine.markerOrder] =
         DistanceToRecombination((ped.GetMarkerInfo(flankingEngine.order[flankingEngine.markerOrder])->position
            - ped.GetMarkerInfo(flankingEngine.order[flankingEngine.markerOrder-1])->position));
      else flankingEngine.recombinations[flankingEngine.markerOrder] = 0.5;

      flankingEngine.frequencies.Dimension(count-1);
      flankingEngine.order.Dimension(count-1);
      flankingEngine.recombinations.Dimension(count-1);

   }else{
      flankingEngine.frequencies.Dimension(flankingMap[1][sparseIndex]-flankingMap[0][sparseIndex]+1);
      flankingEngine.order.Dimension(flankingMap[1][sparseIndex]-flankingMap[0][sparseIndex]+1);
      flankingEngine.recombinations.Dimension(flankingMap[1][sparseIndex]-flankingMap[0][sparseIndex]+1);

      for(int i = 0; i < flankingEngine.order.Length(); i++){
         flankingEngine.order[i] = markers[sparseMap[flankingMap[0][sparseIndex]+i]];
         flankingEngine.frequencies[i] = ped.GetMarkerInfo(markers[sparseMap[flankingMap[0][sparseIndex]+i]])->freq[1];
         if(i > 0)
            flankingEngine.recombinations[i] = DistanceToRecombination(
               (ped.GetMarkerInfo(flankingEngine.order[i])->position
               - ped.GetMarkerInfo(flankingEngine.order[i-1])->position));
      }
      flankingEngine.markerOrder = sparseIndex - flankingMap[0][sparseIndex];
   }
   flankingEngine.SelectLocus();
   for(int f = 0; f < ped.familyCount; f++){
      flankingEngine.es = es[f];
      flankingEngine.SelectFamily(ped.families[f]);
      for(int p = 0; p < ped.families[f]->count; p++){
         flankingEngine.lastPerson = p;
         ES_TRAVERSE *pES = & flankingEngine.esPerson[ped.families[f]->path[p]];
         flankingEngine.ResetESP(ped.families[f]->path[p]);
         flankingEngine.LikelihoodOfGenotype();
         Vector temp = pES->Items[pES->From[pES->Type.Length()-1]];
         double sum = temp.Sum();
         temp.Multiply(1/sum);
         for(int i = 0; i < temp.Length(); i++)
//            if(temp[i] > .0000099)
            if(temp[i] > 1E-20)
               flankingEngine.phase[ped.families[f]->path[p]].Push(pES->States[pES->From[pES->Type.Length()-1]][i]);
         }
      }
   int orderCount = flankingEngine.order.Length();
   IntArray *OneState = new IntArray[orderCount];
   IntArray base(orderCount);

   int temp;
   IntArray states;
   for(int p = 0; p < ped.count; p++){
      for(int i = 0; i < orderCount; i++)
         OneState[i].Dimension(0);
      for(int i = 0; i < orderCount; i++){
         temp = GE->genotypeArray[p][sparseMap[flankingMap[0][sparseIndex]+i]];
         for(int j = 0; j < 4; j++)
            if(temp & (1<<j))
               OneState[i].Push(j);
      }
      states.Dimension(0);
      base[0] = OneState[0].Length();
      for(int i = 1; i < orderCount; i++)
         base[i] = base[i-1] * OneState[i].Length();
      for(int i = 0; i < base[orderCount-1]; i++){
         temp = OneState[0][i%base[0]];
         for(int j = 1; j < orderCount; j++)
            temp |= (OneState[j][(i%base[j])/base[j-1]]<<(2*j));
         states.Push(temp);
      }
      if(p < 5){
         flankingEngine.phase[p].Print();
         states.Print();
      }
//      flankingEngine.phase[p] = states;
   }
   delete []OneState;

}
*/

         /*
         for(int i = 0; i < ped.count; i++){
            if(ped[i].isPhenotyped(traits[trait]) && ped[i].markers[m].isKnown() && ped[i].isFounder()) {
               Y.Push(ped[i].traits[traits[trait]]);
               X.Push(ped[i].markers[m].countAlleles(1));
            }
         } */
/*      printf("======================================================\n"
          "%10s%7s%10s%15s",

          "Position", "t", "pvalue", "Marker");

      if(isFamilyData) printf("%5s", "N");

      printf("\n");
      */

/*         length = X.Length();
         if(length<3) continue;
         meanX = X.Average();
         meanY = Y.Average();
         Sxx = X.SumSquares() - meanX * meanX * length;
         Syy = Y.SumSquares() - meanY * meanY * length;
         if(Sxx<=0) continue;
         Sxy = X.InnerProduct(Y) - meanX * meanY * length;
         beta = Sxy / Sxx;
         X.Multiply(-beta);
         Y.Add(-meanY + beta * meanX);
         Y.Add(X);
         v = sqrt(Y.InnerProduct(Y) / (length - 2) / Sxx);
         stat = beta / v;
         R2 = beta * Sxy /Syy;
         apvalue = tdist(fabs(stat), length - 2); // df = length-2
*/
/*      if(Approach == OLS) {
         if(markerList.Length()){
            IntArray testList(0);
            for(int i = 0; i < markerList.Length(); i++){
               int k = ped.markerNames.Find(markerList[i]);
               if(k==-1)
                  printf("Marker %s cannot be found.\n", (const char*)markerList[i]);
               else
                  testList.Push(k);
            }
            if(testList.Length()==0){
               printf("No markers are specified.");
               markerCount = 0;
            }
         }

         WriteUnrelated();
         return;
      }
      */
/*
   if(Approach == OLS) {
      if(markerList.Length()==0){
         if(traitList.Length()) scanFlag = 1;
         if(scanFlag){
            if(multivariateFlag)
               OLSmultivariate();
            else
               OLSunivariate();
         }else OLSregression();
      }else{
         IntArray testList(0);
         for(int i = 0; i < markerList.Length(); i++){
            int k = ped.markerNames.Find(markerList[i]);
            if(k==-1)
               printf("Marker %s cannot be found.\n", (const char*)markerList[i]);
            else
               testList.Push(k);
         }
         if(testList.Length()==0){
            printf("No markers are specified.");
            markerCount = 0;
            return;
         }
         else
            OLSregression(testList);
      }
     if( last_chromosome && permutationCount)
        WritePermutation();
      return;
   }
*/




/*
void AssociationAnalysis::HaplotypeAnalysis()
{
   FILE *fp = fopen((const char*)haplotypeFile, "rb");
   if(fp==NULL) error("File %s cannot be opened.", (const char*)haplotypeFile);

   if(markerList.Length() == 0){
      if(start == _NAN_ && stop == _NAN_)
         error("Please specify SNPs or a region.");
      else{
         ped.SortMarkersInMapOrder(markers, -1);
         int markerStart = -1;
         int markerStop = -1;
         if(start == _NAN_) markerStart = 0;
         if(stop == _NAN_) markerStop = markers.Length()-1;
         if(markerStart == -1)
            for(start *= 0.01, markerStart = 0; (markerStart < markers.Length())
               && (ped.GetMarkerInfo(markers[markerStart])->position < start); markerStart++);
         if(markerStop == -1)
            for(stop *= 0.01, markerStop = markers.Length()-1; (markerStop >= markerStart)
               && (ped.GetMarkerInfo(markers[markerStop])->position > stop); markerStop--);

         if((markerStart >= markers.Length()) || (markerStart > markerStop) )
            error("Please specify SNPs or a region.");
         if(markerStart > 0)
            for(int i = markerStart; i <= markerStop; i++)
               markers[i-markerStart] = markers[i];
         markers.Dimension(markerStop - markerStart + 1);
         markerCount = markers.Length();
         markerList = markers;
      }
   }

   IntArray HapToPed, PedToHap;

   String line;
   StringArray tokens;
   StringArray Haplotype[2];
   StringArray tID;
   StringArray famID(0);
   StringArray pID(0);
   for(int i = 0; i < 2; i++) Haplotype[i].Dimension(0);
   while(!feof(fp)){
      line.ReadLine(fp);
      tokens.ReplaceColumns(line, ' ');
      if(tokens.Length()>2){
         Haplotype[tokens[1]=="HAPLO2"].Push(tokens[2]);
         tID.ReplaceTokens(tokens[0], "->");
         famID.Push(tID[0]);
         pID.Push(tID[1]);
      }
   }
   fclose(fp);

   int count = Haplotype[0].Length();

   HapToPed.Dimension(count);
   PedToHap.Dimension(ped.count);
   HapToPed.Set(-1);
   PedToHap.Set(-1);
   for(int i = 0; i < count; i++)
      for(int j = 0; j < ped.count; j++)
         if((ped[j].famid == famID[i*2]) && (ped[j].pid == pID[i*2])){
            HapToPed[i] = j;
            PedToHap[j] = i;
         }




   StringIntHash hash;
   IntArray hapCode[2];
   for(int i = 0; i < 2; i++) hapCode[i].Dimension(count);
   int hapCount = 0;
   String shortHaplotype;
   StringArray haplotypeList(0);
   for(int p = 0; p < count; p++)
      for(int h = 0; h < 2; h++){
         shortHaplotype = "";
         for(int i = 0; i < markerList.Length(); i++)
            shortHaplotype += Haplotype[h][p][markerList[i]];
         hapCode[h][p] = hash.Integer(shortHaplotype);
         if(hapCode[h][p]==-1) {
            hapCode[h][p] = hapCount;
            haplotypeList.Push(shortHaplotype);
            hash.SetInteger(shortHaplotype, hapCount++);
         }
      }

   Vector hapFreq(hapCount);
   hapFreq.Zero();
   for(int h = 0; h < 2; h++)
      for(int p = 0; p < count; p++)
         hapFreq[hapCode[h][p]] ++;
   for(int i = 0; i < hapCount; i++)
      hapFreq[i] /= (count*2);

   QuickIndex index;
   index.Index(hapFreq);

   Vector oldfreq, freq;
   IntArray map(hapCount);
   IntArray oldmap(hapCount);
   for(int i = 0; i < hapCount; i++) map[i] = i;
   for(oldfreq = freq = hapFreq, oldmap = map; oldfreq.Min() < rareCutoff; oldfreq = freq, oldmap = map){
      index.Index(oldfreq);
      freq.Dimension(0);
//      int newRank = oldfreq.Length() - 2;
      for(int i = oldfreq.Length()-1; i > 0; i--){
         freq.Push(oldfreq[index[i]]);
         for(int j = 0; j < hapCount; j++)
            if(oldmap[j]==index[i])
               map[j] = oldfreq.Length()-1-i;
      }
      freq[freq.Length()-1] += oldfreq[index[0]];
      for(int j = 0; j < hapCount; j++)
         if(oldmap[j] == index[0])
            map[j] = oldfreq.Length()-2;
   }

   IntArray *oldCodes = new IntArray[freq.Length()];
   for(int k = 0; k < freq.Length(); k++){
      oldCodes[k].Dimension(0);
      for(int i = 0; i < hapCount; i++){
         if(map[i]==k) oldCodes[k].Push(i);
      }
   }

   // allele freq
   Vector alleleFreq(markerList.Length());
   IntArray minorAllele(markerList.Length());
   minorAllele.Set(2);

   for(int i = 0; i < markerList.Length(); i++){
      double sum = 0;
      int count = 0;
      for(int p = 0; p < ped.count; p++){
         if(ped[p].markers[markerList[i]].isKnown()){
            sum += ped[p].markers[markerList[i]].countAlleles(2);
            count += 2;
         }
      }
      sum /= count;
      if(sum < 0.5)
         alleleFreq[i] = sum;
      else{
         alleleFreq[i] = 1 - sum;
         minorAllele[i] = 1;
      }
   }

   Matrix freq2(markerList.Length(), markerList.Length());
   freq2.Zero();
   for(int i = 0; i < markerList.Length(); i++)
      for(int j = i+1; j < markerList.Length(); j++){
         for(int k = 0; k < 2; k++)
            for(int p = 0; p < Haplotype[0].Length(); p++)
               if( (Haplotype[k][p][markerList[i]] == ped.markerInfo[markerList[i]]->alleleLabels[minorAllele[i]][0])
                  && (Haplotype[k][p][markerList[j]] == ped.markerInfo[markerList[j]]->alleleLabels[minorAllele[j]][0]) )
                  freq2[i][j] ++;
         freq2[i][j] /= (Haplotype[0].Length()*2);
//         printf("%lf\t%lf\n", freq2[i][j], alleleFreq[i]*alleleFreq[j]);
      }

   Matrix D, Rsquare;
   D.Dimension(markerList.Length(), markerList.Length());
   Rsquare.Dimension(markerList.Length(), markerList.Length());
   for(int i = 0; i < markerList.Length(); i++)
      for(int j = i+1; j < markerList.Length(); j++){
         double t1 = (1-alleleFreq[i])*alleleFreq[j];
         double t2 = alleleFreq[i]*(1-alleleFreq[j]);
         D[i][j] = freq2[i][j] - alleleFreq[i] * alleleFreq[j];
         Rsquare[i][j] = D[i][j] * D[i][j] / (t1 * t2);
         D[i][j] /= (t1<t2 ? t1: t2);
      }
   printf("%15s%10s%7s%7s%7s%7s\n", "MarkerName", "Position", "Allele", "Freq", "(R2)", "(D')");
   for(int i = 0; i < markerList.Length(); i++){
      printf("%15s%10lf%7s%7.3lf", (const char*)ped.markerNames[markerList[i]],
         ped.markerInfo[markerList[i]]->position*100,
         (const char*)ped.markerInfo[markerList[i]]->alleleLabels[minorAllele[i]], alleleFreq[i]);
      for(int j = 0; j < markerList.Length(); j++)
         if(i < j)   // upper
            printf("%7.3lf", D[i][j]);
         else if(i > j) // lower
            printf("%7.3lf", Rsquare[j][i]);
         else printf("%7s", "");
      printf("\n");
   }
   printf("\n");

   int printLength = markerList.Length() > 9 ? markerList.Length()+2 : 10;
   char format1[80], format2[80], format3[80];
   sprintf(format1, "%s%d%s", "%5s%10s%", printLength, "s%10s\n");
   sprintf(format2, "%s%d%s", "%5d%10.3lf%", printLength, "s%10.3lf\n");
   sprintf(format3, "%s%d%s", "%5s%10s%", printLength, "s%10.3lf\n");
   printf(format1, "Code", "Frequency", "Haplotype","Freqs");

   for(int i = 0; i < freq.Length(); i++){
      printf(format2, i+1, freq[i], (const char*)haplotypeList[oldCodes[i][0]], hapFreq[oldCodes[i][0]]);
      for(int j = 1; j < oldCodes[i].Length(); j++)
         printf(format3, "", "", (const char*)haplotypeList[oldCodes[i][j]], hapFreq[oldCodes[i][j]]);
   }
   delete []oldCodes;

   // Regression of trait values on haplotypes
   OLS_REGRESSION ols;
   Vector Y;
   Matrix X;

   for(int k = 0; k < traits.Length(); k++){
      IntArray pheno(0);
      for(int i = 0; i < ped.count; i++){
         if(!ped[i].isPhenotyped(traits[k])) continue;
         int valid = 1;
         for(int c = 0; c < covariates.Length(); c++)
            if(!ped[i].isControlled(covariates[c])){
               valid = 0;
               break;
            }
         if(valid) pheno.Push(i);
      }

      Y.Dimension(pheno.Length());
      X.Dimension(pheno.Length(), covariates.Length()+1);
      X.Zero();

      printf("\nSingle-SNP Analysis for Trait %s (N=%d)\n",
         (const char*)ped.traitNames[traits[k]], pheno.Length());
      printf("============================================\n");
      printf("%15s%10s%7s%7s%7s%10s%10s%7s\n",
         "MarkerName", "Position", "Allele", "Freq",
         "t", "pvalue", "Effect", "h2");
      for(int m = 0; m < markerList.Length(); m++){
         for(int i = 0; i < pheno.Length(); i++){
            Y[i] = ped[pheno[i]].traits[traits[k]];
            for(int c = 0; c < covariates.Length(); c++)
               X[i][c] = ped[pheno[i]].covariates[covariates[c]];
            int m1 = (Haplotype[0][PedToHap[pheno[i]]][markerList[m]]==
               ped.markerInfo[markerList[m]]->alleleLabels[minorAllele[m]][0]);
            int m2 = (Haplotype[1][PedToHap[pheno[i]]][markerList[m]]==
               ped.markerInfo[markerList[m]]->alleleLabels[minorAllele[m]][0]);
            if((m1==m2) && (m1==1))
               X[i][covariates.Length()] = 2;
            else if( (m1==1) || (m2==1) )
                  X[i][covariates.Length()] = 1;
            else X[i][covariates.Length()] = 0;
         }
         ols.run(Y, X);
         printf("%15s%10lf%7s%7.3lf%7.2lf%10.2G%10.2G%6.1lf%%\n",
            (const char*)ped.markerNames[markerList[m]],
            ped.markerInfo[markerList[m]]->position*100,
            (const char*)ped.markerInfo[markerList[m]]->alleleLabels[minorAllele[m]],
            alleleFreq[m], ols.t_statistic[covariates.Length()],
            ols.pvalue[covariates.Length()], ols.beta[covariates.Length()+1],
            ols.R2[covariates.Length()]*100);
      }
//   }
//   for(int k = 0; k < traits.Length(); k++){
      Y.Dimension(pheno.Length());
      X.Dimension(pheno.Length(), covariates.Length()+freq.Length()-1);
      X.Zero();
      for(int i = 0; i < pheno.Length(); i++){
         Y[i] = ped[pheno[i]].traits[traits[k]];
         for(int c = 0; c < covariates.Length(); c++)
            X[i][c] = ped[pheno[i]].covariates[covariates[c]];
         int m1 = hapCode[0][PedToHap[pheno[i]]];
         int m2 = hapCode[1][PedToHap[pheno[i]]];
         if( (m1==m2) && (m1 < freq.Length()-1) )
            X[i][covariates.Length()+m1] = 2;
         else{
            if(m1 < freq.Length()-1)
               X[i][covariates.Length()+m1] = 1;
            if(m2 < freq.Length()-1)
               X[i][covariates.Length()+m2] = 1;
         }
      }
      ols.run(Y, X);
      printf("\nHaplotype Analysis for Trait %s (N=%d)\n",
         (const char*)ped.traitNames[traits[k]], pheno.Length());
      printf("============================================\n");
      printf("%10s%7s%10s%10s%7s\n", "COV", "t", "pvalue", "Effect", "h2");
      for(int c = 0; c < covariates.Length(); c++)
         printf("%10s%7.2f%10.2G%10.2G%6.1f%%\n",
            (const char*)ped.covariateNames[covariates[c]], ols.t_statistic[c],
            ols.pvalue[c], ols.beta[c+1], ols.R2[c]*100);
      for(int m = 0; m < freq.Length()-1; m++)
         printf("      HAP%d%7.2f%10.2G%10.2G%6.1f%%\n",
            m+1, ols.t_statistic[m+covariates.Length()],
            ols.pvalue[m+covariates.Length()], ols.beta[covariates.Length()+m+1],
            ols.R2[m+covariates.Length()]*100);
   }
}
  */

  /*         bool tPrintFlag = HapEngine.printFlag;
         HapEngine.printFlag = 0;
         HapEngine.pvalue2.Dimension(HapEngine.hapCount);
         double minPvalue = 1.0;
         int bestHap = -1;
         for(int i = 0; i < HapEngine.hapCount; i++){
            HapEngine.regression1(k, i);
            if(HapEngine.pvalue2[i] < minPvalue) {
               minPvalue = HapEngine.pvalue2[i];
               bestHap = i;
            }
         }
         HapEngine.printFlag = tPrintFlag;
         HapEngine.regression1(k, bestHap);
      }
*/

