//////////////////////////////////////////////////////////////////////
// multivar.cpp
// Author: Wei-Min Chen
// March 18, 2007


#include "multivar.h"
#include "Kinship.h"
#include "QuickIndex.h"
#include <math.h>

// ************************** Multivariate Analysis ***********************

POLY_Multivariate::POLY_Multivariate(Pedigree & pedigree):GEE(pedigree)
{
   repeat = NULL;
   pheno = NULL;
   RowOfOne = NULL;
   rStart = rEnd = NULL;
   Idx = NULL;
   mCovariate.Dimension(0);
//   for(int i = 0; i < 20; i++) Model[i] = NULL;
   Model = NULL;
   ModelPreset = 0;
}

POLY_Multivariate::~POLY_Multivariate()
{
   if(repeat) delete []repeat;
   if(RowOfOne) delete []RowOfOne;
   if(rStart) delete []rStart;
   if(rEnd) delete []rEnd;
   if(Idx) delete []Idx;
   for(int i = 0; i < 20; i++)
      if(Model[i]) delete Model[i];
}

Vector POLY_Multivariate::residual(int p)
{
   Vector mResidual(rCount);
   mResidual.Set(_NAN_);
   for(int r = 0; r < rCount; r++){
      int base = coefCount/rCount*r;
      if(!ped[p].isPhenotyped(mCovariate[r])) continue;
//      for(int k = 0; k < ped.covariateCount; k++)
//         if(!ped[p].isControlled(k)) continue;
      for(int k = 0; k < mCovariate.Length(); k++)
         if(!ped[p].isControlled(mCovariate[k])) continue;
      double r = ped[p].traits[mTrait[r]] - coef[base];
      for(int i = 1; i < coefCount/rCount; i++)
         r -= coef[base+i] * ped[p].covariates[mCovariate[i-1]];
   }
   return mResidual;
}

int POLY_Multivariate::constraint(void)
{
   int r = 0;
   for(int i = 0; i < 2; i++)
      for(int r = 0; r < rCount; r++)
         if(variances[i*parCount/2 + r] < 0){
            variances[i*parCount/2 + r] = 0;
            r = 1;
         }
   return r;
}

void POLY_Multivariate::GetPhi(int f)
{
   if(isBalanced[f]){
      size = pheno[f].Length();
      count = size / rCount;
      Kinship kin;
      Phi.Dimension(count, count);
      kin.Setup(*ped.families[f]);
      for(int i = 0; i < count; i++)
         for(int j = i; j < count; j++)
            Phi[j][i] = Phi[i][j] = 2 * kin(ped[pheno[f][i]], ped[pheno[f][j]]);
      return;
   }
   Kinship kin;
   size = pheno[f].Length();
   Phi.Dimension(size, size);
   kin.Setup(*ped.families[f]);
   for(int i = 0; i < size; i++)
      for(int j = i; j < size; j++)
         Phi[j][i] = Phi[i][j] = 2 * kin(ped[pheno[f][i]], ped[pheno[f][j]]);
}

void POLY_Multivariate::RefreshO(int f)
{

   Omega.Dimension(size, size);

   if(isBalanced[f]){

      for(int d = 0; d < rCount; d++){
         for(int i = 0; i < count; i++)   // same trait same person
            Omega[count*d+i][count*d+i] = variances[d] + variances[parCount/2+d];
         for(int i = 0; i < count; i++)
            for(int j = i+1; j < count; j++) // same trait diff person
               Omega[count*d+i][count*d+j] = Omega[count*d+j][count*d+i]
                  = Phi[i][j] * variances[parCount/2+d];
      }
      for(int u = 0; u < rCount; u++)
         for(int v = u+1; v < rCount; v++){
            for(int i = 0; i < count; i++)   // diff trait same person
               Omega[count*u+i][count*v+i] = Omega[count*v+i][count*u+i]
                  = variances[Idx[u][v]] + variances[parCount/2 + Idx[u][v]];
            for(int i = 0; i < count; i++)
               for(int j = i+1; j < count; j++) // diff trait diff person
                  Omega[count*u+i][count*v+j] = Omega[count*u+j][count*v+i]
                  = Omega[count*v+j][count*u+i] = Omega[count*v+i][count*u+j]
                  = Phi[i][j] * variances[parCount/2 + Idx[u][v]];
         }
      return;
   }

   // unbalanced pedigree

   for(int i = 0; i < size; i++){

      int r = repeat[f][i];
      Omega[i][i] = variances[r] + variances[parCount/2+r];
      for(int j = i + 1; j < size; j++){
         int par = Idx[r][repeat[f][j]];
         if(pheno[f][i] == pheno[f][j]) // same person
            Omega[i][j] = Omega[j][i] = variances[par] + variances[parCount/2+par];
         else
            Omega[i][j] = Omega[j][i] = Phi[i][j] * variances[parCount/2 + par];
      }
   }
}

void POLY_Multivariate::RefreshOD(int f)
{
   if(isBalanced[f]){
      if(rCount==2){
         M.Product(Phi, Phi);

         M.Multiply(variances[3]*variances[4]-variances[5]*variances[5]);
         M.AddMultiple(variances[0]*variances[4]+variances[1]*variances[3]-2*variances[2]*variances[5], Phi);
         double t = variances[0]*variances[1]-variances[2]*variances[2];
         for(int i = 0; i < count; i++) M[i][i] += t;
         bi_chol.Decompose(M);
         bi_chol.Invert(); // most computationally intensive part
         M.Product(Phi, bi_chol.inv);
         O[0] = O[1] = O[2] = M;
         O[0].Multiply(variances[4]);
         O[0].AddMultiple(variances[1], bi_chol.inv);
         O[1].Multiply(variances[3]);
         O[1].AddMultiple(variances[0], bi_chol.inv);
         O[2].Multiply(-variances[5]);
         O[2].AddMultiple(-variances[2], bi_chol.inv);

// More efficient than for(int i = 0; i < 3; i++) OP[i].Product(O[i], Phi);
         bi_chol.inv.Transpose(M);
         M.Product(Phi, bi_chol.inv);
         OP[0] = OP[1] = OP[2] = M;
         OP[0].Multiply(variances[4]);
         OP[0].AddMultiple(variances[1], bi_chol.inv);
         OP[1].Multiply(variances[3]);
         OP[1].AddMultiple(variances[0], bi_chol.inv);
         OP[2].Multiply(-variances[5]);
         OP[2].AddMultiple(-variances[2], bi_chol.inv);

         for(int i = 0; i < 6; i++) OD[i].Zero();
         for(int i = 0; i < count; i++)
            for(int j = 0; j < count; j++)
               for(int k = 0; k < 2; k++)
                  for(int u = 0; u < 2; u++){
                     OD[u][count*k+i][count*u+j] = O[Idx[k][u]][i][j];
                     OD[parCount/2+u][count*k+i][count*u+j] = OP[Idx[k][u]][i][j];
                     for(int v = u+1; v < 2; v++){
                        OD[Idx[u][v]][count*k+i][count*u+j] = O[Idx[k][v]][i][j];
                        OD[Idx[u][v]][count*k+i][count*v+j] = O[Idx[k][u]][i][j];
                        OD[parCount/2+Idx[u][v]][count*k+i][count*u+j] = OP[Idx[k][v]][i][j];
                        OD[parCount/2+Idx[u][v]][count*k+i][count*v+j] = OP[Idx[k][u]][i][j];
                     }
                  }
               return;

      }

      // balanced pedigree

      CholeskyOmega.Invert(); // most computationally intensive part
      for(int i = 0; i < parCount/2; i++) O[i].Dimension(count, count);
      for(int i = 0; i < count; i++)
         for(int j = 0; j < count; j++)
            for(int u = 0; u < rCount; u++)
               for(int v = u; v < rCount; v++)
                  O[Idx[u][v]][i][j] = CholeskyOmega.inv[count*u+i][count*v+j];
      for(int u = 0; u < parCount/2; u++) // most computationally intensive part
         OP[u].Product(O[u], Phi);
      for(int i = 0; i < parCount; i++) OD[i].Zero();
      for(int i = 0; i < count; i++)
         for(int j = 0; j < count; j++)
            for(int k = 0; k < rCount; k++)
               for(int u = 0; u < rCount; u++){
                  OD[u][count*k+i][count*u+j] = O[Idx[k][u]][i][j];
                  OD[parCount/2+u][count*k+i][count*u+j] = OP[Idx[k][u]][i][j];
                  for(int v = u+1; v < rCount; v++){
                     OD[Idx[u][v]][count*k+i][count*u+j] = O[Idx[k][v]][i][j];
                     OD[Idx[u][v]][count*k+i][count*v+j] = O[Idx[k][u]][i][j];
                     OD[parCount/2+Idx[u][v]][count*k+i][count*u+j] = OP[Idx[k][v]][i][j];
                     OD[parCount/2+Idx[u][v]][count*k+i][count*v+j] = OP[Idx[k][u]][i][j];
                  }
               }
      return;
   }
   // unbalanced pedigree
   double sum;
   int k;
   CholeskyOmega.Invert();
   for(int i = 0; i < parCount; i++) OD[i].Zero();
   for(int i = 0; i < parCount/2-rCount; i++){
      RowOfOne[i].Dimension(size);
      RowOfOne[i].Set(-1);
   }
   for(int i = 0; i < size; i++)
      for(int j = i+1; j < size; j++)
         if(pheno[f][i]==pheno[f][j]){
            int par = Idx[repeat[f][i]][repeat[f][j]] - rCount;
            RowOfOne[par][j] = i;
            RowOfOne[par][i] = j;
         }

   for(int i = 0; i < size; i++)
      for(int r = 0; r < rCount; r++)
         for(int j = rStart[f][r]; j <= rEnd[f][r]; j++){
            OD[r][i][j] = CholeskyOmega.inv[i][j];
            for(sum=0, k = rStart[f][r]; k <= rEnd[f][r]; k++)
               sum += CholeskyOmega.inv[i][k] * Phi[k][j];
            OD[parCount/2+r][i][j] = sum;
         }
   for(int r1 = 0; r1 < rCount; r1++)
      for(int r2 = r1 + 1; r2 < rCount; r2++){
         int par = Idx[r1][r2];
         for(int i = 0; i < size; i++){
            for(int j = rStart[f][r1]; j <= rEnd[f][r1]; j++){
               if(RowOfOne[par-rCount][j] != -1)
                  OD[par][i][j] = CholeskyOmega.inv[i][RowOfOne[par-rCount][j]];
               for(sum=0, k = rStart[f][r2]; k <= rEnd[f][r2]; k++)
                  sum += CholeskyOmega.inv[i][k] * Phi[k][j];
               OD[parCount/2+par][i][j] = sum;
            }
            for(int j = rStart[f][r2]; j <= rEnd[f][r2]; j++){
               if(RowOfOne[par-rCount][j] != -1)
                  OD[par][i][j] = CholeskyOmega.inv[i][RowOfOne[par-rCount][j]];
               for(sum=0, k = rStart[f][r1]; k <= rEnd[f][r1]; k++)
                  sum += CholeskyOmega.inv[i][k] * Phi[k][j];
               OD[parCount/2+par][i][j] = sum;
            }
         }
      }
}

void POLY_Multivariate::InitModel()
{
   if(Model==NULL) {
      Model = new POLY *[rCount];
      for(int r = 0; r < rCount; r++) Model[r] = NULL;
   }
   for(int r = 0; r < rCount; r++){
      if(Model[r]==NULL) Model[r] = new POLY(ped);
      Model[r]->Epsilon = 0.001;
      Model[r]->trait = mTrait[r];
      Model[r]->mCovariate = mCovariate;
      Model[r]->solve();
   }
}

void POLY_Multivariate::InitCoef()
{
   rCount = mTrait.Length();
   parCount = rCount*(rCount+1);
   if(Idx == NULL) Idx = new IntArray[rCount];
   for(int r = 0; r < rCount; r++) Idx[r].Dimension(rCount);
   int par_length = 0;
   for(int i = 0; i < rCount; i++)
      Idx[i][i] = par_length++;
   for(int i = 1; i < rCount; i++)
      for(int j = 0; j < i; j++)
         Idx[i][j] = Idx[j][i] = par_length++;
   if(RowOfOne==NULL)
      RowOfOne = new IntArray[rCount*(rCount-1)/2];
   if(pheno == NULL) pheno = new IntArray[ped.familyCount];
   if(rStart == NULL) rStart = new IntArray[ped.familyCount];
   if(rEnd == NULL) rEnd = new IntArray[ped.familyCount];
   if(repeat == NULL) repeat = new IntArray[ped.familyCount];
   if(traits == NULL) traits = new Vector[ped.familyCount];
   if(covariates == NULL) covariates = new Matrix[ped.familyCount];
   isBalanced.Dimension(ped.familyCount);
   isBalanced.Zero();
   SampleSize.Dimension(rCount);
   SampleSize.Zero();
   variances.Dimension(parCount);
   variances.Zero();
   coef.Dimension(0);
   if(ModelPreset==0) InitModel();
   for(int r = 0; r < rCount; r++){
      variances[r] = Model[r]->variances[0];
      variances[parCount/2+r] = Model[r]->variances[1];
      coef.Stack(Model[r]->coef);
   }
   coefCount = coef.Length();
   for(int f = 0; f < ped.familyCount; f++) {
      rStart[f].Dimension(rCount);
      rEnd[f].Dimension(rCount);
      traits[f] = Model[0]->traits[f];
      repeat[f].Dimension(0);
      for(int i = 0; i < Model[0]->pheno[f].Length(); i++) repeat[f].Push(0);
      rStart[f][0] = 0;
      rEnd[f][0] = Model[0]->pheno[f].Length()-1;
      pheno[f] = Model[0]->pheno[f];
      for(int r = 1; r < rCount; r++){
         pheno[f].Stack(Model[r]->pheno[f]);
         traits[f].Stack(Model[r]->traits[f]);
         for(int i = 0; i < Model[r]->pheno[f].Length(); i++) repeat[f].Push(r);
         rStart[f][r] = rEnd[f][r-1] + 1;
         rEnd[f][r] = rEnd[f][r-1] + Model[r]->pheno[f].Length();
      }
      covariates[f].Dimension(coefCount, pheno[f].Length());
      covariates[f].Zero();

      int row = 0;

      int col = 0;

      for(int r = 0; r < rCount; r++){
         for(int i = 0; i < Model[r]->coef.Length(); i++)
            for(int j = 0; j < Model[r]->pheno[f].Length(); j++)

               covariates[f][row+i][col+j] = Model[r]->covariates[f][i][j];

         row += Model[r]->coef.Length();

         col += Model[r]->pheno[f].Length();

      }
      if(pheno[f].Length() == Model[0]->pheno[f].Length()*rCount){
         isBalanced[f] = 1;
         count = Model[0]->pheno[f].Length();
         for(int i = 0; i < count; i++){
            for(int r = 1; r < rCount; r++)
               if(pheno[f][r*count+i] != pheno[f][i]){
                  isBalanced[f] = 0;
                  break;
               }
            if(isBalanced[f]==0) break;
         }
      }
      for(int r = 0; r < rCount; r++)
         SampleSize[r] += Model[r]->pheno[f].Length();
   }
   double maxV = -1;
   for(int r = 0; r < rCount; r++){
      double v = variances[r] + variances[parCount/2+r];
      if(v > maxV) maxV = v;
   }
   Epsilon = 1E-7 * maxV * maxV;
}

void POLY_Multivariate::summary()
{
   Rho_g.Dimension(rCount*(rCount-1)/2);
   Rho_e.Dimension(rCount*(rCount-1)/2);
   Rho.Dimension(rCount*(rCount-1)/2);
   double t;
   for(int r1 = 0; r1 < rCount; r1++)
      for(int r2 = r1 + 1; r2 < rCount; r2++){
         t = variances[parCount/2+r1]*variances[parCount/2+r2];
         Rho_g[Idx[r1][r2]-rCount] =
            t > 0? variances[parCount/2+Idx[r1][r2]] / sqrt(t) : 0;
         t = variances[r1]*variances[r2];
         Rho_e[Idx[r1][r2]-rCount] =
            t > 0? variances[Idx[r1][r2]] / sqrt(t) : 0;
         t = (variances[r1]+variances[parCount/2+r1])*(variances[r2]+variances[parCount/2+r2]);
         Rho[Idx[r1][r2]-rCount] =
            t > 0 ? (variances[Idx[r1][r2]] + variances[parCount/2+Idx[r1][r2]])
               / sqrt(t) : 0;
      }
   H2.Dimension(rCount);
   for(int r = 0; r < rCount; r++){
      t = variances[parCount/2+r]+variances[r];
      H2[r] = t > 0 ? variances[parCount/2+r] / t : 0;
   }
}

void POLY_Multivariate::print()
{
   printf("\nFitted Multivariate Polygenic Model\n");
   for(int i = 0; i < 48 + 9*mCovariate.Length(); i++) printf("=");
   printf("\n");
   printf("%15s%9s%9s%6s%9s", "Trait", "Herit", "Variance", "N", "Mean");
   for(int i = 0; i < mCovariate.Length(); i++)
      printf("%9s", (const char*)ped.covariateNames[mCovariate[i]]);
   printf("\n");
   for(int r = 0; r < mTrait.Length(); r++){
      printf("%15s%8.1f%%%9.3f", (const char*)ped.traitNames[mTrait[r]],
         H2[r]*100, variances[parCount/2+r] + variances[r]);
      printf("%6d", SampleSize[r]);
      for(int i = 0; i < coefCount/rCount; i++)
         printf("%9.3f", coef[r*coefCount/rCount + i]);
      printf("\n");
   }
   printf("\n");

   printf("%20s|%15s%15s|\n", "", "", "Genetic   ");
   printf("%-20s|%15s%15s|\n", "Correlation Table:", "Environmental", "");
   printf("%20s|%15s%15s|\n", "", "", "Overall   ");
   int count = 15+10*rCount;
   if(count < 52) count = 52;
   for(int i = 0; i < count; i++) printf("=");
   printf("\n");
   printf("%-15s", "TraitName");
   for(int i = 0; i < rCount; i++)
      printf("%10s", (const char*)ped.traitNames[mTrait[i]]);
   printf("%15s\n", "TraitName");
   for(int i = 0; i < rCount; i++){
      printf("%-15s", (const char*)ped.traitNames[mTrait[i]]);
      for(int j = 0; j < rCount; j++){
         if(i==j) printf("%10s", "");
         else if(i<j) printf("%10.3lf", Rho_g[Idx[i][j]-rCount]);
         else printf("%10.3lf", Rho_e[Idx[i][j]-rCount]);
      }
      printf("\n");
   }
   for(int i = 0; i < rCount-1; i++){
      printf("%15s", "");
      for(int j = 0; j < rCount; j++){
         if(i >= j) printf("%10s", "");
         else printf("%10.3lf", Rho[Idx[i][j]-rCount]);
      }
      printf("%15s\n", (const char*)ped.traitNames[mTrait[i]]);
   }
   printf("\n");
}

/*********** IMPORTANT: Do not delete Moment Estimate of Variances ************
   double R1, R2;
   Vector overall;
   for(int r1 = 0; r1 < rCount; r1++)
      for(int r2 = r1+1; r2 < rCount; r2++){
         int par = Idx[r1][r2]-rCount;
         overall.Dimension(0);
         for(int i = 0; i < ped.count; i++){
            R1 = Model[r1]->residual(i);
            if(R1 ==_NAN_) continue;
            R2 = Model[r2]->residual(i);
            if(R2 ==_NAN_) continue;
            overall.Push(R1*R2);
         }
         variances[par+rCount] = overall.Sum() / overall.Length();
      }

   Kinship kin;
   Vector num, den;
   for(int r1 = 0; r1 < rCount; r1++)
      for(int r2 = r1+1; r2 < rCount; r2++){
         int par = Idx[r1][r2]-rCount;
         overall.Dimension(0);
         num.Dimension(0);
         den.Dimension(0);
         for(int f = 0; f < ped.familyCount; f++){
            size = ped.families[f]->last - ped.families[f]->first + 1;
            Matrix Phi;
            Phi.Dimension(size, size);
            kin.Setup(*ped.families[f]);
            for(int i = ped.families[f]->first; i <= ped.families[f]->last; i++)
               for(int j = i; j <= ped.families[f]->last; j++)
                  Phi[i-ped.families[f]->first][j-ped.families[f]->first]
                     = Phi[j-ped.families[f]->first][i-ped.families[f]->first]
                     = 2 * kin(ped[i], ped[j]);
            for(int i = ped.families[f]->first; i <= ped.families[f]->last; i++)
               for(int j = ped.families[f]->first; j <= ped.families[f]->last; j++){
                  if(Phi[i-ped.families[f]->first][j-ped.families[f]->first]==0.0)
                     continue;
                  if(i==j) continue;
                  R1 = Model[r1]->residual(i);
                  if(R1 ==_NAN_) continue;
                  R2 = Model[r2]->residual(j);
                  if(R2 ==_NAN_) continue;
                  num.Push(R1*R2*Phi[i-ped.families[f]->first][j-ped.families[f]->first]);
                  den.Push(Phi[i-ped.families[f]->first][j-ped.families[f]->first]*
                     Phi[i-ped.families[f]->first][j-ped.families[f]->first]);
               }
//            overall.Push(vec.Sum()/vec.Length());
         }
//         variances[parCount/2+Idx[r1][r2]] = overall.Sum() / overall.Length();
         variances[parCount/2+Idx[r1][r2]] = num.Sum() / den.Sum();
         variances[Idx[r1][r2]] -= variances[parCount/2+Idx[r1][r2]];
      }
*/

/*
   printf("\nFitted Polygenic Models\n");
   for(int i = 0; i < 45 + 10*mCovariate.Length(); i++) printf("=");
   printf("\n");
   printf("%15s%10s%10s%10s", "Trait", "Herit", "Variance", "Mean");
   for(int i = 0; i < mCovariate.Length(); i++)
      printf("%10s", (const char*)ped.covariateNames[mCovariate[i]]);
   printf("\n");
   for(int r = 0; r < mTrait.Length(); r++){
      printf("%15s%9.1f%%%10.3f", (const char*)ped.traitNames[mTrait[r]],
         Model[r]->variances[1] / (Model[r]->variances[0]+Model[r]->variances[1]) * 100,
         Model[r]->variances[0]+Model[r]->variances[1]);
      for(int i = 0; i < Model[r]->coef.Length(); i++)
         printf("%10.3f", Model[r]->coef[i]);
      printf("\n");
   }
   printf("\n");
*/


