// unrelated.cpp
// 11/06/2006 Wei-Min Chen

#include "MathGenMin.h"
#include "MathStats.h"
#include "Kinship.h"
#include "analysis.h"
#include "OLS.h"
#include "permutation.h"
#include "unrelated.h"
#include "QuickIndex.h"
#include "MathSVD.h"

void OLS_Association::pre_genome()
{
   if(causalSNPList==NULL){
      SubIndex.Dimension(traits.Length());
      for(int k = 0; k < traits.Length(); k++)
         SubIndex[k] = k;
      causalSNPList = new IntArray[traits.Length()];
      for(int k = 0; k < traits.Length(); k++)
         causalSNPList[k].Dimension(0);
      GetUnrelated();
      if(normalization)
         for(int tr = 0; tr < traits.Length(); tr++)
            InvNorm(tr);
      if(writeTransformation){
         WriteUnrelated();
         no_chromosome = 1;
         return;
      }
      if(markerList.Length()) {
         OLSregression(markerList);
         no_chromosome = 1;
         return;
      }
   }else{
      printf("\nA new round of genome scan ... \n");
   }
   
   peakStatistic.Dimension(SubIndex.Length());
   peakPvalue.Dimension(SubIndex.Length());
   peakPosition.Dimension(SubIndex.Length());
   peakGenomePvalue.Dimension(SubIndex.Length());
   peakGenomePosition.Dimension(SubIndex.Length());
   peakGenomePvalue.Set(1.0);
   peakGenomePosition.Zero();

   if(scanFlag){
      GetResiduals();
      if(permutationCount){
         if(multivariateFlag)
            permutationMatrix.Dimension(1, permutationCount);
         else
            permutationMatrix.Dimension(SubIndex.Length(), permutationCount);

         permutationMatrix.Set(1.0);
      }
      statisticGreaterThan = ninv(pvalueLessThan/2);
      statisticGreaterThan = statisticGreaterThan * statisticGreaterThan;
      if(statArray==NULL) statArray = new Vector[SubIndex.Length()];
      if(pvalueArray==NULL) pvalueArray = new Vector[SubIndex.Length()];
      if(pvaluePosition==NULL) pvaluePosition = new IntArray[SubIndex.Length()];
   }
}

void OLS_Association::post_genome()
{
   if(no_chromosome) return;
   if(permutationCount)
      WritePermutation();
   bool overFlag = true;
   
   if(!multivariateFlag){
      if(many_chromosomes){
         printf("\nSummary of GWA results from all chromosomes\n");
         printf("==================================================================\n");
         printf("%15s%10s%10s%15s%15s\n",
            "Trait_Name", "P-Value", "Chrom", "Position", "SNP_Name");
         for(int k = 0; k < SubIndex.Length(); k++)
            printf("%15s%10.3G%10d%15f%15s\n",
               (const char*)ped.traitNames[traits[SubIndex[k]]],
               peakGenomePvalue[k],
               ped.GetMarkerInfo(peakGenomePosition[k])->chromosome,
               ped.GetMarkerInfo(peakGenomePosition[k])->position*100,
               (const char*)ped.GetMarkerInfo(peakGenomePosition[k])->name);
      }

      IntArray tempArray(0);
      for(int k = 0; k < SubIndex.Length(); k++){
         if(peakGenomePvalue[k] < pvalueLessThan){
            IntArray localList(0);
            for(int m = 0; m < causalSNPList[SubIndex[k]].Length(); m++)
               localList.Push(causalSNPList[SubIndex[k]][m]);
            localList.Push(peakGenomePosition[k]);
            OLSregression(localList, k);
            if(sequentialFlag)
               causalSNPList[SubIndex[k]].Push(peakGenomePosition[k]);
            overFlag = false;
            tempArray.Push(k);
         }
      }
      for(int k = 0; k < tempArray.Length(); k++)
         SubIndex[k] = SubIndex[tempArray[k]];
      SubIndex.Dimension(tempArray.Length());
      if(sequentialFlag && overFlag){
         sequentialFlag = false;
         printf("\nSummary of GWA Sequential Analysis...\n");
         for(int k = 0; k < traits.Length(); k++){
            if(causalSNPList[k].Length()==0) continue;
            SubIndex.Dimension(traits.Length());
            for(int i = 0; i < traits.Length(); i++)
               SubIndex[i] = i;
            OLSregression(causalSNPList[k], k);
         }
      }
   }else{
      if(many_chromosomes){
         printf("\nSummary of GWA results from all chromosomes\n");
         printf("==================================================================\n");
         printf("%10s%10s%15s%15s\n",
            "P-Value", "Chrom", "Position", "SNP_Name");
         printf("%10.3G%10d%15f%15s\n",
               peakGenomePvalue[0],
               ped.GetMarkerInfo(peakGenomePosition[0])->chromosome,
               ped.GetMarkerInfo(peakGenomePosition[0])->position*100,
               (const char*)ped.GetMarkerInfo(peakGenomePosition[0])->name);
      }
      if(peakGenomePvalue[0] < pvalueLessThan){
         overFlag = false;
//         if(sequentialFlag)
            causalSNPList[0].Push(peakGenomePosition[0]);
         MultivariateRegression(causalSNPList[0]);
      }

      if(sequentialFlag && overFlag){
         sequentialFlag = false;
         printf("\nSummary of GWA Sequential Analysis...\n");
         MultivariateRegression(causalSNPList[0]);
      }
   }
}

void OLS_Association::OLSregression(IntArray testList, int tr)
{
   OLS_REGRESSION ols;
   IntArray sample(0);
   for(int i = 0; i < pheno.Length(); i++){
      int valid = 1;
      if(!ped[pheno[i]].isPhenotyped(traits[SubIndex[tr]])) continue;
      for(int m = 0; m < testList.Length(); m++)
         if(!ped[pheno[i]].markers[testList[m]].isKnown()){
            valid = 0;
            break;
         }
      if(valid) sample.Push(i);
   }
   int N = sample.Length();
   Vector Y(N);
   for(int i = 0; i < N; i++)
      Y[i] = ped[pheno[sample[i]]].traits[traits[SubIndex[tr]]];
   Matrix X(N, covariates.Length());
   for(int i = 0; i < N; i++)
      for(int m = 0; m < covariates.Length(); m++)
         X[i][m] = ped[pheno[sample[i]]].covariates[covariates[m]];
   ols.run(Y, X);
   double loglik0 = ols.loglik;

   X.Dimension(N, covariates.Length()+testList.Length());
   for(int i = 0; i < N; i++){
      for(int m = 0; m < covariates.Length(); m++)
         X[i][m] = ped[pheno[sample[i]]].covariates[covariates[m]];
      for(int m = 0; m < testList.Length(); m++)
         X[i][m+covariates.Length()] =
            ped[pheno[sample[i]]].markers[testList[m]].countAlleles(1);
   }

   ols.run(Y, X);
   if(ols.failure)
      printf("Regression model for %s failed due to colinearity.\n",
         (const char*)ped.traitNames[traits[SubIndex[tr]]], N);
   else{
      printf("\nSummary of associations for %s (N=%d)\n",
      (const char*)ped.traitNames[traits[SubIndex[tr]]], N);
      printf("====================================================================\n");
      printf("%15s%7s%12s%7s%10s%7s%8s\n",
         "Covariate", "Chrom", "Position", "t", "pvalue", "Effect", "Eff_adj");
      double effect = 0;
      for(int m = 0; m < covariates.Length(); m++){
         printf("%15s%7s%12s%7.2f%10.2G%6.1f%%%8s\n",
            (const char*)ped.covariateNames[covariates[m]], "", "",
            ols.t_statistic[m+1], ols.pvalue[m+1], ols.R2[m]*100,"");
            effect += ols.R2[m];
      }
      effect = 1 / (1-effect);
      for(int m = covariates.Length();
         m < covariates.Length() + testList.Length(); m++)
         printf("%15s%7d%12f%7.2f%10.2G%6.1f%%%7.1f%%\n",
            (const char*)ped.GetMarkerInfo(testList[m-covariates.Length()])->name,
            ped.GetMarkerInfo(testList[m-covariates.Length()])->chromosome,
            ped.GetMarkerInfo(testList[m-covariates.Length()])->position*100,
            ols.t_statistic[m+1], ols.pvalue[m+1],
            ols.R2[m]*100, ols.R2[m] * effect * 100);
      double lrt = 2 * (ols.loglik - loglik0);
      double pvalue = (lrt >= 0.0) ? chidist(lrt, testList.Length()): 1.0;
      printf("LRT statistic: %.1f\tp-value: %.3G\n", lrt, pvalue);
   }
}

void OLS_Association::Analyze()
{
   if(scanFlag){
      if(multivariateFlag) OLSmultivariate();
      else OLSunivariate();
   }else OLSregression();
}

void OLS_Association::SetupGlobals()
{
   AssociationAnalysis::SetupGlobals();
}

void OLS_Association::GetResiduals()
{
   int K = SubIndex.Length();
   int N = pheno.Length();
   Vector V;
   Matrix M;
   SVD svd;
   Residual.Dimension(K, N);
   Residual.Set(_NAN_);

   if(covariates.Length() || sequentialFlag){
      printf("\nFitted Models Without Genome Scan\n");
      printf("=================================================================\n");
      printf("%15s%8s", "Trait", "Mean");
      for(int i = 0; i < covariates.Length(); i++)
         printf("%8s", (const char*)ped.covariateNames[covariates[i]]);
      if(sequentialFlag) printf("%8s\n", "(SNPs)");
      else printf("%7s\n", "N");
   }
   if(multivariateFlag){
      int P = covariates.Length() + 1 + causalSNPList[0].Length();
      int K = traits.Length();
      Matrix Y(K, N);   // Y[i]: traits for ith individual
      Matrix Z(N, P);
      Matrix OR(N, K);  // OR: (Omega^-1 R)'
      Matrix B(K, P);   // B[k]: regression coefficients for kth trait
      Matrix O(K, K);   // O[k][l]: covariance between trait k and l
      Vector tV;
      Matrix tM;
      Cholesky chol;
      for(int i = 0; i < N; i++){
         Z[i][0] = 1;
         for(int c = 1; c < covariates.Length()+1; c++)
            Z[i][c] = ped[pheno[i]].covariates[covariates[c-1]];
         for(int m = 0; m < causalSNPList[0].Length(); m++)
            Z[i][m+covariates.Length()+1] =
            ped[pheno[i]].markers[causalSNPList[0][m]].countAlleles(1);
         for(int k = 0; k < K; k++)
            Y[k][i] = ped[pheno[i]].traits[traits[k]];
      }
      for(int k = 0; k < K; k++){
         svd.Decompose(Z);
         svd.Edit();
         svd.BackSubst(Y[k]);
         B[k] = svd.x;
      }
      Residual = Y;
      for(int k = 0; k < K; k++)
         for(int i = 0; i < N; i++)
            Residual[k][i] -= Z[i].InnerProduct(B[k]);
      for(int k = 0; k < K; k++){
         O[k][k] = Residual[k].Var();
         for(int l = k+1; l < K; l++)
            O[l][k] = O[k][l] = (Residual[k].InnerProduct(Residual[l]) -
               Residual[k].Sum()*Residual[l].Sum()/N) / (N-1);
      }
      tM.Transpose(Residual);
      chol.Decompose(O);
      for(int i = 0; i < N; i++){
         chol.BackSubst(tM[i]);
         OR[i] = chol.x;
      }
      Residual.Transpose(OR);

      if(P>1){
/*         printf("\nFitted Models Without Genotypes (N=%d)\n", N);
         printf("=================================================================\n");
         printf("%15s%10s", "Trait", "Mean");
         for(int i = 0; i < P-1; i++)
            printf("%10s", (const char*)ped.covariateNames[covariates[i]]);
         printf("\n");
*/
         for(int k = 0; k < K; k++){
            printf("%15s", (const char*)ped.traitNames[traits[k]]);
            for(int i = 0; i < P; i++)
               printf("%8.1G", B[k][i]);
            printf("\n");
         }
         printf("\n");
      }
   }else{
      if(IndexR == NULL) IndexR = new IntArray[K];
      for(int k = 0; k < K; k++){
         IntArray validPerson(pheno.Length());
         validPerson.Set(1);
         for(int i = 0; i < pheno.Length(); i++){
            if(!ped[pheno[i]].isPhenotyped(traits[SubIndex[k]])){
               validPerson[i] = 0;
               continue;
            }
            for(int c = 0; c < covariates.Length(); c++)
               if(!ped[pheno[i]].isControlled(covariates[c])){
                  validPerson[i] = 0;
                  break;
               }
            if(!validPerson[i]) continue;
            for(int m = 0; m < causalSNPList[SubIndex[k]].Length(); m++){
               if(!ped[pheno[i]].markers[causalSNPList[SubIndex[k]][m]].isKnown()){
                  validPerson[i] = 0;
                  break;
               }
            }
         }

         Vector Y;   // Y[i]: traits for ith individual
         Vector R;   // R[i]: residual for ith individual
         Vector B;   // B: regression coefficients for kth trait
         Vector V2;

         IntArray sample;
         int P = covariates.Length() + 1 + causalSNPList[SubIndex[k]].Length();
         B.Dimension(P);

         // obtain residuals
         sample.Dimension(0);
         for(int i = 0; i < pheno.Length(); i++)
            if(validPerson[i]) sample.Push(i);
         int N0 = sample.Length();
         Matrix Z(N0, P);
         Y.Dimension(N0);
         R.Dimension(N0);

         for(int i = 0; i < N0; i++){
            Z[i][0] = 1;
            for(int c = 0; c < covariates.Length(); c++)
               Z[i][c+1] = ped[pheno[sample[i]]].covariates[covariates[c]];
            for(int m = 0; m < causalSNPList[SubIndex[k]].Length(); m++)
               Z[i][m+covariates.Length()+1] =
   ped[pheno[sample[i]]].markers[causalSNPList[SubIndex[k]][m]].countAlleles(1);
            Y[i] = ped[pheno[sample[i]]].traits[traits[SubIndex[k]]];
         }
         svd.Decompose(Z);
         svd.Edit();
         svd.BackSubst(Y);
         B = svd.x;
         IndexR[k].Dimension(0);
         for(int i = 0; i < N0; i++){
            Residual[k][sample[i]] = Y[i] - Z[i].InnerProduct(B);
            IndexR[k].Push(sample[i]);
         }
         if(P>1 || sequentialFlag){
            printf("%15s", (const char*)ped.traitNames[traits[SubIndex[k]]]);
            for(int i = 0; i < P; i++)
               printf("%8.1G", B[i]);
            if(sequentialFlag) printf("\n");
            else printf("%7d\n", N0);
         }
      }
   }
}

void OLS_Association::OLSunivariate()
{
   int K = SubIndex.Length();
   int N = pheno.Length();
   Vector V;
   Matrix M;
   Vector X(N), G, Y;
   double statistic, pvalue, temp;
   for(int k = 0; k < K; k++){
      statArray[k].Dimension(0);
      pvalueArray[k].Dimension(0);
      pvaluePosition[k].Dimension(0);
   }
   IntArray IndexRCount(K);
   for(int k = 0; k < K; k++)
      IndexRCount[k] = IndexR[k].Length();
   for(int m = 0; m < markers.Length(); m++){
      X.Set(_NAN_);
      for(int i = 0; i < N; i++)
         if(ped[pheno[i]].markers[markers[m]].isKnown())
            X[i] = ped[pheno[i]].markers[markers[m]].countAlleles(1);
      for(int k = 0; k < K; k++){
         G.Dimension(0);
         Y.Dimension(0);
         for(int i = 0; i < IndexRCount[k]; i++)
            if(X[IndexR[k][i]] != _NAN_){
               G.Push(X[IndexR[k][i]]);
               Y.Push(Residual[k][IndexR[k][i]]);
            }
         temp = G.Sum()/G.Length();
         G.Add(-temp);
         V.Dimension(G.Length());
         for(int i = 0; i < G.Length(); i++)
            V[i] = G[i] * Y[i];
         temp = V.SumSquares();
         if(temp < 1E-50) continue;
         statistic = V.Sum();
         statistic = statistic * statistic / temp;
         pvalue = _NAN_;
         if(statistic > peakStatistic[k]){
            pvalue = chidist(statistic, 1);
            peakPvalue[k] = pvalue;
            peakPosition[k] = m;
            peakStatistic[k] = statistic;
         }
         if(statistic > statisticGreaterThan){
            if(pvalue==_NAN_)
               pvalue = chidist(statistic, 1);
            statArray[k].Push(statistic);
            pvalueArray[k].Push(pvalue);
            pvaluePosition[k].Push(m);
         }
      }
   }
   for(int k = 0; k < K; k++){
      if(pvalueArray[k].Length()==0) continue;
      printf("\nGenome-Wide Association Analysis of %s\n",
         (const char*)ped.traitNames[traits[SubIndex[k]]]);
      printf("======================================================================\n",
         (const char*)ped.traitNames[traits[SubIndex[k]]]);
      printf("%10s%7s%10s%15s\n", "Position", "LOD", "pvalue", "Marker");
      for(int i = 0; i < pvalueArray[k].Length(); i++){
         int m = pvaluePosition[k][i];
         double LOD = statArray[k][i] * ToLOD;
         printf("%10f%7.2f%10.2G%15s", ped.GetMarkerInfo(markers[m])->position*100,
            LOD, pvalueArray[k][i], (const char*)ped.GetMarkerInfo(markers[m])->name);
         printf("\n");
      }
   }
   for(int k = 0; k < K; k++)
      if(peakPvalue[k] < peakGenomePvalue[k]){
         peakGenomePvalue[k] = peakPvalue[k];
         peakGenomePosition[k] = markers[peakPosition[k]];
      }
   if(!sequentialFlag)
      PrintSummary();

   // Permutation Test
   if(permutationCount==0) return;
   PERMUTATION permu;
   permu.init(N, randomSeed);
   IntArray Index(permutationCount * N);
   for(int p = 0; p < permutationCount; p++){
      permu.shuffle();
      for(int i = 0; i < N; i++)
         Index[p*N+i] = permu[i];
   }
   double t;
   Matrix largest(K, permutationCount);
   largest.Set(-1.0);
   Vector X0(N);

   for(int m = 0; m < markers.Length(); m++){
      X0.Set(_NAN_);
      for(int i = 0; i < N; i++)
         if(ped[pheno[i]].markers[markers[m]].isKnown())
            X0[i] = ped[pheno[i]].markers[markers[m]].countAlleles(1);
      for(int p = 0; p < permutationCount; p++){
         for(int i = 0; i < N; i++)
            X[i] = X0[Index[p*N+i]];
         for(int k = 0; k < K; k++){
            G.Dimension(0);
            Y.Dimension(0);
            for(int i = 0; i < IndexRCount[k]; i++)
               if(X[IndexR[k][i]] != _NAN_){
                  G.Push(X[IndexR[k][i]]);
                  Y.Push(Residual[k][IndexR[k][i]]);
               }
            if(G.Length()==0) continue;
            temp = G.Sum()/G.Length();
            V.Dimension(G.Length());
            for(int i = 0; i < G.Length(); i++)
               V[i] = (G[i] - temp) * Y[i];
            temp = V.SumSquares();
            if(temp < 1E-50) continue;
            t = V.Sum();
            temp = t * t / temp;
            if(temp > largest[k][p]) largest[k][p] = temp;
         }
      }
   }
   for(int k = 0; k < K; k++)
      for(int p = 0; p < permutationCount; p++){
         temp = chidist(largest[k][p], 1);
         if(temp < permutationMatrix[k][p])
            permutationMatrix[k][p] = temp;
      }
}

void OLS_Association::MultivariateRegression(IntArray testList)
{
   int K = traits.Length();
   int P = covariates.Length()+1+testList.Length();

   IntArray sample(0);
   for(int i = 0; i < pheno.Length(); i++){
      int valid = 1;
      for(int k = 0; k < K; k++)
         if(!ped[pheno[i]].isPhenotyped(traits[k])) {
            valid = 0;
            break;
         }
      if(!valid) continue;
      for(int m = 0; m < testList.Length(); m++)
         if(!ped[pheno[i]].markers[testList[m]].isKnown()){
            valid = 0;
            break;
         }
      if(valid) sample.Push(i);
   }
   int N = sample.Length();

   Matrix Y(K, N);
   for(int k = 0; k < K; k++)
      for(int i = 0; i < N; i++)
         Y[k][i] = ped[pheno[sample[i]]].traits[traits[k]];
   Matrix X(N, P);
   for(int i = 0; i < N; i++){
      X[i][0] = 1;
      for(int m = 0; m < covariates.Length(); m++)
         X[i][m+1] = ped[pheno[sample[i]]].covariates[covariates[m]];
      for(int m = 0; m < testList.Length(); m++)
         X[i][m+1+covariates.Length()] = ped[pheno[sample[i]]].markers[testList[m]].countAlleles(1);
   }


   Matrix beta(K, 1+covariates.Length()+testList.Length());
   SVD svd;
   svd.Decompose(X);
   svd.Edit();
   Matrix R(K, N);
   for(int k = 0; k < K; k++){
      svd.BackSubst(Y[k]);
      beta[k] = svd.x;
      R[k].Product(X, beta[k]);
      R[k].Multiply(-1.0);
      R[k].Add(Y[k]);
   }

   Matrix O(K, K);
   for(int k = 0; k < K; k++){
      O[k][k] = R[k].Var();
      for(int l = k+1; l < K; l++)
         O[l][k] = O[k][l] = (R[k].InnerProduct(R[l]) -
            R[k].Sum()*R[l].Sum()/N) / (N-1);
   }

   Matrix *XX = new Matrix[N];
   Matrix XY(N, P*K);
   for(int i = 0; i < N; i++)
      XX[i].Dimension(P*K, P*K);
   Cholesky chol;
//   if(!chol.TryDecompose(O)) error("Failed 1");
   chol.Decompose(O);
   chol.Invert();

   Matrix tXX(P, P);
   for(int i = 0; i < N; i++){
      for(int u = 0; u < P; u++)
         for(int v = 0; v < P; v++)
            tXX[u][v] = X[i][u] * X[i][v];
      for(int s = 0; s < K; s++)
         for(int t = 0; t < K; t++)
            for(int u = 0; u < P; u++)
               for(int v = 0; v < P; v++)
                  XX[i][s*P+u][t*P+v] = tXX[u][v] * chol.inv[s][t];
      for(int s = 0; s < K; s++){
         double sum = 0;
         for(int k = 0; k < K; k++)
            sum += chol.inv[s][k] * R[k][i];
         for(int u = 0; u < P; u++)
            XY[i][s*P + u] = X[i][u] * sum;
      }
   }

   Matrix Covariance(K*P, K*P);
   tXX.Dimension(K*P, K*P);
   for(int i = 0; i < N; i++)
      tXX.Add(XX[i]);

//   if(!chol.TryDecompose(tXX)) error("Failed 2");
   chol.Decompose(tXX);
   chol.Invert();
   for(int i = 0; i < N; i++)
      for(int u = 0; u < K*P; u++)
         for(int v = 0; v < K*P; v++)
            XX[i][u][v] = XY[i][u] * XY[i][v];
   tXX.Zero();
   for(int i = 0; i < N; i++)
      tXX.Add(XX[i]);
   Matrix tM;
   tM.Product(chol.inv, tXX);
   Covariance.Product(tM, chol.inv);
   delete []XX;

   Matrix R2(K, P-1);
   tM.Transpose(X);
   for(int k = 0; k < K; k++){
      Y[k].Add(-Y[k].Sum()/N);
      for(int i = 0; i < P-1; i++)
         R2[k][i] = tM[i+1].InnerProduct(Y[k]) * beta[k][i+1] / Y[k].SumSquares();
   }

   printf("\nSummary of Multivariate Analysis\n");
   printf("============================================================================\n");
   printf("%20s", "(P-value)");
   for(int k = 0; k < K; k++)
      printf("%15s", (const char*)ped.traitNames[traits[k]]);
   if(K>1) printf("%15s", "AllTraits");
   printf("\n");
   double pvalue, stat;
   Vector testBeta;
   Matrix testCov(K, K);
   for(int c = 0; c < covariates.Length(); c++){
      printf("%20s", (const char*)ped.covariateNames[covariates[c]]);
      for(int k = 0; k < K; k++){
         stat = beta[k][1+c] * beta[k][1+c] / Covariance[k*P+c+1][k*P+c+1];
         pvalue = chidist(stat, 1);
         printf("%15.2G", pvalue);
      }
      testBeta.Dimension(0);
      for(int k = 0; k < K; k++)
         testBeta.Push(beta[k][1+c]);
      for(int k = 0; k < K; k++)
         for(int l = 0; l < K; l++)
            testCov[k][l] = Covariance[k*P+c+1][l*P+c+1];
      chol.Decompose(testCov);
      chol.Invert();
      stat = 0;
      for(int k = 0; k < K; k++)
         for(int l = 0; l < K; l++)
            stat += testBeta[k] * chol.inv[k][l] * testBeta[l];
      pvalue = chidist(stat, K);
      if(K>1){
         printf("%15.2G", pvalue);
      }
      printf("\n");
   }
   for(int m = 0; m < testList.Length(); m++){
      String temp = ped.markerNames[testList[m]];
      String temp2 = temp;
      temp2 = ped.GetMarkerInfo(testList[m])->chromosome;
      temp.Add('('); temp.Add(temp2); temp.Add(')');
      printf("%20s", (const char*)temp);
      for(int k = 0; k < K; k++){
         double t = beta[k][1+covariates.Length()+m] * beta[k][1+covariates.Length()+m]
            / Covariance[k*P+covariates.Length()+1+m][k*P+covariates.Length()+1+m];
         pvalue = chidist(t, 1);
         printf("%15.2G", pvalue);
      }
      testBeta.Dimension(0);
      for(int k = 0; k < K; k++)
         testBeta.Push(beta[k][1+covariates.Length()+m]);
      for(int k = 0; k < K; k++)
         for(int l = 0; l < K; l++)
            testCov[k][l] = Covariance[k*P+covariates.Length()+1+m][l*P+covariates.Length()+1+m];
      chol.Decompose(testCov);
      chol.Invert();
      stat = 0;
      for(int k = 0; k < K; k++)
         for(int l = 0; l < K; l++)
            stat += testBeta[k] * chol.inv[k][l] * testBeta[l];
      pvalue = chidist(stat, K);
      if(K>1) printf("%15.2G", pvalue);
      printf("\n");
   }

   printf("\n%20s", "(Effect)");
   for(int k = 0; k < K; k++)
      printf("%15s", (const char*)ped.traitNames[traits[k]]);
   if(K>1) printf("%15s", "Corr(T1,T2)");
   printf("\n");
   for(int c = 0; c < covariates.Length(); c++){
      printf("%20s", (const char*)ped.covariateNames[covariates[c]]);
      for(int k = 0; k < K; k++)
         printf("%14.2f%%", R2[k][c]*100);
      if(K>1){
         double corr = Covariance[c+1][P+c+1] / sqrt(Covariance[c+1][c+1]*Covariance[P+c+1][P+c+1]);
         printf("%15.2f", corr);
      }
      printf("\n");
   }
   for(int m = 0; m < testList.Length(); m++){
      String temp = ped.markerNames[testList[m]];
      String temp2 = temp;
      temp2 = ped.GetMarkerInfo(testList[m])->chromosome;
      temp.Add('('); temp.Add(temp2); temp.Add(')');
      printf("%20s", (const char*)temp);
      for(int k = 0; k < K; k++)
         printf("%14.2f%%", R2[k][covariates.Length()+m]*100);
      if(K>1){
            double corr = Covariance[covariates.Length()+m+1][P+covariates.Length()+m+1]
            / sqrt(Covariance[covariates.Length()+m+1][covariates.Length()+m+1]
             * Covariance[P+covariates.Length()+m+1][P+covariates.Length()+m+1] );
            printf("%15.2f", corr);
      }
      printf("\n");
   }
}

void OLS_Association::OLSmultivariate()
{
   Cholesky chol;
   int K = traits.Length();
   int N;
   Vector X;   // genotype score
   Matrix OR;  // OR: (Omega^-1 R)'
   IntArray subset;
   Vector tV(K);
   Matrix tM;
   double statistic, pvalue=1.0;
   bool printHeader = true;

   for(int m = 0; m < markers.Length(); m++){
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < pheno.Length(); i++)
         if(ped[pheno[i]].markers[markers[m]].isKnown()){
            X.Push(ped[pheno[i]].markers[markers[m]].countAlleles(1));
            subset.Push(i);
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);
      if(X.Var()<1E-50) continue;

      tM.Transpose(Residual);
      if(N < pheno.Length()){
         OR.Dimension(N, K);
         for(int i = 0; i < N; i++)
            OR[i] = tM[subset[i]];
      }else OR = tM;
      for(int i = 0; i < N; i++)
         OR[i].Multiply(X[i]);
      tV.Set(0.0);
      for(int i = 0; i < N; i++)
         tV.Add(OR[i]);
      tM.Transpose(OR);
      Matrix tM2;
      tM2.Product(tM, OR);
      chol.Decompose(tM2);
      chol.BackSubst0(tV);
      statistic = chol.x.SumSquares();
      pvalue = _NAN_;
      if(statistic > peakStatistic[0]){
         pvalue = chidist(statistic, K);
         peakPvalue[0] = pvalue;
         peakPosition[0] = m;
         peakStatistic[0] = statistic;
         if(pvalue > pvalueLessThan) continue;
      }
      if(statistic > statisticGreaterThan){
         if(pvalue==_NAN_)
            pvalue = chidist(statistic, K);
         if(pvalue > pvalueLessThan) continue;
         if(printHeader){
            printf("Pedigree-Wide Association Analysis\n");
            printf("==================================================\n");
            printf("%10s%7s%10s%15s%5s\n",
               "Position", "chisq", "pvalue", "Marker", "N");

            printHeader = false;

         }
         printf("%10f%7.2f%10.2G%15s%5d\n",
            ped.GetMarkerInfo(markers[m])->position*100, statistic, pvalue,
            (const char*)ped.GetMarkerInfo(markers[m])->name, N);
      }
   }

   printf("\nSummary of GWA results\n");
   printf("===================================================\n");
   printf("%10s%10s%15s%15s\n",
      "P-Value", "Chrom", "Position", "SNP_Name");
   int m = peakPosition[0];
   printf("%10.3G%10d%15f%15s\n",
      peakPvalue[0], ped.GetMarkerInfo(markers[m])->chromosome,
      ped.GetMarkerInfo(markers[m])->position*100,
      (const char*)ped.GetMarkerInfo(markers[m])->name);
   if(peakPvalue[0] < peakGenomePvalue[0]){
      peakGenomePvalue[0] = peakPvalue[0];
      peakGenomePosition[0] = markers[peakPosition[0]];
   }

   // Permutation Test
   if(permutationCount==0) return;
   double temp;
   Matrix tM2, tM3;
   IntArray *Index = new IntArray[permutationCount];
   PERMUTATION permu;
   permu.init(pheno.Length(), randomSeed);
   Vector largest(permutationCount);
   largest.Set(-1.0);

   for(int p = 0; p < permutationCount; p++){
      permu.shuffle();
      Index[p].Dimension(permu.count);
      for(int i = 0; i < permu.count; i++)
         Index[p][i] = permu[i];
   }
   tV.Dimension(K);
   tM.Dimension(K, K);
   for(int m = 0; m < markers.Length(); m++){
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < pheno.Length(); i++)
         if(ped[pheno[i]].markers[markers[m]].isKnown()){
            subset.Push(i);
            X.Push(ped[pheno[i]].markers[markers[m]].countAlleles(1));
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);
      if(X.Var()<1E-50) continue;

      tM.Transpose(Residual);
      OR.Dimension(N, K);
      for(int p = 0; p < permutationCount; p++){
         for(int i = 0; i < N; i++){
            OR[i] = tM[Index[p][subset[i]]];
            OR[i].Multiply(X[i]);
         }
         tM2.Transpose(OR);
         tM3.Product(tM2, OR);
         chol.Decompose(tM3);
         tV.Set(0.0);
         for(int i = 0; i < N; i++)
            tV.Add(OR[i]);
         chol.BackSubst0(tV);
         temp = chol.x.SumSquares();
         if(temp > largest[p])
            largest[p] = temp;
      }
   }
   for(int p = 0; p < permutationCount; p++){
      temp = chidist(largest[p], K);
      if(temp < permutationMatrix[0][p])
         permutationMatrix[0][p] = temp;
   }
   delete [] Index;
}

void OLS_Association::WriteUnrelated()
{
   IntArray sample(ped.count);

   sample.Zero();

   for(int i = 0; i < pheno.Length(); i++)

      sample[pheno[i]] = 1;


   // eliminate sample[i]=0

   int count = pheno.Length();

   Person ** newPersons = new Person * [count];

   int newCount = 0;

   for(int f = 0; f < ped.familyCount; f++)

      for(int i = ped.families[f]->count-1; i >= 0; i--){

         if(sample[ped.families[f]->path[i]]==0)
            delete ped.persons[ped.families[f]->path[i]];
         else{
            newPersons[newCount] = ped.persons[ped.families[f]->path[i]];
            newPersons[newCount]->famid.Add(newPersons[newCount]->pid);
            newPersons[newCount]->motid.Copy("0");
            newPersons[newCount]->fatid.Copy("0");
            newCount ++;
         }
      }

   delete [] ped.persons;

   ped.persons = newPersons;
   ped.count = newCount;
   ped.Sort();
   if(covariates.Length()==0) {
      ped.covariateCount = 0;

      printf("No covariates are written.\n");

   }

   if(traits.Length() < ped.traitCount){  // drop traits

      Vector V(traits.Length());

      for(int i = 0; i < ped.count; i++){

         for(int k = 0; k < traits.Length(); k++)

            V[k] = ped[i].traits[traits[k]];

         ped[i].traits = V;

      }

      ped.traitCount = traits.Length();

      StringArray S(traits.Length());

      for(int k = 0; k < traits.Length(); k++)

         S[k] = ped.traitNames[traits[k]];

      ped.traitNames = S;

   }


   // write file

   String datafile, pedfile;

   datafile.Copy("new");
   datafile.Add(prefix);
   datafile.Add(".dat");
   pedfile.Copy("new");
   pedfile.Add(prefix);
   pedfile.Add(".ped");
   ped.WriteDataFile(datafile);
   ped.WritePedigreeFile(pedfile);
   printf("\n%d individuals have complete genotypes/phenotypes.\n", completeCount);
   printf("\n%d unrelated individuals are saved in %s and %s.\n",
      pheno.Length(), (const char*)datafile, (const char*)pedfile);
}


void OLS_Association::GetUnrelated()

{
   pheno.Dimension(0);

   bool isUnrelated = true;
   for(int f = 0; f < ped.familyCount; f++)
      if(ped.families[f]->count>1) {
         isUnrelated = false;
         break;
      }
   if(isUnrelated){
      for(int i = 0; i < ped.count; i++){

         int valid = 0;

         if(multivariateFlag){

            valid = 1;

            for(int tr = 0; tr < traits.Length(); tr++)

               if(!ped[i].isPhenotyped(traits[tr])) {valid = 0; break;}
         }else{

            for(int tr = 0; tr < traits.Length(); tr++)
               if(ped[i].isPhenotyped(traits[tr])) {valid = 1; break;}
         }
         if(!valid) continue;
         for(int c = 0; c < covariates.Length(); c++)
            if(!ped[i].isControlled(covariates[c])) {valid = 0; break;}
         pheno.Push(i);
      }
      return;

   }


   IntArray valid;
   IntArray connectionCount;
   IntArray sample;
   Kinship kin;

   Vector *kinship = NULL;
   kinship = new Vector[ped.familyCount];
   for(int f = 0; f < ped.familyCount; f++){
      int count = ped.families[f]->count;
      int first = ped.families[f]->first;
      if(count==0) continue;
      kin.Setup(*ped.families[f]);
      kinship[f].Dimension(count * count);
      for(int i = ped.families[f]->first; i <= ped.families[f]->last; i++){
         kinship[f][(i-first)*count+i-first] = kin(ped[i], ped[i]);
         for(int j = ped.families[f]->first; j < i; j++)
            kinship[f][(j-first)*count+i-first] =
               kinship[f][(i-first)*count+j-first] = kin(ped[i], ped[j]);
      }
   }
   IntArray validPerson(ped.count);
   validPerson.Set(0);

   for(int i = 0; i < ped.count; i++){
      if(multivariateFlag){
         validPerson[i] = 1;

         for(int tr = 0; tr < traits.Length(); tr++)

            if(!ped[i].isPhenotyped(traits[tr])) {validPerson[i] = 0; break;}
      }else{

         for(trait = 0; trait < traits.Length(); trait++)
            if(ped[i].isPhenotyped(traits[trait])) {validPerson[i] = 1; break;}
      }
      if(!validPerson[i]) continue;
      for(int c = 0; c < covariates.Length(); c++)
         if(!ped[i].isControlled(covariates[c])) {validPerson[i] = 0; break;}
   }

   int testCount = 5000;
   if(ped.markerCount < testCount) testCount = ped.markerCount;
   for(int i = 0; i < ped.count; i++){
      int count = 0;
      for(int m = 0; m < testCount; m++)
         if(ped[i].markers[m].isKnown())
            count ++;
      if(count <= testCount/2) validPerson[i] = 0;   // invalid markers
   }

   sample.Dimension(ped.count);
   sample.Zero();
   for(int f = 0; f < ped.familyCount; f++){
      int count = ped.families[f]->count;
      int first = ped.families[f]->first;
      valid.Dimension(0);
      for(int i = first; i <= ped.families[f]->last; i++)
         if(validPerson[i])
            valid.Push(i);
     if(valid.Length() == 0) continue;
     else if(valid.Length() == 1){
         sample[valid[0]] = 1;
         continue;
      }
      connectionCount.Dimension(valid.Length());
      connectionCount.Zero();
      for(;;){
         for(int i = 0; i < valid.Length(); i++)
            if(connectionCount[i]>-1)
               connectionCount[i] = 0;
         for(int i = 0; i < valid.Length(); i++){
            if(connectionCount[i]==-1) continue;
            for(int j = 0; j < i; j++){
               if(connectionCount[j]==-1) continue;
               if(kinship[f][(valid[i]-first)*count + valid[j]-first] > 0){
                  connectionCount[i]++;
                  connectionCount[j]++;
               }
            }
         }
         double min = 9999999;
         int k = -1;
         for(int i = 0; (i < valid.Length()); i++){
            if(connectionCount[i]==-1) continue;
            if(connectionCount[i] < min){
               min = connectionCount[i];
               k = i;
            }
         }
         if(k == -1) break;
         sample[valid[k]] = 1;
         for(int i = 0; i < valid.Length(); i++){
            if(connectionCount[i]==-1) continue;
            if(kinship[f][(valid[i]-first)*count + valid[k]-first] > 0)
               connectionCount[i] = -1;
         }
      }
   }
   for(int i = 0; i < ped.count; i++)
      if(sample[i]) pheno.Push(i);

   completeCount = validPerson.Sum();
//   printf("%d individuals have valid phenotypes/genotypes.\n", );
//   printf("%d unrelated individuals are selected.\n", sample.Sum());
   if(kinship) delete []kinship;
}


void OLS_Association::OLSregression(IntArray testList)
{
   OLS_REGRESSION ols;
   Vector Y;
   Matrix X;
   int N;
   IntArray sample;
   char tempName[256];
   printf("\n");

   printf("%15s%10s%15s\n", "MarkerName", "Alias", "Position");
   StringArray Alias(testList.Length());
   for(int m = 0; m < testList.Length(); m++){
      Alias[m].catprintf("SNP%d", m+1);
      printf("%15s%10s%15f\n", (const char*)ped.GetMarkerInfo(testList[m])->name,
         (const char*)Alias[m], ped.GetMarkerInfo(testList[m])->position*100);
   }
   printf("\n");

   int covariateCount = covariates.Length();

   StringArray covariateNames(0);
   for(int c = 0; c < covariateCount; c++)
      covariateNames.Push(ped.covariateNames[covariates[c]]);

   for(trait = 0; trait < traits.Length(); trait++){
      sample.Dimension(0);
      for(int i = 0; i < pheno.Length(); i++){
         if(!ped[pheno[i]].isPhenotyped(traits[trait])) continue;
         int valid = 1;
         for(int c = 0; c < covariates.Length(); c++)
            if(!ped[pheno[i]].isControlled(covariates[c])){
               valid = 0;
               break;
            }
         for(int m = 0; m < testList.Length(); m++)
            if(!ped[pheno[i]].markers[testList[m]].isKnown()){
               valid = 0;
               break;
            }
         if(valid) sample.Push(i);
      }
      N = sample.Length();

      Y.Dimension(sample.Length());
      X.Dimension(sample.Length(), covariateCount);
      X.Zero();
      for(int i = 0; i < N; i++){
         Y[i] = ped[pheno[sample[i]]].traits[traits[trait]];
         for(int c = 0; c < covariateCount; c++)
            X[i][c] = ped[pheno[sample[i]]].covariates[covariates[c]];
      }
      double loglik0;
      if(covariateCount==0)
         loglik0 = -0.5 * N * log(Y.Var() * (N-1.0) / N);
      else{
         ols.run(Y, X);
         loglik0 = ols.loglik;
      }

      Y.Dimension(N);
      X.Dimension(N, 1+covariateCount);
      for(int i = 0; i < N; i++)
         Y[i] = ped[pheno[sample[i]]].traits[traits[trait]];

      for(int i = 0; i < N; i++)
         for(int c = 0; c < covariateCount; c++)
            X[i][c] = ped[pheno[sample[i]]].covariates[covariates[c]];
      for(int marker = 0; marker < testList.Length(); marker++){
         int m = testList[marker];
         for(int i = 0; i < N; i++)
            X[i][covariateCount] = ped[pheno[sample[i]]].markers[m].countAlleles(2);
         ols.run(Y, X);
         ols.covariateNames = covariateNames;
         ols.covariateNames.Push(Alias[marker]);
         sprintf(tempName, "\nSingle-SNP Association for %s",
            (const char*)ped.traitNames[traits[trait]]);
         ols.nuisanceCount = covariateCount;
         ols.Print(tempName);
         double lrt = 2 * (ols.loglik - loglik0);
         printf("LRT statistic: %.1f\tp-value: %.3G\n", lrt, chidist(lrt > 0? lrt: 0, 1));
      }
      if(testList.Length() < 2) continue;

      // N-SNP Analysis
      Y.Dimension(N);
      X.Dimension(N, covariateCount + testList.Length());
      for(int i = 0; i < N; i++){
         Y[i] = ped[pheno[sample[i]]].traits[traits[trait]];
         for(int c = 0; c < covariateCount; c++)
            X[i][c] = ped[pheno[sample[i]]].covariates[covariates[c]];
         for(int m = 0; m < testList.Length(); m++)
            X[i][covariateCount+m] = ped[pheno[sample[i]]].markers[testList[m]].countAlleles(2);
      }
      ols.run(Y, X);
      ols.covariateNames = covariateNames;
      for(int m = 0; m < testList.Length(); m++)
         ols.covariateNames.Push(Alias[m]);
      sprintf(tempName, "\nMulti-SNP Association for %s",
         (const char*)ped.traitNames[traits[trait]]);
      ols.nuisanceCount = covariateCount;         
      ols.Print(tempName);
      double lrt = 2 * (ols.loglik - loglik0);
      printf("LRT statistic: %.1f\tp-value: %.3G\n", lrt, chidist(lrt > 0? lrt: 0, testList.Length()));
   }
}


/*
      // N-SNP Interaction Analysis
      Y.Dimension(N);
      X.Dimension(N, testList.Length() + 1);
      for(int i = 0; i < N; i++){
         Y[i] = ped[pheno[sample[i]]].traits[traits[trait]];
         for(int m = 0; m < testList.Length(); m++)
            X[i][m] = ped[pheno[sample[i]]].markers[testList[m]].countAlleles(1);
         X[i][testList.Length()] = X[i][0] * X[i][1];
      }
      ols.run(Y, X);
      printf("\n%d-SNP Interaction Analysis (loglik=%.1f, df=%d)\n",
         testList.Length(), ols.loglik, testList.Length()+1);
      printf("=====================================================\n");
      printf("%10s%7s%10s%7s\n", "SNP", "t", "pvalue", "Effect");
      for(int m = 0; m < testList.Length(); m++)
         printf("%10s%7.2f%10.2G%6.1f%%\n",
            (const char*)Alias[m], ols.t_statistic[m], ols.pvalue[m], ols.R2[m]*100);
      int m = testList.Length();
      printf("%10s%7.2f%10.2G%6.1f%%\n",
          "SNP1xSNP2", ols.t_statistic[m], ols.pvalue[m], ols.R2[m]*100);
      lrt = 2 * (ols.loglik - loglik0);
      printf("LRT statistic: %.1f\tp-value: %.3G\n",
         lrt, chidist(lrt, testList.Length()+1));
*/



/*

         printf("%10s%7.2f%10.2G%10.2G%6.1f%%%10.1f%5d\n",
            (const char*)Alias[marker], ols.t_statistic[covariateCount+1],
            ols.pvalue[covariateCount+1], ols.beta[covariateCount+1],
            ols.R2[covariateCount]*100, ols.loglik, 1);
*/
//         ols.covariateNames.Dimension(0);
//         for(int c = 0; c < covariateCount; c++)
//            ols.covariateNames.Push(ped.covariateNames[covariates[c]]);


void OLS_Association::OLSregression()

{
   Vector Y, X;
   double beta;
   double Sxx, Sxy, Syy, meanX, meanY, v, R2;
   int length;
   double stat;
   double apvalue;

   if(pvalueArray==NULL) pvalueArray = new Vector[traits.Length()];
   if(pvaluePosition==NULL) pvaluePosition = new IntArray[traits.Length()];

   for(int t = 0; t < traits.Length(); t++){
      pvalueArray[t].Dimension(0);
      pvaluePosition[t].Dimension(0);
   }

   int printHead=0;
   for(trait = 0; trait < traits.Length(); trait++){
      if(printHead) printf("\n");
      printHead = 0;
      printf("Pedigree-Wide Association Analysis (%s)\n",
         (const char*)ped.traitNames[traits[trait]]);
      for(int m = 0; m < markerCount; m++){
         Y.Dimension(0);
         X.Dimension(0);
         for(int i = 0; i < pheno.Length(); i++){
            if(!ped[pheno[i]].markers[markers[m]].isKnown()) continue; 
            Y.Push(ped[pheno[i]].traits[traits[trait]]);
            X.Push(ped[pheno[i]].markers[markers[m]].countAlleles(1));
         }
         length = X.Length();
         if(length<3) continue;
         meanX = X.Average();
         meanY = Y.Average();
         Sxx = X.SumSquares() - meanX * meanX * length;
         Syy = Y.SumSquares() - meanY * meanY * length;
         if(Sxx<=0) continue;
         Sxy = X.InnerProduct(Y) - meanX * meanY * length;
         beta = Sxy / Sxx;
         X.Multiply(-beta);
         Y.Add(-meanY + beta * meanX);
         Y.Add(X);
         v = sqrt(Y.SumSquares() / (length - 2) / Sxx);
         stat = beta / v;
         R2 = beta * Sxy /Syy;
         apvalue = tdist(fabs(stat), length - 2); // df = length-2

         if(apvalue < peakPvalue[trait]){
            peakPvalue[trait] = apvalue;
            peakPosition[trait] = m;
            peakStatistic[trait] = stat;
         }
         if(apvalue < pvalueLessThan) {
            pvalueArray[trait].Push(apvalue);
            pvaluePosition[trait].Push(m);
         }
         if(apvalue < pvalueLessThan){
            if(printHead==0){
               printf("=======================================================\n"
                  "%10s%7s%10s%7s%15s", "Position", "t", "pvalue", "Effect", "Marker");

               printf("\n");

               printHead = 1;
            }
            printf("%10f%7.2f%10.2G%6.1f%%%15s",
               ped.GetMarkerInfo(markers[m])->position*100, stat,
               apvalue, R2*100, (const char*)ped.GetMarkerInfo(markers[m])->name);
            printf("\n");
         }
      }
   }
   PrintSummary();
}


void OLS_Association::OLSmultivariate(IntArray testList)
{   // for backup only
   IntArray validPerson(ped.count);
   validPerson.Set(1);
   for(int i = 0; i < ped.count; i++){
      for(int k = 0; k < traits.Length(); k++)
         if(!ped[i].isPhenotyped(traits[k])){
            validPerson[i] = 0;
            break;
         }
      if(!validPerson[i]) continue;
      for(int c = 0; c < covariates.Length(); c++)
         if(!ped[i].isControlled(covariates[c])){
            validPerson[i] = 0;
            break;
         }
   }

   Matrix Y;   // Y[i]: traits for ith individual
   Matrix R;   // R[i]: residual for ith individual
   Matrix OR0;  // OR: Omega^-1 R
   Matrix B;   // B[k]: regression coefficients for kth trait
   Matrix O;   // O[k][l]: covariance between trait k and l
   Vector tV, tV2;
   Matrix tM;
   Cholesky chol;
   IntArray sample;
   int P = covariates.Length() + 1;
   int K = traits.Length();
   B.Dimension(K, P);
   O.Dimension(K, K);

   // obtain residuals
   sample.Dimension(0);
   for(int i = 0; i < ped.count; i++)
      if(validPerson[i]) sample.Push(i);

   int N0 = sample.Length();
   Matrix Z(N0, P);

   Y.Dimension(N0, K);
   R.Dimension(N0, K);

   for(int i = 0; i < N0; i++){
      Z[i][0] = 1;
      for(int c = 1; c < P; c++)
         Z[i][c] = ped[sample[i]].covariates[covariates[c-1]];
      for(int k = 0; k < K; k++)
         Y[i][k] = ped[sample[i]].traits[traits[k]];
   }

   tM.Dimension(P, P);  // tM = ZZ'
   tM.Set(0.0);
   for(int s = 0; s < P; s++)
      for(int t = 0; t < P; t++)
         for(int i = 0; i < N0; i++)
            tM[s][t] += Z[i][s] * Z[i][t];
   chol.Decompose(tM);
   tV.Dimension(P);    // tV = ZY
   for(int k = 0; k < K; k++){
      tV.Set(0.0);
      for(int i = 0; i < N0; i++){
         tV2 = Z[i];
         tV2.Multiply(Y[i][k]);
         tV.Add(tV2);
      }
      chol.BackSubst(tV);
      B[k] = chol.x;
   }
   for(int i = 0; i < N0; i++){
      R[i] = Y[i];
      for(int k = 0; k < K; k++)
         R[i][k] -= Z[i].InnerProduct(B[k]);
   }

   tM.Transpose(R);    // covariance matrix
   for(int k = 0; k < K; k++){
      O[k][k] = tM[k].Var();
      for(int l = k+1; l < K; l++)
         O[l][k] = O[k][l] = (tM[k].InnerProduct(tM[l]) -
            tM[k].Sum()*tM[l].Sum()/N0) / (N0-1);
   }
   OR0.Dimension(N0, K);
   chol.Decompose(O);
   for(int i = 0; i < N0; i++){
      chol.BackSubst(R[i]);
      OR0[i] = chol.x;
   }

   if(P>1){
      printf("\nFitted Models Without Genotypes (N=%d)\n", N0);
      printf("=================================================================\n");
      printf("%15s%10s", "Trait", "Mean");
      for(int i = 0; i < P-1; i++)
         printf("%10s", (const char*)ped.covariateNames[covariates[i]]);
      printf("\n");
      for(int k = 0; k < K; k++){
         printf("%15s", (const char*)ped.traitNames[traits[k]]);
         for(int i = 0; i < P; i++)
            printf("%10.3f", B[k][i]);
         printf("\n");
      }
      printf("\n");
   }

   // GWA
   Vector X;   // genotype score
   Matrix OR;  // OR: Omega^-1 R
   IntArray subset;
   int N;
   Vector statistic(traits.Length()+1);
   Vector pvalue(traits.Length()+1);
   pvalue.Set(1.0);
   printf("Pedigree-Wide Association Analysis\n");
   printf("=================================================================\n");
   printf("%10s%15s%7s%10s%15s%5s\n",
      "Position", "TraitName", "chisq", "pvalue", "Marker", "N");

   tV.Dimension(K);
   for(int m = 0; m < markers.Length(); m++){
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < sample.Length(); i++)
         if(ped[sample[i]].markers[markers[m]].isKnown()){
            subset.Push(i);
            X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);
      if(X.Var()<1E-50) continue;

      OR.Dimension(N, K);
      for(int i = 0; i < N; i++){
         OR[i] = OR0[subset[i]];
         OR[i].Multiply(X[i]);
      }
      tV.Set(0.0);
      for(int i = 0; i < N; i++)
         tV.Add(OR[i]);
      tM.Dimension(K, K);
      tM.Set(0.0);
      for(int k = 0; k < K; k++)
         for(int l = 0; l < K; l++)
            for(int i = 0; i < N; i++)
               tM[k][l] += OR[i][k] * OR[i][l];
      chol.Decompose(tM);
      chol.BackSubst0(tV);
      statistic[K] = chol.x.SumSquares();
      pvalue[K] = chidist(statistic[K], K);

      // Univariate Analysis
      Y.Dimension(N, K);
      for(int i = 0; i < N; i++){
         Y[i] = R[subset[i]];
         Y[i].Multiply(X[i]);
      }
      tM.Transpose(Y);
      for(int k = 0; k < K; k++){
         double t = tM[k].Sum();
         statistic[k] = t * t / tM[k].SumSquares();
         pvalue[k] = chidist(statistic[k], 1);
      }

      if(pvalue[K] < peakPvalue[0]){
         peakPvalue[0] = pvalue[K];
         peakPosition[0] = m;
         peakStatistic[0] = statistic[K];
      }
      if(pvalue[K] < pvalueLessThan){
         printf("%10f%15s%7.2f%10.2G%15s%5d\n",
            ped.GetMarkerInfo(markers[m])->position*100,
            "Multivariate", statistic[K],
            pvalue[K], (const char*)ped.GetMarkerInfo(markers[m])->name, N);
         for(int k = 0; k < K; k++)
            printf("%10s%15s%7.2f%10.2G%15s%5s\n",
               "", (const char*)ped.traitNames[traits[k]], statistic[k],
               pvalue[k], "", "");
         }
   }
   printf("\nSummary of GWA results\n");
   printf("==================================================================\n");
   printf("%15s%10s%10s%15s%15s\n",
      "Trait_Name", "P-Value", "Chrom", "Position", "SNP_Name");
   int m = peakPosition[0];
   printf("%15s%10.3G%10d%15f%15s\n", "Multivariate",
      peakPvalue[0], ped.GetMarkerInfo(markers[m])->chromosome,
      ped.GetMarkerInfo(markers[m])->position*100,
      (const char*)ped.GetMarkerInfo(markers[m])->name);

   // Permutation Test
   if(permutationCount==0) return;
   printf("\nPermutation test (%d tests per scan, %d times)...\n",
      permutationCount, markerCount);
   Vector largest;
   double temp;
   IntArray *Index;
   if(permutationCount){
      PERMUTATION permu;
      permu.init(N0, randomSeed);
      largest.Dimension(permutationCount);
      largest.Set(-1.0);
      Index = new IntArray[permutationCount];
      for(int p = 0; p < permutationCount; p++){
         permu.shuffle();
         Index[p].Dimension(permu.count);
         for(int i = 0; i < permu.count; i++)
            Index[p][i] = permu[i];
      }
   }
   tV.Dimension(K);
   tM.Dimension(K, K);
   for(int m = 0; m < markers.Length(); m++){
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < sample.Length(); i++)
         if(ped[sample[i]].markers[markers[m]].isKnown()){
            subset.Push(i);
            X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);
      if(X.Var()<1E-50) continue;
      OR.Dimension(N, K);
      for(int p = 0; p < permutationCount; p++){
         for(int i = 0; i < N; i++){
            OR[i] = OR0[Index[p][subset[i]]];
            OR[i].Multiply(X[i]);
         }
         tV.Set(0.0);
         for(int i = 0; i < N; i++)
            tV.Add(OR[i]);
         tM.Set(0.0);
         for(int k = 0; k < K; k++)
            for(int l = 0; l < K; l++)
               for(int i = 0; i < N; i++)
                  tM[k][l] += OR[i][k] * OR[i][l];
         chol.Decompose(tM);
         chol.BackSubst0(tV);
         temp = chol.x.SumSquares();
         if(temp > largest[p])
            largest[p] = temp;
      }
   }

   Vector smallest;
   smallest.Dimension(permutationCount);
   for(int p = 0; p < permutationCount; p++)
      smallest[p] = chidist(largest[p], K);
   if(permutationMatrix.cols == 0){ // one chromosome analysis
      permutationMatrix.Dimension(permutationCount, 1);
      for(int p = 0; p < permutationCount; p++)
         permutationMatrix[p][0] = smallest[p];
   }else{   // multiple chromosomes
      for(int p = 0; p < permutationCount; p++)
         if(smallest[p] < permutationMatrix[p][0])
            permutationMatrix[p][0] = smallest[p];
   }
   delete []Index;
}


void OLS_Association::InvNorm(int tr)

{
   IntArray individuals(0);
   Vector myTraits(0);
   QuickIndex idx;
   for (int i = 0; i < pheno.Length(); i++){
      int missing = 0;
      for(int k = 0; k < covariates.Length(); k++)
         if(!ped[pheno[i]].isControlled(covariates[k])){
            missing = 1;
            break;
         }
      if(!missing && ped[pheno[i]].isPhenotyped(traits[tr])){
         individuals.Push(i);
         myTraits.Push(ped[pheno[i]].traits[traits[tr]]);
      }
   }
   idx.Index(myTraits);
   for(int i = 0; i < individuals.Length();){
      int start = i, end = i + 1;
      while(end < individuals.Length() &&
            ped[pheno[individuals[idx[end]]]].traits[traits[tr]] ==
            ped[pheno[individuals[idx[start]]]].traits[traits[tr]])
         end++;
      end --;
      double q = ninv((start + (end-start)/2.0 + 0.5)/individuals.Length());
      for(int j = start; j <= end; j++)
         ped[pheno[individuals[idx[j]]]].traits[traits[tr]] = q;
      i = end + 1;
   }
}












/*
void OLS_Association::OLSregression()
{
   Vector Y, X;
   double beta;
   double Sxx, Sxy, Syy, meanX, meanY, v, R2;
   int length;
   IntArray valid;
   IntArray connectionCount;
   IntArray sample;
   Kinship kin;
   double stat;
   double apvalue;

   if(pvalueArray==NULL) pvalueArray = new Vector[traits.Length()];
   if(pvaluePosition==NULL) pvaluePosition = new IntArray[traits.Length()];

   for(int t = 0; t < traits.Length(); t++){
      pvalueArray[t].Dimension(0);
      pvaluePosition[t].Dimension(0);
   }

   int isFamilyData = 0;
   for(int f = 0; f < ped.familyCount; f++)
      if(ped.families[f]->count>1){
         isFamilyData = 1;
         break;
      }

   Vector *kinship = NULL;
   if(isFamilyData){
      kinship = new Vector[ped.familyCount];
      for(int f = 0; f < ped.familyCount; f++){
         int count = ped.families[f]->count;
         int first = ped.families[f]->first;
         if(count==0) continue;
         kin.Setup(*ped.families[f]);
         kinship[f].Dimension(count * count);
         for(int i = ped.families[f]->first; i <= ped.families[f]->last; i++){
            kinship[f][(i-first)*count+i-first] = kin(ped[i], ped[i]);
            for(int j = ped.families[f]->first; j < i; j++)
               kinship[f][(j-first)*count+i-first] =
                  kinship[f][(i-first)*count+j-first] = kin(ped[i], ped[j]);
         }
      }
   }
   int printHead=0;
   IntArray validPerson(ped.count);
   for(trait = 0; trait < traits.Length(); trait++){
      validPerson.Set(1);
      for(int i = 0; i < ped.count; i++){
         if(!ped[i].isPhenotyped(traits[trait])) {validPerson[i] = 0; continue;}
         for(int c = 0; c < covariates.Length(); c++)
            if(!ped[i].isControlled(covariates[c])) {validPerson[i] = 0; break;}
      }
      if(printHead) printf("\n");
      printHead = 0;
      printf("Pedigree-Wide Association Analysis (%s)\n",
         (const char*)ped.traitNames[traits[trait]]);
      for(int m = 0; m < markerCount; m++){
         if(ped.GetMarkerInfo(markers[m])->freq.Length()>3) continue;
         sample.Dimension(0);
         if(isFamilyData){
            for(int f = 0; f < ped.familyCount; f++){
               int count = ped.families[f]->count;
               int first = ped.families[f]->first;
               valid.Dimension(0);
               for(int i = first; i <= ped.families[f]->last; i++)
                  if(validPerson[i] && ped[i].markers[markers[m]].isKnown())
                     valid.Push(i);
               if(valid.Length() == 0) continue;
               else if(valid.Length() == 1){
                  sample.Push(valid[0]);
                  continue;
               }
               connectionCount.Dimension(valid.Length());
               connectionCount.Zero();
               for(;;){
                  for(int i = 0; i < valid.Length(); i++)
                     if(connectionCount[i]>-1) connectionCount[i] = 0;
                  for(int i = 0; i < valid.Length(); i++){
                     if(connectionCount[i]==-1) continue;
                     for(int j = 0; j < i; j++){
                        if(connectionCount[j]==-1) continue;
                        if(kinship[f][(valid[i]-first)*count + valid[j]-first] > 0){
                           connectionCount[i]++;
                           connectionCount[j]++;
                        }
                     }
                  }
                  double min = 9999999;
                  int k = -1;
                  for(int i = 0; (i < valid.Length()); i++){
                     if(connectionCount[i]==-1) continue;
                     if(connectionCount[i] < min){
                        min = connectionCount[i];
                        k = i;
                     }
                  }
                  if(k == -1) break;
                  sample.Push(valid[k]);
                  for(int i = 0; i < valid.Length(); i++){
                     if(connectionCount[i]==-1) continue;
                     if(kinship[f][(valid[i]-first)*count + valid[k]-first] > 0)
                        connectionCount[i] = -1;
                  }
               }
            }
         }else // population data
            for(int i = 0; i < ped.count; i++)
               if(ped[i].isPhenotyped(traits[trait]) && ped[i].markers[markers[m]].isKnown())
                  sample.Push(i);
         Y.Dimension(0);
         X.Dimension(0);
         for(int i = 0; i < sample.Length(); i++){
            Y.Push(ped[sample[i]].traits[traits[trait]]);
            X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
         }
         length = X.Length();
         if(length<3) continue;
         meanX = X.Average();
         meanY = Y.Average();
         Sxx = X.SumSquares() - meanX * meanX * length;
         Syy = Y.SumSquares() - meanY * meanY * length;
         if(Sxx<=0) continue;
         Sxy = X.InnerProduct(Y) - meanX * meanY * length;
         beta = Sxy / Sxx;
         X.Multiply(-beta);
         Y.Add(-meanY + beta * meanX);
         Y.Add(X);
         v = sqrt(Y.SumSquares() / (length - 2) / Sxx);
         stat = beta / v;
         R2 = beta * Sxy /Syy;
         apvalue = tdist(fabs(stat), length - 2); // df = length-2

         if(apvalue < peakPvalue[trait]){
            peakPvalue[trait] = apvalue;
            peakPosition[trait] = m;
            peakStatistic[trait] = stat;
         }
         if(apvalue < pvalueLessThan) {
            pvalueArray[trait].Push(apvalue);
            pvaluePosition[trait].Push(m);
         }
         if(apvalue < pvalueLessThan){
            if(printHead==0){
               printf("=======================================================\n"
                  "%10s%7s%10s%7s%15s", "Position", "t", "pvalue", "Effect", "Marker");

               if(isFamilyData) printf("%5s", "N");

               printf("\n");
               printHead = 1;
            }
            printf("%10f%7.2f%10.2G%6.1f%%%15s",
               ped.GetMarkerInfo(markers[m])->position*100, stat,
               apvalue, R2*100, (const char*)ped.GetMarkerInfo(markers[m])->name);
            if(isFamilyData) printf("%5d", length);
            printf("\n");
         }
      }
   }
   PrintSummary();
   if(kinship) delete []kinship;
}

void OLS_Association::OLSmultivariate(IntArray testList)
{   // for backup only
   IntArray validPerson(ped.count);
   validPerson.Set(1);
   for(int i = 0; i < ped.count; i++){
      for(int k = 0; k < traits.Length(); k++)
         if(!ped[i].isPhenotyped(traits[k])){
            validPerson[i] = 0;
            break;
         }
      if(!validPerson[i]) continue;
      for(int c = 0; c < covariates.Length(); c++)
         if(!ped[i].isControlled(covariates[c])){
            validPerson[i] = 0;
            break;
         }
   }

   Matrix Y;   // Y[i]: traits for ith individual
   Matrix R;   // R[i]: residual for ith individual
   Matrix OR0;  // OR: Omega^-1 R
   Matrix B;   // B[k]: regression coefficients for kth trait
   Matrix O;   // O[k][l]: covariance between trait k and l
   Vector tV, tV2;
   Matrix tM;
   Cholesky chol;
   IntArray sample;
   int P = covariates.Length() + 1;
   int K = traits.Length();
   B.Dimension(K, P);
   O.Dimension(K, K);

   // obtain residuals
   sample.Dimension(0);
   for(int i = 0; i < ped.count; i++)
      if(validPerson[i]) sample.Push(i);

   int N0 = sample.Length();
   Matrix Z(N0, P);

   Y.Dimension(N0, K);
   R.Dimension(N0, K);

   for(int i = 0; i < N0; i++){
      Z[i][0] = 1;
      for(int c = 1; c < P; c++)
         Z[i][c] = ped[sample[i]].covariates[covariates[c-1]];
      for(int k = 0; k < K; k++)
         Y[i][k] = ped[sample[i]].traits[traits[k]];
   }

   tM.Dimension(P, P);  // tM = ZZ'
   tM.Set(0.0);
   for(int s = 0; s < P; s++)
      for(int t = 0; t < P; t++)
         for(int i = 0; i < N0; i++)
            tM[s][t] += Z[i][s] * Z[i][t];
   chol.Decompose(tM);
   tV.Dimension(P);    // tV = ZY
   for(int k = 0; k < K; k++){
      tV.Set(0.0);
      for(int i = 0; i < N0; i++){
         tV2 = Z[i];
         tV2.Multiply(Y[i][k]);
         tV.Add(tV2);
      }
      chol.BackSubst(tV);
      B[k] = chol.x;
   }
   for(int i = 0; i < N0; i++){
      R[i] = Y[i];
      for(int k = 0; k < K; k++)
         R[i][k] -= Z[i].InnerProduct(B[k]);
   }

   tM.Transpose(R);    // covariance matrix
   for(int k = 0; k < K; k++){
      O[k][k] = tM[k].Var();
      for(int l = k+1; l < K; l++)
         O[l][k] = O[k][l] = (tM[k].InnerProduct(tM[l]) -
            tM[k].Sum()*tM[l].Sum()/N0) / (N0-1);
   }
   OR0.Dimension(N0, K);
   chol.Decompose(O);
   for(int i = 0; i < N0; i++){
      chol.BackSubst(R[i]);
      OR0[i] = chol.x;
   }

   if(P>1){
      printf("\nFitted Models Without Genotypes (N=%d)\n", N0);
      printf("=================================================================\n");
      printf("%15s%10s", "Trait", "Mean");
      for(int i = 0; i < P-1; i++)
         printf("%10s", (const char*)ped.covariateNames[covariates[i]]);
      printf("\n");
      for(int k = 0; k < K; k++){
         printf("%15s", (const char*)ped.traitNames[traits[k]]);
         for(int i = 0; i < P; i++)
            printf("%10.3f", B[k][i]);
         printf("\n");
      }
      printf("\n");
   }

   // GWA
   Vector X;   // genotype score
   Matrix OR;  // OR: Omega^-1 R
   IntArray subset;
   int N;
   Vector statistic(traits.Length()+1);
   Vector pvalue(traits.Length()+1);
   pvalue.Set(1.0);
   printf("Pedigree-Wide Association Analysis\n");
   printf("=================================================================\n");
   printf("%10s%15s%7s%10s%15s%5s\n",
      "Position", "TraitName", "chisq", "pvalue", "Marker", "N");

   tV.Dimension(K);
   for(int m = 0; m < markers.Length(); m++){
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < sample.Length(); i++)
         if(ped[sample[i]].markers[markers[m]].isKnown()){
            subset.Push(i);
            X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);
      if(X.Var()<1E-50) continue;

      OR.Dimension(N, K);
      for(int i = 0; i < N; i++){
         OR[i] = OR0[subset[i]];
         OR[i].Multiply(X[i]);
      }
      tV.Set(0.0);
      for(int i = 0; i < N; i++)
         tV.Add(OR[i]);
      tM.Dimension(K, K);
      tM.Set(0.0);
      for(int k = 0; k < K; k++)
         for(int l = 0; l < K; l++)
            for(int i = 0; i < N; i++)
               tM[k][l] += OR[i][k] * OR[i][l];
      chol.Decompose(tM);
      chol.BackSubst0(tV);
      statistic[K] = chol.x.SumSquares();
      pvalue[K] = chidist(statistic[K], K);

      // Univariate Analysis
      Y.Dimension(N, K);
      for(int i = 0; i < N; i++){
         Y[i] = R[subset[i]];
         Y[i].Multiply(X[i]);
      }
      tM.Transpose(Y);
      for(int k = 0; k < K; k++){
         double t = tM[k].Sum();
         statistic[k] = t * t / tM[k].SumSquares();
         pvalue[k] = chidist(statistic[k], 1);
      }

      if(pvalue[K] < peakPvalue[0]){
         peakPvalue[0] = pvalue[K];
         peakPosition[0] = m;
         peakStatistic[0] = statistic[K];
      }
      if(pvalue[K] < pvalueLessThan){
         printf("%10f%15s%7.2f%10.2G%15s%5d\n",
            ped.GetMarkerInfo(markers[m])->position*100,
            "Multivariate", statistic[K],
            pvalue[K], (const char*)ped.GetMarkerInfo(markers[m])->name, N);
         for(int k = 0; k < K; k++)
            printf("%10s%15s%7.2f%10.2G%15s%5s\n",
               "", (const char*)ped.traitNames[traits[k]], statistic[k],
               pvalue[k], "", "");
         }
   }
   printf("\nSummary of GWA results\n");
   printf("==================================================================\n");
   printf("%15s%10s%10s%15s%15s\n",
      "Trait_Name", "P-Value", "Chrom", "Position", "SNP_Name");
   int m = peakPosition[0];
   printf("%15s%10.3G%10d%15f%15s\n", "Multivariate",
      peakPvalue[0], ped.GetMarkerInfo(markers[m])->chromosome,
      ped.GetMarkerInfo(markers[m])->position*100,
      (const char*)ped.GetMarkerInfo(markers[m])->name);

   // Permutation Test
   if(permutationCount==0) return;
   printf("\nPermutation test (%d tests per scan, %d times)...\n",
      permutationCount, markerCount);
   Vector largest;
   double temp;
   IntArray *Index;
   if(permutationCount){
      PERMUTATION permu;
      permu.init(N0, randomSeed);
      largest.Dimension(permutationCount);
      largest.Set(-1.0);
      Index = new IntArray[permutationCount];
      for(int p = 0; p < permutationCount; p++){
         permu.shuffle();
         Index[p].Dimension(permu.count);
         for(int i = 0; i < permu.count; i++)
            Index[p][i] = permu[i];
      }
   }
   tV.Dimension(K);
   tM.Dimension(K, K);
   for(int m = 0; m < markers.Length(); m++){
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < sample.Length(); i++)
         if(ped[sample[i]].markers[markers[m]].isKnown()){
            subset.Push(i);
            X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);
      if(X.Var()<1E-50) continue;
      OR.Dimension(N, K);
      for(int p = 0; p < permutationCount; p++){
         for(int i = 0; i < N; i++){
            OR[i] = OR0[Index[p][subset[i]]];
            OR[i].Multiply(X[i]);
         }
         tV.Set(0.0);
         for(int i = 0; i < N; i++)
            tV.Add(OR[i]);
         tM.Set(0.0);
         for(int k = 0; k < K; k++)
            for(int l = 0; l < K; l++)
               for(int i = 0; i < N; i++)
                  tM[k][l] += OR[i][k] * OR[i][l];
         chol.Decompose(tM);
         chol.BackSubst0(tV);
         temp = chol.x.SumSquares();
         if(temp > largest[p])
            largest[p] = temp;
      }
   }

   Vector smallest;
   smallest.Dimension(permutationCount);
   for(int p = 0; p < permutationCount; p++)
      smallest[p] = chidist(largest[p], K);
   if(permutationMatrix.cols == 0){ // one chromosome analysis
      permutationMatrix.Dimension(permutationCount, 1);
      for(int p = 0; p < permutationCount; p++)
         permutationMatrix[p][0] = smallest[p];
   }else{   // multiple chromosomes
      for(int p = 0; p < permutationCount; p++)
         if(smallest[p] < permutationMatrix[p][0])
            permutationMatrix[p][0] = smallest[p];
   }
   delete []Index;
}


void OLS_Association::InvNorm(int tr)

{
   IntArray individuals(0);
   Vector myTraits(0);
   QuickIndex idx;
   for (int i = 0; i < pheno.Length(); i++){
      int missing = 0;
      for(int k = 0; k < covariates.Length(); k++)
         if(!ped[pheno[i]].isControlled(covariates[k])){
            missing = 1;
            break;
         }
      if(!missing && ped[pheno[i]].isPhenotyped(traits[tr])){
         individuals.Push(i);
         myTraits.Push(ped[pheno[i]].traits[traits[tr]]);
      }
   }
   idx.Index(myTraits);
   for(int i = 0; i < individuals.Length();){
      int start = i, end = i + 1;
      while(end < individuals.Length() &&
            ped[pheno[individuals[idx[end]]]].traits[traits[tr]] ==
            ped[pheno[individuals[idx[start]]]].traits[traits[tr]])
         end++;
      end --;
      double q = ninv((start + (end-start)/2.0 + 0.5)/individuals.Length());
      for(int j = start; j <= end; j++)
         ped[pheno[individuals[idx[j]]]].traits[traits[tr]] = q;
      i = end + 1;
   }
}


*/












   /*

   if(permutationCount < 1000)
      printf("P-value necessary for chromosome-wise significance (%.2G) is: %.2G\n",
         1.0/permutationCount, chidist(largest.Max(), K) );
   else{
      largest.Sort();
      printf("Point-wise p-value necessary for chromosome-wise significance (.002) is: %10.1G\n",
         chidist(largest[permutationCount-permutationCount/500], K) );
   } */


 /*

   if( permutationMatrix.cols && (tr==0) ){ // multiple chromosomes
      for(int p = 0; p < permutationCount; p++)
         if(smallest[p] < permutationMatrix[p][tr])
            permutationMatrix[p][tr] = smallest[p];
   }else{ // one chromosome analysis
      permutationMatrix.Dimension(permutationCount, traits.Length());
      permutationMatrix.Set(1.0);
      for(int p = 0; p < permutationCount; p++)
         permutationMatrix[p][tr] = smallest[p];
   }
   */
/*

   if(permutationMatrix.cols == 0){
      permutationMatrix.Dimension(permutationCount, traits.Length());
      permutationMatrix.Set(1.0);
      for(int p = 0; p < permutationCount; p++)
         permutationMatrix[p][tr] = smallest[p];
   }else{
      for(int p = 0; p < permutationCount; p++)
         if(smallest[p] < permutationMatrix[p][tr])
            permutationMatrix[p][tr] = smallest[p];
   }
*/


/*

void AssociationAnalysis::OLSunivariate(int tr)
{
   IntArray validPerson(ped.count);
   validPerson.Set(1);
   for(int i = 0; i < ped.count; i++){
      if(!ped[i].isPhenotyped(traits[tr])){
         validPerson[i] = 0;
         continue;
      }
      for(int c = 0; c < covariates.Length(); c++)
         if(!ped[i].isControlled(covariates[c])){
            validPerson[i] = 0;
            break;
         }
   }

   Vector Y;   // Y[i]: traits for ith individual
   Vector R;   // R[i]: residual for ith individual
   Vector B;   // B: regression coefficients for kth trait
   Vector tV, tV2;
   Matrix tM;

   Cholesky chol;
   IntArray sample;
   int P = covariates.Length() + 1;
   B.Dimension(P);

   // obtain residuals
   sample.Dimension(0);
     for(int i = 0; i < ped.count; i++)
      if(validPerson[i]) sample.Push(i);

   int N0 = sample.Length();
   Matrix Z(N0, P);
   Y.Dimension(N0);
   R.Dimension(N0);

   for(int i = 0; i < N0; i++){
      Z[i][0] = 1;
      for(int c = 1; c < P; c++)
         Z[i][c] = ped[sample[i]].covariates[covariates[c-1]];
      Y[i] = ped[sample[i]].traits[traits[tr]];
   }
   tM.Transpose(Z);
   for(int i = 1; i < covariates.Length()+1; i++)
      if(tM[i].Var() < 1E-20) {
         Z.DeleteColumn(i);
         P--;
      }

   tM.Dimension(P, P);  // tM = ZZ'
   tM.Set(0.0);
   for(int s = 0; s < P; s++)
      for(int t = 0; t < P; t++)
         for(int i = 0; i < N0; i++)
            tM[s][t] += Z[i][s] * Z[i][t];
   chol.Decompose(tM);
   tV.Dimension(P);    // tV = ZY
   tV.Set(0.0);
   for(int i = 0; i < N0; i++){
      tV2 = Z[i];
      tV2.Multiply(Y[i]);
      tV.Add(tV2);
   }
   chol.BackSubst(tV);
   B = chol.x;
   for(int i = 0; i < N0; i++)
      R[i] = Y[i] - Z[i].InnerProduct(B);

   if(P>1){
      printf("%15s", (const char*)ped.traitNames[traits[tr]]);
      for(int i = 0; i < P; i++)
         printf("%10.3f", B[i]);
      for(int i = P; i < covariates.Length()+1; i++)
         printf("%10s", "");
      printf("%7d\n", N0);
   }

   // GWA
   Vector X;   // genotype score
   IntArray subset;
   int N;
   statArray[tr].Dimension(0);
   pvalueArray[tr].Dimension(0);
   pvaluePosition[tr].Dimension(0);

   for(int m = 0; m < markers.Length(); m++){
      if(ped.GetMarkerInfo(markers[m])->freq.Length()>3) continue;
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < sample.Length(); i++)
         if(ped[sample[i]].markers[markers[m]].isKnown()){
            subset.Push(i);
            X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);

      Y.Dimension(N);
      for(int i = 0; i < N; i++)
         Y[i] = R[subset[i]] * X[i];
      double t = Y.Sum();
      double statistic = t * t / Y.SumSquares();
      double pvalue = chidist(statistic, 1);
      if(pvalue < pvalueSummary) {
         statArray[tr].Push(statistic);
         pvalueArray[tr].Push(pvalue);
         pvaluePosition[tr].Push(m);
      }
   }

   // Permutation Test
   if(permutationCount==0) return;
   PERMUTATION permu;
   Vector largest;
   double temp;
   permu.init(N0);
   largest.Dimension(permutationCount);
   largest.Set(-1.0);

   for(int p = 0; p < permutationCount; p++){
      permu.shuffle();
      index[p].Dimension(permu.count);
      for(int i = 0; i < permu.count; i++)
         index[p][i] = permu[i];
   }
   for(int m = 0; m < ped.markerCount; m++){
      if(ped.GetMarkerInfo(m)->freq.Length()>3) continue;
      subset.Dimension(0);
      X.Dimension(0);
      for(int i = 0; i < sample.Length(); i++)
         if(ped[sample[i]].markers[m].isKnown()){
            subset.Push(i);
            X.Push(ped[sample[i]].markers[m].countAlleles(1));
         }
      N = subset.Length();
      X.Add(-X.Sum()/N);
      Y.Dimension(N);

      for(int p = 0; p < permutationCount; p++){
         for(int i = 0; i < N; i++)
            Y[i] = R[index[p][subset[i]]] * X[i];
         temp = Y.Sum();
         temp = temp * temp / Y.SumSquares();
         if(temp > largest[p]) largest[p] = temp;
      }
   }
   for(int p = 0; p < permutationCount; p++){
      temp = chidist(largest[p], 1);
      if(temp < permutationMatrix[p][tr])
         permutationMatrix[p][tr] = temp;
   }
}

void AssociationAnalysis::OLSunivariate()
{
   int K = traits.Length();
   if(statArray==NULL) statArray = new Vector[K];
   if(pvalueArray==NULL) pvalueArray = new Vector[K];
   if(pvaluePosition==NULL) pvaluePosition = new IntArray[K];
   if(permutationCount){
      permutationMatrix.Dimension(permutationCount, K);
      permutationMatrix.Set(1.0);
      index = new IntArray[permutationCount];
   }
   if(covariates.Length()){
      printf("\nFitted Models Without Genotypes\n");
      printf("=================================================================\n");
      printf("%15s%10s", "Trait", "Mean");
      for(int i = 0; i < covariates.Length(); i++)
         printf("%10s", (const char*)ped.covariateNames[covariates[i]]);
      printf("%7s\n", "N");
   }
   for(int k = 0; k < K; k++)
      OLSunivariate(k);
   for(int k = 0; k < K; k++){
      printf("\nPedigree-Wide Association Analysis of %s\n");
      if(pvalueArray[k].Length()==0) continue;
      printf("======================================================================\n",
         (const char*)ped.traitNames[traits[k]]);
      printf("%10s%7s%10s%15s\n", "Position", "LOD", "pvalue", "Marker");
      for(int i = 0; i < pvalueArray[k].Length(); i++){
         int m = pvaluePosition[k][i];
         if(pvalueArray[k][i] > pvalueLessThan) continue;
         double LOD = statArray[k][i] * ToLOD;
         printf("%10f%7.2f%10.2G%15s", ped.GetMarkerInfo(markers[m])->position*100,
            LOD, pvalueArray[k][i], (const char*)ped.GetMarkerInfo(markers[m])->name);
         printf("\n");
      }
   }

   PrintSummary();
   if(index) delete []index;
}
*/

/*   if(statArray==NULL) statArray = new Vector[K];

   if(pvalueArray==NULL) pvalueArray = new Vector[K];
   if(pvaluePosition==NULL) pvaluePosition = new IntArray[K];

   for(int k = 0; k < K; k++){
      statArray[k].Dimension(0);
      pvalueArray[k].Dimension(0);
      pvaluePosition[k].Dimension(0);
   }
*/
         /*
         if(pvalue[k] < peakPvalue[k]){
            peakPvalue[k] = pvalue[k];
            peakPosition[k] = m;
            peakStatistic[k] = statistic[k];
         }
         if(pvalue[k] < pvalueLessThan) {
            statArray[k].Push(statistic[k]);
            pvalueArray[k].Push(pvalue[k]);
            pvaluePosition[k].Push(m);
         } */



/*

void AssociationAnalysis::OLSunivariate()
{
   int K = traits.Length();
   int N = ped.count;
   Vector V;
   Matrix M;

   if( first_chromosome && permutationCount ){
      Residual.Dimension(K, N);
      Residual.Set(_NAN_);
      permutationMatrix.Dimension(K, permutationCount);
      permutationMatrix.Set(1.0);
   }
   if(statArray==NULL) statArray = new Vector[K];
   if(pvalueArray==NULL) pvalueArray = new Vector[K];
   if(pvaluePosition==NULL) pvaluePosition = new IntArray[K];

   if(first_chromosome && covariates.Length()){
      printf("\nFitted Models Without Genotypes\n");
      printf("=================================================================\n");
      printf("%15s%10s", "Trait", "Mean");
      for(int i = 0; i < covariates.Length(); i++)
         printf("%10s", (const char*)ped.covariateNames[covariates[i]]);
      printf("%7s\n", "N");
   }
   for(int k = 0; k < K; k++){
      IntArray validPerson(ped.count);
      validPerson.Set(1);
      for(int i = 0; i < ped.count; i++){
         if(!ped[i].isPhenotyped(traits[k])){
            validPerson[i] = 0;
            continue;
         }
         for(int c = 0; c < covariates.Length(); c++)
            if(!ped[i].isControlled(covariates[c])){
               validPerson[i] = 0;
               break;
            }
      }

      Vector Y;   // Y[i]: traits for ith individual
      Vector R;   // R[i]: residual for ith individual
      Vector B;   // B: regression coefficients for kth trait
      Vector V2;
      Cholesky chol;
      IntArray sample;
      int P = covariates.Length() + 1;
      B.Dimension(P);

      // obtain residuals
      sample.Dimension(0);
      for(int i = 0; i < ped.count; i++)
         if(validPerson[i]) sample.Push(i);
      int N0 = sample.Length();
      Matrix Z(N0, P);
      Y.Dimension(N0);
      R.Dimension(N0);

      for(int i = 0; i < N0; i++){
         Z[i][0] = 1;
         for(int c = 1; c < P; c++)
            Z[i][c] = ped[sample[i]].covariates[covariates[c-1]];
         Y[i] = ped[sample[i]].traits[traits[k]];
      }
      M.Transpose(Z);
      for(int i = 1; i < covariates.Length()+1; i++)
         if(M[i].Var() < 1E-20) {
            Z.DeleteColumn(i);
            P--;
         }
      M.Dimension(P, P);  // M = ZZ'
      M.Set(0.0);
      for(int s = 0; s < P; s++)
         for(int t = 0; t < P; t++)
            for(int i = 0; i < N0; i++)
               M[s][t] += Z[i][s] * Z[i][t];
      chol.Decompose(M);
      V.Dimension(P);    // V = ZY
      V.Set(0.0);
      for(int i = 0; i < N0; i++){
         V2 = Z[i];
         V2.Multiply(Y[i]);
         V.Add(V2);
      }
      chol.BackSubst(V);
      B = chol.x;
      for(int i = 0; i < N0; i++)
         R[i] = Y[i] - Z[i].InnerProduct(B);
      if(permutationCount)
         for(int i = 0; i < N0; i++)
            if(validPerson[sample[i]])
               Residual[k][sample[i]] = R[i];
      if( first_chromosome && (P>1) ){
         printf("%15s", (const char*)ped.traitNames[traits[k]]);
         for(int i = 0; i < P; i++)
            printf("%10.3f", B[i]);
         for(int i = P; i < covariates.Length()+1; i++)
            printf("%10s", "");
         printf("%7d\n", N0);
      }

      // GWA
      Vector X;   // genotype score
      IntArray subset;
      statArray[k].Dimension(0);
      pvalueArray[k].Dimension(0);
      pvaluePosition[k].Dimension(0);
      double statistic, pvalue, temp;

      for(int m = 0; m < markers.Length(); m++){
         subset.Dimension(0);
         X.Dimension(0);
         for(int i = 0; i < sample.Length(); i++)
            if(ped[sample[i]].markers[markers[m]].isKnown()){
               subset.Push(i);
               X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
            }
         N0 = subset.Length();
         X.Add(-X.Sum()/N0);
         Y.Dimension(N0);
         for(int i = 0; i < N0; i++)
            Y[i] = R[subset[i]] * X[i];
         temp = Y.SumSquares();
         if(temp < 1E-50) continue;
         else{
            statistic = Y.Sum();
            statistic = statistic * statistic / temp;
            pvalue = chidist(statistic, 1);
            if(pvalue < peakPvalue[k]){
               peakPvalue[k] = pvalue;
               peakPosition[k] = m;
               peakStatistic[k] = statistic;
            }
            if(pvalue < pvalueLessThan){
               statArray[k].Push(statistic);
               pvalueArray[k].Push(pvalue);
               pvaluePosition[k].Push(m);
            }
         }
      }
   }

   for(int k = 0; k < K; k++){
      printf("Pedigree-Wide Association Analysis of %s\n",
         (const char*)ped.traitNames[k]);
      if(pvalueArray[k].Length()==0) continue;
      printf("======================================================================\n",
         (const char*)ped.traitNames[traits[k]]);
      printf("%10s%7s%10s%15s\n", "Position", "LOD", "pvalue", "Marker");
      for(int i = 0; i < pvalueArray[k].Length(); i++){
         int m = pvaluePosition[k][i];
         double LOD = statArray[k][i] * ToLOD;
         printf("%10f%7.2f%10.2G%15s", ped.GetMarkerInfo(markers[m])->position*100,
            LOD, pvalueArray[k][i], (const char*)ped.GetMarkerInfo(markers[m])->name);
         printf("\n");
      }
      printf("\n");
   }
   PrintSummary();

   // Permutation Test
   if(permutationCount==0) return;
   PERMUTATION permu;
   permu.init(N, randomSeed);
   IntArray Index(permutationCount * N);
   for(int p = 0; p < permutationCount; p++){
      permu.shuffle();
      for(int i = 0; i < N; i++)
         Index[p*N+i] = permu[i];
   }
   double temp, t;
   Matrix largest(K, permutationCount);
   largest.Set(-1.0);
   Vector X0(N), X(N);
   Vector Y, G;

   for(int m = 0; m < markers.Length(); m++){
      X0.Set(_NAN_);
      for(int i = 0; i < N; i++)
         if(ped[i].markers[markers[m]].isKnown())
            X0[i] = ped[i].markers[markers[m]].countAlleles(1);
      for(int p = 0; p < permutationCount; p++){
         for(int i = 0; i < N; i++)
            X[i] = X0[Index[p*N+i]];
         for(int k = 0; k < K; k++){
            G.Dimension(0);
            Y.Dimension(0);
            for(int i = 0; i < N; i++)
               if( (X[i]!=_NAN_) && (Residual[k][i]!=_NAN_) ){
                  G.Push(X[i]);
                  Y.Push(Residual[k][i]);
               }
            temp = G.Sum()/G.Length();
            V.Dimension(G.Length());
            for(int i = 0; i < G.Length(); i++)
               V[i] = (G[i] - temp) * Y[i];
            temp = V.SumSquares();
            if(temp < 1E-50) continue;
            t = V.Sum();
            temp = t * t / temp;
            if(temp > largest[k][p]) largest[k][p] = temp;
         }
      }
   }
   for(int k = 0; k < K; k++)
      for(int p = 0; p < permutationCount; p++){
         temp = chidist(largest[k][p], 1);
         if(temp < permutationMatrix[k][p])
            permutationMatrix[k][p] = temp;
      }
}
*/
/*         for(int i = 0; i < N; i++)

            if( (X[i]!=_NAN_) && (Residual[k][i]!=_NAN_) ){
               G.Push(X[i]);
               Y.Push(Residual[k][i]);
            }
*/




/*

   IntArray valid;
   IntArray connectionCount;
   IntArray sample;
   Kinship kin;

   Vector *kinship = NULL;
   kinship = new Vector[ped.familyCount];
   for(int f = 0; f < ped.familyCount; f++){
      int count = ped.families[f]->count;
      int first = ped.families[f]->first;
      if(count==0) continue;
      kin.Setup(*ped.families[f]);
      kinship[f].Dimension(count * count);
      for(int i = ped.families[f]->first; i <= ped.families[f]->last; i++){
         kinship[f][(i-first)*count+i-first] = kin(ped[i], ped[i]);
         for(int j = ped.families[f]->first; j < i; j++)
            kinship[f][(j-first)*count+i-first] =
               kinship[f][(i-first)*count+j-first] = kin(ped[i], ped[j]);
      }
   }
   IntArray validPerson(ped.count);
   validPerson.Set(0);

   for(int i = 0; i < ped.count; i++){
      for(trait = 0; trait < traits.Length(); trait++)
         if(ped[i].isPhenotyped(traits[trait])) {validPerson[i] = 1; break;}
      if(!validPerson[i]) continue;
      for(int c = 0; c < covariates.Length(); c++)
         if(!ped[i].isControlled(covariates[c])) {validPerson[i] = 0; break;}
   }
   int testCount = 1000;
   if(ped.markerCount < testCount) testCount = ped.markerCount;
   for(int i = 0; i < ped.count; i++){
      int count = 0;
      for(int m = 0; m < testCount; m++)
         if(ped[i].markers[m].isKnown())
            count ++;
      if(count <= testCount/2) validPerson[i] = 0;   // invalid markers
   }

   sample.Dimension(ped.count);
   sample.Zero();
   for(int f = 0; f < ped.familyCount; f++){
      int count = ped.families[f]->count;
      int first = ped.families[f]->first;
      valid.Dimension(0);
      for(int i = first; i <= ped.families[f]->last; i++)
         if(validPerson[i])
            valid.Push(i);
     if(valid.Length() == 0) continue;
     else if(valid.Length() == 1){
         sample[valid[0]] = 1;
         continue;
      }
      connectionCount.Dimension(valid.Length());
      connectionCount.Zero();
      for(;;){
         for(int i = 0; i < valid.Length(); i++)
            if(connectionCount[i]>-1)
               connectionCount[i] = 0;
         for(int i = 0; i < valid.Length(); i++){
            if(connectionCount[i]==-1) continue;
            for(int j = 0; j < i; j++){
               if(connectionCount[j]==-1) continue;
               if(kinship[f][(valid[i]-first)*count + valid[j]-first] > 0){
                  connectionCount[i]++;
                  connectionCount[j]++;
               }
            }
         }
         double min = 9999999;
         int k = -1;
         for(int i = 0; (i < valid.Length()); i++){
            if(connectionCount[i]==-1) continue;
            if(connectionCount[i] < min){
               min = connectionCount[i];
               k = i;
            }
         }
         if(k == -1) break;
         sample[valid[k]] = 1;
         for(int i = 0; i < valid.Length(); i++){
            if(connectionCount[i]==-1) continue;
            if(kinship[f][(valid[i]-first)*count + valid[k]-first] > 0)
               connectionCount[i] = -1;
         }
      }
   }
   if(kinship) delete []kinship;
*/



//   IntArray valid;

//   IntArray connectionCount;
//   Kinship kin;
   /*
   int isFamilyData = 0;
   for(int f = 0; f < ped.familyCount; f++)
      if(ped.families[f]->count>1){
         isFamilyData = 1;
         break;
      }

   Vector *kinship = NULL;
   if(isFamilyData){
      kinship = new Vector[ped.familyCount];
      for(int f = 0; f < ped.familyCount; f++){
         int count = ped.families[f]->count;
         int first = ped.families[f]->first;
         if(count==0) continue;
         kin.Setup(*ped.families[f]);
         kinship[f].Dimension(count * count);
         for(int i = ped.families[f]->first; i <= ped.families[f]->last; i++){
            kinship[f][(i-first)*count+i-first] = kin(ped[i], ped[i]);
            for(int j = ped.families[f]->first; j < i; j++)
               kinship[f][(j-first)*count+i-first] =
                  kinship[f][(i-first)*count+j-first] = kin(ped[i], ped[j]);
         }
      }
   }
   for(trait = 0; trait < traits.Length(); trait++){
      sample.Dimension(0);
      if(isFamilyData){
         for(int f = 0; f < ped.familyCount; f++){
            int count = ped.families[f]->count;
            int first = ped.families[f]->first;
            valid.Dimension(0);
            for(int i = first; i <= ped.families[f]->last; i++){
               int v = 1;
               if(!ped[i].isPhenotyped(traits[trait]))
                  continue;
               for(int c = 0; c < covariates.Length(); c++)
                  if(!ped[i].isControlled(covariates[c]))
                     {v = 0; break;}
               if(!v) continue;
               for(int m = 0; m < testList.Length(); m++)
                  if(!ped[i].markers[testList[m]].isKnown())
                     {v = 0; break;}
               if(v==0) continue;
               valid.Push(i);
            } 
            if(valid.Length() == 0) continue;
            else if(valid.Length() == 1){
               sample.Push(valid[0]);
               continue;
            }
            connectionCount.Dimension(valid.Length());
            connectionCount.Zero();
            for(;;){
               for(int i = 0; i < valid.Length(); i++)
                  if(connectionCount[i]>-1) connectionCount[i] = 0;
               for(int i = 0; i < valid.Length(); i++){
                  if(connectionCount[i]==-1) continue;
                  for(int j = 0; j < i; j++){
                     if(connectionCount[j]==-1) continue;
                     if(kinship[f][(valid[i]-first)*count + valid[j]-first] > 0){
                        connectionCount[i]++;
                        connectionCount[j]++;
                     }
                  }
               }
               double min = 9999999;
               int k = -1;
               for(int i = 0; (i < valid.Length()); i++){
                  if(connectionCount[i]==-1) continue;
                  if(connectionCount[i] < min){
                     min = connectionCount[i];
                     k = i;
                  }
               }
               if(k == -1) break;
               sample.Push(valid[k]);
               for(int i = 0; i < valid.Length(); i++){
                  if(connectionCount[i]==-1) continue;
                  if(kinship[f][(valid[i]-first)*count + valid[k]-first] > 0)
                     connectionCount[i] = -1;
               }
            }
         }
      }else // population data
         for(int i = 0; i < ped.count; i++){
            int v = 1;
            if(!ped[i].isPhenotyped(traits[trait]))
               continue;
            for(int c = 0; c < covariates.Length(); c++)
               if(!ped[i].isControlled(covariates[c]))
                  {v = 0; break;}
            if(!v) continue;
            for(int m = 0; m < testList.Length(); m++)
               if(!ped[i].markers[testList[m]].isKnown())
                  {v = 0; break;}
            if(!v) continue;
            sample.Push(i);
         }
  */

/*

      M.Transpose(Z);

      for(int i = 1; i < covariates.Length()+1; i++)

         if(M[i].Var() < 1E-20) {
            Z.DeleteColumn(i);
            P--;
         }
*/
/*      Matrix temp;

      temp.Transpose(Z);
      M.Product(temp, Z);

      V.Dimension(P);    // V = ZY
      V.Set(0.0);
      for(int i = 0; i < N0; i++){
         V2 = Z[i];
         V2.Multiply(Y[i]);
         V.Add(V2);
      }
*/

/*

   IntArray validPerson(ped.count);
   validPerson.Set(1);
   for(int i = 0; i < ped.count; i++){
      for(int k = 0; k < traits.Length(); k++)
         if(!ped[i].isPhenotyped(traits[k])){
            validPerson[i] = 0;
            break;
         }
      if(!validPerson[i]) continue;
      for(int c = 0; c < covariates.Length(); c++)
         if(!ped[i].isControlled(covariates[c])){
            validPerson[i] = 0;
            break;
         }
   }

   Matrix Y;   // Y[i]: traits for ith individual
   Matrix R;   // R[i]: residual for ith individual
   Matrix OR0;  // OR: Omega^-1 R
   Matrix B;   // B[k]: regression coefficients for kth trait
   Matrix O;   // O[k][l]: covariance between trait k and l
   Vector tV, tV2;
   Matrix tM;
   Cholesky chol;
   IntArray sample;
   int P = covariates.Length() + 1;
   int K = traits.Length();
   B.Dimension(K, P);
   O.Dimension(K, K);

   // obtain residuals
   sample.Dimension(0);
   for(int i = 0; i < ped.count; i++)
      if(validPerson[i]) sample.Push(i);

   int N0 = sample.Length();
   Matrix Z(N0, P);

   Y.Dimension(N0, K);
   R.Dimension(N0, K);

   for(int i = 0; i < N0; i++){
      Z[i][0] = 1;
      for(int c = 1; c < P; c++)
         Z[i][c] = ped[sample[i]].covariates[covariates[c-1]];
      for(int k = 0; k < K; k++)
         Y[i][k] = ped[sample[i]].traits[traits[k]];
   }

   tM.Dimension(P, P);  // tM = ZZ'
   tM.Set(0.0);
   for(int s = 0; s < P; s++)
      for(int t = 0; t < P; t++)
         for(int i = 0; i < N0; i++)
            tM[s][t] += Z[i][s] * Z[i][t];
   chol.Decompose(tM);
   tV.Dimension(P);    // tV = ZY
   for(int k = 0; k < K; k++){
      tV.Set(0.0);
      for(int i = 0; i < N0; i++){
         tV2 = Z[i];
         tV2.Multiply(Y[i][k]);
         tV.Add(tV2);
      }
      chol.BackSubst(tV);
      B[k] = chol.x;
   }
   for(int i = 0; i < N0; i++){
      R[i] = Y[i];
      for(int k = 0; k < K; k++)
         R[i][k] -= Z[i].InnerProduct(B[k]);
   }

   tM.Transpose(R);    // covariance matrix
   for(int k = 0; k < K; k++){
      O[k][k] = tM[k].Var();
      for(int l = k+1; l < K; l++)
         O[l][k] = O[k][l] = (tM[k].InnerProduct(tM[l]) -
            tM[k].Sum()*tM[l].Sum()/N0) / (N0-1);
   }
   OR0.Dimension(N0, K);
   chol.Decompose(O);
   for(int i = 0; i < N0; i++){
      chol.BackSubst(R[i]);
      OR0[i] = chol.x;
   }

   if(P>1){
      printf("\nFitted Models Without Genotypes (N=%d)\n", N0);
      printf("=================================================================\n");
      printf("%15s%10s", "Trait", "Mean");
      for(int i = 0; i < P-1; i++)
         printf("%10s", (const char*)ped.covariateNames[covariates[i]]);
      printf("\n");
      for(int k = 0; k < K; k++){
         printf("%15s", (const char*)ped.traitNames[traits[k]]);
         for(int i = 0; i < P; i++)
            printf("%10.3f", B[k][i]);
         printf("\n");
      }
      printf("\n");
   }
*/
   // GWA

/*

   Vector smallest;
   smallest.Dimension(permutationCount);
   for(int p = 0; p < permutationCount; p++)
      smallest[p] = chidist(largest[p], K);
   if(permutationMatrix.cols == 0){ // one chromosome analysis
      permutationMatrix.Dimension(permutationCount, 1);
      for(int p = 0; p < permutationCount; p++)
         permutationMatrix[p][0] = smallest[p];
   }else{   // multiple chromosomes
      for(int p = 0; p < permutationCount; p++)
         if(smallest[p] < permutationMatrix[p][0])
            permutationMatrix[p][0] = smallest[p];
   }*/
/*

void OLS_Association::OLSregression()

{
   Vector Y, X;
   double beta;
   double Sxx, Sxy, Syy, meanX, meanY, v, R2;
   int length;
   IntArray valid;
   IntArray connectionCount;
   IntArray sample;
   Kinship kin;
   double stat;
   double apvalue;

   if(pvalueArray==NULL) pvalueArray = new Vector[traits.Length()];
   if(pvaluePosition==NULL) pvaluePosition = new IntArray[traits.Length()];

   for(int t = 0; t < traits.Length(); t++){
      pvalueArray[t].Dimension(0);
      pvaluePosition[t].Dimension(0);
   }

   int isFamilyData = 0;
   for(int f = 0; f < ped.familyCount; f++)
      if(ped.families[f]->count>1){
         isFamilyData = 1;
         break;
      }

   Vector *kinship = NULL;
   if(isFamilyData){
      kinship = new Vector[ped.familyCount];
      for(int f = 0; f < ped.familyCount; f++){
         int count = ped.families[f]->count;
         int first = ped.families[f]->first;
         if(count==0) continue;
         kin.Setup(*ped.families[f]);
         kinship[f].Dimension(count * count);
         for(int i = ped.families[f]->first; i <= ped.families[f]->last; i++){
            kinship[f][(i-first)*count+i-first] = kin(ped[i], ped[i]);
            for(int j = ped.families[f]->first; j < i; j++)
               kinship[f][(j-first)*count+i-first] =
                  kinship[f][(i-first)*count+j-first] = kin(ped[i], ped[j]);
         }
      }
   }
   int printHead=0;
   IntArray validPerson(ped.count);
   for(trait = 0; trait < traits.Length(); trait++){
      validPerson.Set(1);
      for(int i = 0; i < ped.count; i++){
         if(!ped[i].isPhenotyped(traits[trait])) {validPerson[i] = 0; continue;}
         for(int c = 0; c < covariates.Length(); c++)
            if(!ped[i].isControlled(covariates[c])) {validPerson[i] = 0; break;}
      }
      if(printHead) printf("\n");
      printHead = 0;
      printf("Pedigree-Wide Association Analysis (%s)\n",
         (const char*)ped.traitNames[traits[trait]]);
      for(int m = 0; m < markerCount; m++){
         if(ped.GetMarkerInfo(markers[m])->freq.Length()>3) continue;
         sample.Dimension(0);
         if(isFamilyData){
            for(int f = 0; f < ped.familyCount; f++){
               int count = ped.families[f]->count;
               int first = ped.families[f]->first;
               valid.Dimension(0);
               for(int i = first; i <= ped.families[f]->last; i++)
                  if(validPerson[i] && ped[i].markers[markers[m]].isKnown())
                     valid.Push(i);
               if(valid.Length() == 0) continue;
               else if(valid.Length() == 1){
                  sample.Push(valid[0]);
                  continue;
               }
               connectionCount.Dimension(valid.Length());
               connectionCount.Zero();
               for(;;){
                  for(int i = 0; i < valid.Length(); i++)
                     if(connectionCount[i]>-1) connectionCount[i] = 0;
                  for(int i = 0; i < valid.Length(); i++){
                     if(connectionCount[i]==-1) continue;
                     for(int j = 0; j < i; j++){
                        if(connectionCount[j]==-1) continue;
                        if(kinship[f][(valid[i]-first)*count + valid[j]-first] > 0){
                           connectionCount[i]++;
                           connectionCount[j]++;
                        }
                     }
                  }
                  double min = 9999999;
                  int k = -1;
                  for(int i = 0; (i < valid.Length()); i++){
                     if(connectionCount[i]==-1) continue;
                     if(connectionCount[i] < min){
                        min = connectionCount[i];
                        k = i;
                     }
                  }
                  if(k == -1) break;
                  sample.Push(valid[k]);
                  for(int i = 0; i < valid.Length(); i++){
                     if(connectionCount[i]==-1) continue;
                     if(kinship[f][(valid[i]-first)*count + valid[k]-first] > 0)
                        connectionCount[i] = -1;
                  }
               }
            }
         }else // population data
            for(int i = 0; i < ped.count; i++)
               if(ped[i].isPhenotyped(traits[trait]) && ped[i].markers[markers[m]].isKnown())
                  sample.Push(i);
         Y.Dimension(0);
         X.Dimension(0);
         for(int i = 0; i < sample.Length(); i++){
            Y.Push(ped[sample[i]].traits[traits[trait]]);
            X.Push(ped[sample[i]].markers[markers[m]].countAlleles(1));
         }
         length = X.Length();
         if(length<3) continue;
         meanX = X.Average();
         meanY = Y.Average();
         Sxx = X.SumSquares() - meanX * meanX * length;
         Syy = Y.SumSquares() - meanY * meanY * length;
         if(Sxx<=0) continue;
         Sxy = X.InnerProduct(Y) - meanX * meanY * length;
         beta = Sxy / Sxx;
         X.Multiply(-beta);
         Y.Add(-meanY + beta * meanX);
         Y.Add(X);
         v = sqrt(Y.SumSquares() / (length - 2) / Sxx);
         stat = beta / v;
         R2 = beta * Sxy /Syy;
         apvalue = tdist(fabs(stat), length - 2); // df = length-2

         if(apvalue < peakPvalue[trait]){
            peakPvalue[trait] = apvalue;
            peakPosition[trait] = m;
            peakStatistic[trait] = stat;
         }
         if(apvalue < pvalueLessThan) {
            pvalueArray[trait].Push(apvalue);
            pvaluePosition[trait].Push(m);
         }
         if(apvalue < pvalueLessThan){
            if(printHead==0){
               printf("=======================================================\n"
                  "%10s%7s%10s%7s%15s", "Position", "t", "pvalue", "Effect", "Marker");

               if(isFamilyData) printf("%5s", "N");

               printf("\n");
               printHead = 1;
            }
            printf("%10f%7.2f%10.2G%6.1f%%%15s",
               ped.GetMarkerInfo(markers[m])->position*100, stat,
               apvalue, R2*100, (const char*)ped.GetMarkerInfo(markers[m])->name);
            if(isFamilyData) printf("%5d", length);
            printf("\n");
         }
      }
   }
   PrintSummary();
   if(kinship) delete []kinship;
}

*/




/*

      printf("\n%d-SNP Association Analysis (loglik=%.1f, df=%d)\n",
         testList.Length(), ols.loglik, testList.Length());
      printf("=====================================================\n");
      printf("%10s%7s%10s%10s%7s\n", "SNP", "t", "pvalue", "Effect", "h2");
      for(int c = 0; c < covariateCount; c++){
         printf("%10s%7.2f%10.2G%10.2G%6.1f%%\n",
            (const char*)ped.covariateNames[covariates[c]], ols.t_statistic[c+1],
            ols.pvalue[c+1], ols.beta[c+1], ols.R2[c]*100);
      }
      for(int m = 0; m < testList.Length(); m++)
         printf("%10s%7.2f%10.2G%10.2G%6.1f%%\n",
            (const char*)Alias[m], ols.t_statistic[covariateCount+m+1],
            ols.pvalue[covariateCount+m+1], ols.beta[covariateCount+m+1],
            ols.R2[covariateCount+m]*100);
*/
//      double loglik0 = -N*0.5*log(Y.Var()*(N-1)/N);

/*
      printf("Single-SNP Association Analysis (%s, N=%d)\n",
         (const char*)ped.traitNames[traits[trait]], N);
      printf("===========================================================\n");
      printf("%10s%7s%10s%10s%7s%10s%5s\n",
         "SNP", "t", "pvalue", "Effect", "h2", "loglik", "df");
*/


