#ifndef __GENOTYPELOADER_H__
#define __GENOTYPELOADER_H__

#include "StringArray.h"
#include "StringHash.h"
#include "Pedigree.h"
#include "IntArray.h"
#include "IgnoreSamples.h"
#include "LogManager.h"

extern LogManager logMgr;

// Class loads genotypes one line at a time
// Each genotype is coded as 0 (missing), 1 and 2 (homozygous) or 3 (heterozygous)

class GenotypeLoader
   {
   public:
      // List of marker and sample ids
      StringArray    columnLabels;
      StringIntHash  sampleIds;
      StringIntHash  markerIds;
      IntArray       columnToSampleId;
      StringArray    markerLabels;
      IntArray       individualMask;

      // Summary of pedigree information for each column
      IntArray       sexCodes;
      IntArray       motherColumn;
      IntArray       fatherColumn;

      // quality information
      IntArray  mendelErrors;
      int       totalMendelErrors;
      float     mendelErrorRate;

      // duplicate information from header
      IntArray *   duplicateSampleSets;
      IntArray     duplicatedSampleIds;
      /* stores whether the sample is a duplicate -- TDT will be performed only on non duplicates or the first of a set of duplicates*/
      IntArray     isDuplicateSample; 


      // Status information for current line
      bool       isValid;
      bool       isMarkerFailure;
      bool       isMonomorphic;
      bool       isDuplicateMarker;
      bool       isLineEmpty;
      bool       isEmptyMarker;
      String     failure;
      IntArray   genotypes;
      char       alleles[2];
      String     currMarker;
      
      GenotypeLoader();
      ~GenotypeLoader();

      // Input control functions
      void LoadHeader(FILE * input, IgnoreSamples & ignoreSamples, Pedigree & ped);
      void LoadMarker(FILE * input, IntArray & isSampleSkipped);

      void Rewind(FILE * input);

      // Transfer pedigree information
      void PopulatePedigreeInformation(Pedigree & ped);

      // Simple summaries for the current marker
      int CountGenotypes();
      void setMonomorphic();

      bool MendelCheck(int child, int parent);
      bool MendelCheckMother(int child, int mother) { return MendelCheck(child, mother); }
      bool MendelCheckFather(int child, int father) { return MendelCheck(child, father); }
      bool MendelCheck(int child, int mother, int father);
      void UpdateMendelErrors(FILE * mendelLog, bool isSexLinked);
      void countMendelErrors(IntArray & mask, bool isSexLinked);
      void ReleaseMemory();

   private:
      void ClearAlleles()
         { alleles[0] = alleles[1] = 0; }

      int GetAllele(char al);

      void buildDuplicateList();
      void OutputErrorTrios(FILE * mendelLog, int sampleColumn, int fatherColumn, int motherColumn, bool isSexLinked);

      String buffer;
      StringArray tokens;
   };

#endif


