#include "QCHistograms.h"

#include <stdlib.h>

double QCHistograms::red0[]   = {0.0, 0.0, 0.0, 0.0, 1.0 };
double QCHistograms::red1[]   = {0.0, 0.0, 0.0, 1.0, 1.0 };
double QCHistograms::blue0[] = {0.0, 0.0, 0.5, 1.0, 1.0 };
double QCHistograms::blue1[] = {0.0, 1.0, 1.0, 1.0, 0.0 };
double QCHistograms::green0[]  = {0.5, 1.0, 0.0, 0.0, 0.0 };
double QCHistograms::green1[]  = {1.0, 1.0, 0.0, 0.0, 0.0 };

void QCHistograms::ProcessMetric(Vector &statistic, IntArray &seriesCode, int numSeries, PDF & pdf, double min, double max, String title, String xLabel, int maxBins)
   {
   PDFBarChart    barChart;

   pdf.page.OpenPage();

   barChart.title = title;
   barChart.xLabel = xLabel;
   barChart.AllocateSeries(numSeries);
   
   if(numSeries == 1)
   {
      // if only one series, plot color is gray - medium
      barChart.red[0]   = 0.5;
      barChart.green[0] = 0.5;
      barChart.blue[0]  = 0.5;
   }
   else
   {
      double incrementColor = 1.0/(numSeries - 1);
   
      for(int i=0; i < numSeries; i++)
      {
         double z = 1.0 - i*incrementColor;
         barChart.red[i] = GetColor(z, red0, red1);
         barChart.green[i] = GetColor(z, green0, green1);
         barChart.blue[i] = GetColor(z, blue0, blue1);
      }
   }

   barChart.SetupBins(min, max, statistic.Length(), maxBins);

   for (int i = 0; i < statistic.Length(); i++)
      if (seriesCode[i] > -1)
         barChart.Bin(seriesCode[i], statistic[i]);

   barChart.Layout(0, 0, pdf.page.GetWidth(), pdf.page.GetHeight());
   barChart.Print(pdf);
   }

void QCHistograms::GetMarkerStat(StringArray &input, int column, Vector &statistic)
{
   StringArray tokens;
   statistic.Dimension(input.Length() - 1);
   statistic.Set(0);
   for(int i = 1; i < input.Length(); i++)
   {
      tokens.ReplaceTokens(input[i]);
      if (tokens.Length() < (column + 1)) continue;
      if (tokens[column].Compare("-") == 0)
         statistic[i-1] = -1; // this is only used for all +ve statistics (in particular the TDT stats)
      else
         statistic[i-1] = tokens[column].AsDouble();
   }
}

bool QCHistograms::BuildMarkerHistograms(String filename, String outputName, Vector &minimums, Vector &maximums, bool plotQuality, bool TDT, bool Assoc)
   {
   PDF pdf;
   StringArray input;
   
   input.Read(filename);
   if (input.Length() == 0)
      {
      return false;
      }
   
   StringArray header;
   StringArray tokens;
   IntArray passSeries;
   header.AddTokens(input[0]);
   
   int numSeries = 1;
   int passfail = header.SlowFind("Flagged");
   
   if (passfail != -1)
   {
      passSeries.Dimension(input.Length() - 1);
      passSeries.Set(-1);
      for (int i = 1; i < input.Length(); i++)
      {
         tokens.ReplaceColumns(input[i]);
	 passSeries[i - 1] = ((tokens[passfail] == "PASSED") ? 1 : 0);
      }
      numSeries = 2;
   }

   pdf.OpenFile(outputName + ".snp.pdf");
   pdf.page.SetSize(psLetterR);
  
   // first column is markername, second is maf allele
   for (int i = 2; i <= NUMSTATS; i++)
   {
      Vector statistic;
      GetMarkerStat(input, i, statistic);

      ProcessMetric(statistic, passSeries, numSeries, pdf, minimums[i-2], maximums[i-2], MarkerTitles[i-2], MarkerXLabels[i-2], QC_Settings::MARKER_PDF_MAX_BINS);
   }
   
   if (plotQuality)
   {
      Vector statistic;
      GetMarkerStat(input, NUMSTATS + 1, statistic);
      ProcessMetric(statistic, passSeries, numSeries, pdf, minimums[NUMSTATS-1], maximums[NUMSTATS-1], MarkerTitles[NUMSTATS-1], MarkerXLabels[NUMSTATS-1], QC_Settings::MARKER_PDF_MAX_BINS);
   }

   if (TDT)
   {
      int temp = 1;
      for (int i = 0; i < 5; i++)
      {
         Vector statistic;
         GetMarkerStat(input, NUMSTATS+temp+i+3, statistic); // there are 2 allele columns between the regular stats and the TDT stats.
	 ProcessMetric(statistic, passSeries, numSeries, pdf, minimums[NUMSTATS+i], maximums[NUMSTATS+i], TDTTitles[i], TDTXLabels[i], QC_Settings::MARKER_PDF_MAX_BINS);
      }
   }
   
   if (Assoc)
   {
      int temp = 2; 
      if (TDT)   temp += 7;
      Vector statistic;
      //Association Chisq
      GetMarkerStat(input, NUMSTATS+temp, statistic);
      ProcessMetric(statistic, passSeries, numSeries, pdf, statistic.Min(), statistic.Max(), AssocTitles[0], AssocXLabels[0], QC_Settings::MARKER_PDF_MAX_BINS);

      //Association pvalues
      statistic.Clear();
      GetMarkerStat(input, NUMSTATS+temp+1, statistic);
      ProcessMetric(statistic, passSeries, numSeries, pdf, 0.0, 1.0, AssocTitles[1], AssocXLabels[1], QC_Settings::MARKER_PDF_MAX_BINS);
   }

   pdf.CloseFile();
   return true;
   }

bool QCHistograms::BuildSampleHistograms(FirstPass &firstPass, String outputName, bool plotQuality, SampleLabel &sampLabel, IntArray & sexCodes)
{
   PDF pdf;
   pdf.OpenFile(outputName + ".sample.pdf");
   pdf.page.SetSize(psLetterR);

   int numSeries = 2;
   IntArray sampleSeries;
   
   
   if (sampLabel.sampleLabels.Sum() == -sampLabel.sampleLabels.Length())
      GetSampleQCStatus(firstPass.mask, sampleSeries);
   else
   {
      sampleSeries = sampLabel.sampleLabels;
      numSeries = sampLabel.sampleLabels.Max() + 1;
   }

   Vector statistic;
   for (int i=0; i < 6; i++)
   {
      statistic.Clear();
      int currNumSeries = numSeries;
      IntArray currSampleSeries = sampleSeries;
      
      switch(i)
      {
      case 0:   ConvertToVector(firstPass.genotypeCounts, statistic);
                statistic /= double(firstPass.markers);
                break;
      case 1:   ConvertToVector(firstPass.heterozygoteCounts, firstPass.genotypeCounts, statistic);
                break;
      case 2:   ConvertToVector(firstPass.sampleMendelErrors, statistic);
      		statistic /= double(firstPass.markers);
                break;
      case 3:   statistic = firstPass.logSexOdds;
                statistic.DeleteDimension(0);
		currSampleSeries.Clear();
		for (int j=1; j < sexCodes.Length(); j++)
		   currSampleSeries.Push(sexCodes[j]);
		currNumSeries = 3;
                break;
      case 4:   statistic = firstPass.logLikelihood;
                statistic.DeleteDimension(0);
                break;
      case 5:   statistic = firstPass.sampleQualityScores;
                statistic /= double(firstPass.markers);
                statistic.DeleteDimension(0);
                break;
      }
      if(i == 5 && !plotQuality)  break;
      fflush(stdout);
      
      ProcessMetric(statistic, currSampleSeries, currNumSeries, pdf, statistic.Min(), statistic.Max(), SampleTitles[i], SampleXLabels[i], QC_Settings::MARKER_PDF_MAX_BINS);
      fflush(stdout);
   }
   pdf.CloseFile();
   return true;
}


void QCHistograms::ConvertToVector(IntArray &input, Vector &output)
{
   output.Clear();
   output.Dimension(input.Length() - 1);
   
   for(int i=1; i < input.Length(); i++)
      output[i-1] = double(input[i]);
}

void QCHistograms::ConvertToVector(IntArray &numerator, IntArray &denominator, Vector &output)
{
   output.Clear();
   if (numerator.Length() != denominator.Length())
      return;

   output.Dimension(numerator.Length() - 1);

   for(int i=1; i < numerator.Length(); i++)
      output[i-1] = (numerator[i]*1.0)/(denominator[i] + 1e-20);
}


void QCHistograms::GetSampleQCStatus(IntArray &mask, IntArray &status)
{
   status.Clear();
   status.Dimension(mask.Length() - 1);
   
   for(int i=1; i < mask.Length(); i++)
   {
      status[i-1] = ((mask[i]==0) ? 1 : 0);
   }
}


double QCHistograms::GetColor(double z, double color0[], double color1[])
{
   if (z <= 0.0) return color0[0];
   if (z >= 1.0) return color1[4];

   int base = (int) (z * 5);
   double delta = (z - 0.2 * base) / 0.2;

   return color0[base] + (color1[base] - color0[base]) * delta;
}

void QCHistograms::BuildRelationHistograms(Vector &estimatedKinship, IntArray &putativeRelations, IntArray &relationError, String outputName)
{
   PDF pdf;
   pdf.OpenFile(outputName + ".relation.pdf");
   pdf.page.SetSize(psLetterR);
   
   Vector statistics[5];
   IntArray passFailSeries[5];

   for (int i = 0; i < 5; i++)
   {
      statistics[i].Clear();
      passFailSeries[i].Clear();
   }
   
   for (int i = 0; i < estimatedKinship.Length(); i++)
   {
      passFailSeries[putativeRelations[i] - 1].Push(relationError[i]);
      statistics[putativeRelations[i] - 1].Push(estimatedKinship[i]);
   }

   for (int i = 0; i < 5; i++)
      PlotKinship(statistics[i], passFailSeries[i], pdf, RelationTitles[i], RelationXLabel);

   pdf.CloseFile();
}

void QCHistograms::PlotKinship(Vector &kinship, IntArray &seriesCode, PDF &pdf, String title, String xLabel)
{
   if (kinship.Length() == 0)
      return;

   int numSeries;
   if (seriesCode.Sum() == 0 || seriesCode.Sum() == seriesCode.Length())
      numSeries = 1;
   else numSeries = 2;
   
   PDFBarChart    barChart;

   pdf.page.OpenPage();

   barChart.title = title;
   barChart.xLabel = xLabel;
   barChart.AllocateSeries(numSeries);
   if (numSeries == 1)
   {
      barChart.red[0]   = GetColor((seriesCode.Sum()*1.0/seriesCode.Length()), red0, red1);
      barChart.green[0] = GetColor((seriesCode.Sum()*1.0/seriesCode.Length()), green0, green1);
      barChart.blue[0]  = GetColor((seriesCode.Sum()*1.0/seriesCode.Length()), blue0, blue1);
   }
   else
   {
      double incrementColor = 1.0/(numSeries - 1);
   
      for(int i=0; i < numSeries; i++)
      {
         double z = i*incrementColor;
         barChart.red[i] = GetColor(z, red0, red1);
         barChart.green[i] = GetColor(z, green0, green1);
         barChart.blue[i] = GetColor(z, blue0, blue1);
      }
   }
   
   barChart.SetupBins(0.0, 1.0, kinship.Length(), QC_Settings::RELATION_BINS);

   for (int i = 0; i < kinship.Length(); i++)
      barChart.Bin(seriesCode[i], kinship[i]);

   barChart.Layout(0, 0, pdf.page.GetWidth(), pdf.page.GetHeight());
   barChart.Print(pdf);
}
