#include "QC_Settings.h"
#include "Quality.h"

Quality::Quality()
   {
   genotypeCount = 0;
   }

bool Quality::LoadHeader(FILE * qualityFile, StringArray & genoFileSampleIds)
   {
   genotypeCount = genoFileSampleIds.Length();

   do
   {
      buffer.ReadLine(qualityFile);
      tokens.ReplaceTokens(buffer);
      headerColumns = tokens.Length();
   }while(headerColumns == 0 && !feof(qualityFile));

   if (headerColumns == 0)
   {
      failure.catprintf("WARNING: Empty Quality score file, ignored");
      logMgr.WriteToLog(failure, 0);
      failure.Clear();
      return false;
   }

   sampleColumn.Dimension(genoFileSampleIds.Length());
   sampleColumn.Set(-1);

   for (int i = 1; i < genoFileSampleIds.Length(); i++)
   {
      int columnNumber = tokens.Find(genoFileSampleIds[i]);
      if (columnNumber < 0)
      {
         failure.catprintf("WARNING: Sample %s doesn't have quality scores", (const char *)genoFileSampleIds[i]);
	 logMgr.WriteToLog(failure, 10);
	 failure.Clear();
      }

      sampleColumn[i] = columnNumber;
   }

   return true;
}

void Quality::PopulateMarkerHash(FILE * qualityFile)
{
   int line = 0;
   fpos_t * position;
   do
   {
      line++;
      position = new fpos_t;
      fgetpos(qualityFile, position);
      buffer.ReadLine(qualityFile);
      tokens.ReplaceTokens(buffer);
      if (tokens.Length() == 0)
         continue;

      if (markerLine.Find(tokens[0]) > -1)
      {
         failure.catprintf("WARNING: Duplicate Marker %s on line %d; line ignored", (const char *)tokens[0], line);
	 logMgr.WriteToLog(failure, 11);
	 failure.Clear();
         continue;
      }
      markerLine.Add(tokens[0], position);
   }while(!feof(qualityFile));
}

bool Quality::CheckQualityScores(String & markerName, IntArray & genotypes, FILE * qualityFile, bool & noGenoPostQual)
{
   noGenoPostQual = false;
   avgQualityScorePreThreshold = 0;
   if (qualityFile == NULL || markerLine.Entries() == 0)
      return true;

   fpos_t * position;
   failure = "";
   if (markerLine.Find(markerName) > -1)
   {
      position = (fpos_t *)markerLine.Object(markerName);
   }
   else
   {
      failure.catprintf("WARNING: Marker %s not found in quality file, genotypes kept as is", (const char *)markerName);
      logMgr.WriteToLog(failure, 16);
      failure.Clear();
      return false;
   }
   fsetpos(qualityFile, position);
   buffer.ReadLine(qualityFile);
   tokens.ReplaceTokens(buffer);

   // Check that the right number of tokens is available (comparing to header)
   if (tokens.Length() != headerColumns)
   {
      failure.catprintf("WARNING: Incorrect number of quality scores for marker %s: genotypes unchanged", (const char *) markerName);
      logMgr.WriteToLog(failure, 17);
      failure.Clear();
      return false;
   }

   noGenoPostQual = true;
   for (int i = 1; i < genotypes.Length(); i++)
   {
      if (sampleColumn[i] >= 0)
      {
         double score = tokens[sampleColumn[i]].AsDouble();
         avgQualityScorePreThreshold += score;
         if (score < QC_Settings::QUALITY_THRESHOLD)
            genotypes[i] = 0;
      }
      if (genotypes[i] != 0)
         noGenoPostQual = false;
   }
   avgQualityScorePreThreshold /= genotypeCount;
      
   return true;
}

void Quality::LoadQualityScores(FILE * qualityFile, String & markerName, IntArray & scores)
   {
   scores.Clear();
   scores.Dimension(genotypeCount);
   scores.Zero();
   scores.Set(-1);
   

   if (qualityFile == NULL || markerLine.Entries() == 0)
      return;

   fpos_t * position = (fpos_t *)markerLine.Object(markerName);

   if (position == NULL)
      {
      failure.printf("WARNING: Marker %s not found in quality file", (const char *)markerName);
      logMgr.WriteToLog(failure, 18);
      failure.Clear();
      return;
      }

   fsetpos(qualityFile, position);
   buffer.ReadLine(qualityFile);
   tokens.ReplaceTokens(buffer);

   // Check that the right number of tokens is available (comparing to header)
   if (tokens.Length() != headerColumns)
      {
      failure.printf("WARNING: Incorrect number of quality scores for marker %s", (const char *) markerName);
      logMgr.WriteToLog(failure, 19);
      return;
      }

   for (int i = 1; i < genotypeCount; i++)
      if (sampleColumn[i] >= 0)
         scores[i] = tokens[sampleColumn[i]].AsInteger();
   }
